/*
 * DCA compatible decoder data
 * Copyright (C) 2004 Gildas Bazin
 * Copyright (c) 2006 Benjamin Larsson
 *
 * This file is part of FFmpeg.
 *
 * FFmpeg is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * FFmpeg is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with FFmpeg; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
 */

#include <stdint.h>

#include "libavutil/mem.h"

#include "dca.h"
#include "dcadata.h"

/* Generic tables */

const uint32_t ff_dca_bit_rates[32] = {
      32000,   56000,   64000,   96000,  112000, 128000,
     192000,  224000,  256000,  320000,  384000,
     448000,  512000,  576000,  640000,  768000,
     896000, 1024000, 1152000, 1280000, 1344000,
    1408000, 1411200, 1472000, 1536000, 1920000,
    2048000, 3072000, 3840000, 1 /* open */, 2 /* variable */, 3 /* lossless */
};

const uint8_t ff_dca_channels[16] = {
    1, 2, 2, 2, 2, 3, 3, 4, 4, 5, 6, 6, 6, 7, 8, 8
};

const uint8_t ff_dca_bits_per_sample[8] = {
    16, 16, 20, 20, 0, 24, 24, 0
};

const uint8_t ff_dca_dmix_primary_nch[8] = {
    1, 2, 2, 3, 3, 4, 4, 0
};

const uint8_t ff_dca_audio_mode_ch_mask[] = {
    DCA_SPEAKER_LAYOUT_MONO,
    DCA_SPEAKER_LAYOUT_STEREO,
    DCA_SPEAKER_LAYOUT_STEREO,
    DCA_SPEAKER_LAYOUT_STEREO,
    DCA_SPEAKER_LAYOUT_STEREO,
    DCA_SPEAKER_LAYOUT_3_0,
    DCA_SPEAKER_LAYOUT_2_1,
    DCA_SPEAKER_LAYOUT_3_1,
    DCA_SPEAKER_LAYOUT_2_2,
    DCA_SPEAKER_LAYOUT_5POINT0
};

const uint8_t ff_dca_quant_index_sel_nbits[DCA_CODE_BOOKS] = {
    1, 2, 2, 2, 2, 3, 3, 3, 3, 3
};

const uint8_t ff_dca_quant_index_group_size[DCA_CODE_BOOKS] = {
    1, 3, 3, 3, 3, 7, 7, 7, 7, 7
};

/* ADPCM data */

/* 16 bits signed fractional Q13 binary codes */
const int16_t ff_dca_adpcm_vb[4096][4] = {
    {   9928,  -2618,  -1093, -1263 },
    {  11077,  -2876,  -1747,  -308 },
    {  10503,  -1082,  -1426, -1167 },
    {   9337,  -2403,  -1495,   274 },
    {  10698,  -2529,   -532, -1122 },
    {  10368,  -3974,  -1264,  -750 },
    {  10070,  -3667,    346,   863 },
    {  10278,  -3093,    311,  -576 },
    {   9894,  -1330,  -1428,  -860 },
    {  10544,  -1923,  -1058,  -971 },
    {  10996,  -1632,   -841, -1404 },
    {  11832,  -3465,   1658, -1990 },
    {  10852,   -688,  -2658,  -499 },
    {  10546,  -1749,   -147, -1733 },
    {  10801,  -1004,   -708, -1453 },
    {  10588,   -441,  -2113,  -952 },
    {  10141,  -3331,   -582, -1432 },
    {   9608,  -2590,    383,   258 },
    {  11422,  -3265,    229, -1544 },
    {  10460,  -1338,   -713, -1568 },
    {  10306,  -1721,  -1660,  -603 },
    {   9580,  -1812,  -1235, -1061 },
    {  11471,  -2285,  -1617,  -607 },
    {  10081,  -2225,  -1408,  -868 },
    {  10715,  -2624,  -1367,  -704 },
    {  10616,  -1871,  -2770,   -35 },
    {   9352,  -2340,  -1024, -1566 },
    {  11065,  -1458,  -1926,  -735 },
    {  11334,  -2056,  -1041, -1144 },
    {   9825,  -2048,   -794, -1536 },
    {  11850,  -2695,  -1123,  -867 },
    {  10654,  -2226,  -1891,  -373 },
    {  10024,  -1557,   -808, -1069 },
    {  11142,  -1266,  -3238,   128 },
    {  11729,  -3282,   -514, -1011 },
    {  11402,  -2094,  -2335,  -189 },
    {  10195,  -3658,    181, -1875 },
    {  11431,  -2626,   -404, -1377 },
    {  11001,  -3868,   -619, -1077 },
    {  10894,  -2559,    274, -1758 },
    {   9633,  -1482,  -2253,  -773 },
    {  11245,  -3321,    830, -1972 },
    {   9768,  -2701,   -199, -1859 },
    {  10500,  -2042,    525, -2043 },
    {  11669,  -4069,    293, -1468 },
    {   9192,  -1991,   -583,   -61 },
    {  10057,  -3220,  -2015,  -473 },
    {   9497,  -2315,  -2490,  -467 },
    {  10455,  -3069,  -1194, -1007 },
    {   9994,  -1936,    -60, -1225 },
    {   9295,  -2156,  -1761, -1134 },
    {  10085,  -3748,  -1026,   197 },
    {   9334,  -2360,    804,  -351 },
    {  11561,  -2553,   1352, -2313 },
    {  12837,  -3998,   1195, -1958 },
    {  10114,  -1100,  -2414,  -394 },
    {   9341,  -2530,    315,   755 },
    {  10131,  -3164,   1411,  -674 },
    {   9535,   -905,  -1551,   579 },
    {  11717,  -1519,  -3051,    91 },
    {   9824,  -2911,  -2775,   192 },
    {   9662,  -2934,   -561,  1450 },
    {  11085,  -3392,  -1298,  -659 },
    {   8955,  -2102,  -1899,   703 },
    {   8607,  -1742,  -4348,   814 },
    {   7640,  -2063,  -3617,    52 },
    {   7074,   -826,  -4325,  4375 },
    {   7714,    584,  -4238,  1927 },
    {   6355,   -952,  -4912,  3127 },
    {   7069,   -660,  -6413,  4087 },
    {   8313,   -132,  -2964,  -876 },
    {   6952,  -1422,  -3962,   -24 },
    {   9299,   -734,  -3088,  -263 },
    {   9484,   -574,  -4513,   466 },
    {   7246,    -91,  -3735,  -704 },
    {   8325,  -1417,  -3090,  -530 },
    {   6469,  -1226,  -4757,   829 },
    {   6652,   -368,  -5682,  1393 },
    {   7971,  -1278,  -2284,  1205 },
    {   7229,   -699,  -3556,  1840 },
    {   7994,   1284,  -2729,   732 },
    {   9005,   -698,  -4522,  2189 },
    {   6963,    197,  -2727,   380 },
    {   8527,    135,  -3991,  -213 },
    {   8840,    934,  -3014,  -567 },
    {  10125,    418,  -3284,  -371 },
    {   6367,    361,  -2318,  2554 },
    {   7892,    172,  -5247,  4673 },
    {   6674,    387,  -5424,  4398 },
    {   6240,    684,  -4047,  1219 },
    {  11170,   -794,  -5081,  1195 },
    {  11765,   -648,  -6265,  2052 },
    {  10845,   -775,  -3837,   366 },
    {  12496,   -689,  -8260,  3562 },
    {   7893,  -1166,  -4972,   988 },
    {   8592,   1052,  -5986,  3087 },
    {   7277,   1874,  -5685,  3579 },
    {   6900,   2016,  -4809,  3491 },
    {   8530,  -2405,  -3250,  1986 },
    {   9426,    494,  -7067,  5038 },
    {  10285,    564,  -8210,  5370 },
    {   8749,  -2207,  -3980,  2852 },
    {   9653,  -2686,  -4300,  1400 },
    {   9770,  -2286,  -5663,  4233 },
    {   8490,     -4,  -7048,  4496 },
    {   7697,  -1209,  -5328,  3183 },
    {   6451,    801,  -4324,  -554 },
    {   7387,   1806,  -5265,   545 },
    {   7450,  -2302,  -4445,  1418 },
    {   8817,  -1370,  -5827,  2168 },
    {  10324,  -2406,  -5629,  2579 },
    {   8863,  -2578,  -3537,   467 },
    {   6901,  -1624,  -3169,  3392 },
    {   7846,    156,  -6948,  3381 },
    {   7928,  -1115,  -5972,  4816 },
    {   6089,   -599,  -4368,  -320 },
    {   7833,   1246,  -3960,  -621 },
    {   8931,   2521,  -6768,  2052 },
    {   8900,   1944,  -4126,    40 },
    {   7661,    -34,  -2855,  2480 },
    {   5873,    474,  -3262,  3712 },
    {   7535,   -234,  -4699,   216 },
    {   5856,    143,  -5142,    73 },
    {   8944,   -106,  -5874,  3663 },
    {   7134,    426,  -5879,  2895 },
    {  10199,   1011,  -4762,   369 },
    {   8454,    264,  -5971,  1291 },
    {   7822,  -2449,  -4333,  4540 },
    {   6200,  -2758,  -2632,  1497 },
    {   6070,  -4315,  -2699,   414 },
    {   7047,  -3739,  -3210,  1060 },
    {   5675,  -3801,  -2717,  -407 },
    {   4789,  -4063,  -2628,  -744 },
    {   4023,  -3366,  -3133,  -726 },
    {   4296,  -2407,  -3381,  -513 },
    {   4388,  -2931,  -2820,  1512 },
    {   4559,  -4233,  -1941,  1976 },
    {   6702,  -3208,  -1755,  1680 },
    {   4416,  -3521,  -1052,  2984 },
    {   7154,  -4266,  -1203,  3732 },
    {   3625,  -4242,  -3244,  1395 },
    {   6518,  -2856,  -1304,  2887 },
    {   6170,  -1949,  -3014,  3973 },
    {   5189,  -2451,  -4020,  3477 },
    {   6218,  -2988,  -1921,  3844 },
    {   4827,  -3688,  -1928,  3343 },
    {   6668,  -3991,  -2805,  3095 },
    {   5297,  -3115,  -3684,  2390 },
    {   5354,  -4614,  -2662,  1504 },
    {   4196,  -3091,  -4147,  1135 },
    {   3540,  -2893,  -4007,   100 },
    {   5569,  -1602,  -4007,  1909 },
    {   4341,  -2091,  -4272,   252 },
    {   5559,  -2878,  -3832,   498 },
    {   4548,  -4479,  -2898,   -27 },
    {   5176,  -2494,  -4635,  1476 },
    {   3294,  -3485,  -3738,   716 },
    {   4920,  -1229,  -4195,  -365 },
    {   3257,  -3518,  -3349,  2862 },
    {   5286,  -1948,  -3485,  -778 },
    {   6502,  -3051,   -152,  2854 },
    {   5864,  -4192,  -1076,  3451 },
    {   4656,  -3122,  -3448,   179 },
    {   5907,   -754,  -1596,  3116 },
    {   7229,  -3680,  -1590,  2892 },
    {   5107,  -3888,  -3364,   806 },
    {   6764,  -2635,  -3450,   134 },
    {   5258,  -2827,  -2844, -1052 },
    {   5798,  -1725,  -4305,   205 },
    {   5404,  -1213,  -3362,   449 },
    {   6224,  -2738,  -3046,  -581 },
    {   4223,  -2438,  -2725,  3745 },
    {   4751,  -3411,  -2123,   116 },
    {   3868,  -3000,  -3954,  2297 },
    {   6819,  -2899,  -4277,  2825 },
    {   4207,  -4754,  -2808,   865 },
    {   4804,  -1494,  -1997,  4688 },
    {   5282,  -2213,   -548,  3559 },
    {   5580,  -1912,   -566,  4370 },
    {   6168,  -2857,   -672,  4053 },
    {   6583,  -4515,  -2850,  1670 },
    {   6511,  -3093,  -3988,  1421 },
    {   4646,  -1790,  -1443,  3650 },
    {   5915,   -924,  -2020,   896 },
    {   7814,  -4181,  -3152,  2007 },
    {   6190,  -2238,  -4817,  2279 },
    {   4737,  -4034,  -3288,  1835 },
    {   8161,  -3633,  -3423,  3137 },
    {   7415,  -2351,  -2088,  4290 },
    {   4106,  -2517,    -62,  2905 },
    {   4909,  -3145,   -614,  4112 },
    {   4938,  -3281,   -397,  1100 },
    {   -173,    919,   1589, -5363 },
    {    -13,    796,   -295, -6655 },
    {  -1860,   -829,   1141, -4555 },
    {   2298,   -838,   -664, -5005 },
    {   -884,  -1097,   2074, -4613 },
    {   -101,    281,   2846, -4535 },
    {   1166,    453,   2429, -5910 },
    {    879,   -664,   2370, -5452 },
    {   1415,   -370,  -1699, -4727 },
    {  -1413,   1277,   -669, -6649 },
    {   2133,    304,   -968, -4624 },
    {    380,    586,  -2087, -4892 },
    {   1336,    275,    -82, -5789 },
    {  -2459,   1057,    -34, -5416 },
    {   2278,  -1758,    866, -5653 },
    {   1945,  -2295,   -149, -5302 },
    {   1287,  -3525,    996, -5255 },
    {   2297,    803,   1177, -6067 },
    {    187,   -180,   -619, -6202 },
    {   -793,  -2537,   1554, -5057 },
    {  -2703,   -204,   -629, -5853 },
    {  -1007,   -146,    313, -5582 },
    {    830,    357,    869, -6363 },
    {   -228,   -575,  -3177, -4433 },
    {  -1001,  -1553,   -142, -5708 },
    {  -1644,   1683,   1721, -4533 },
    {    893,   1924,    -15, -5791 },
    {   2195,   2061,   -262, -5471 },
    {   3031,    270,    311, -5096 },
    {   1912,   1638,  -1523, -4677 },
    {  -3142,    -55,    253, -4914 },
    {    356,  -1680,    343, -6123 },
    {  -2241,  -1734,   -976, -5939 },
    {  -2196,  -2893,    547, -4938 },
    {  -1245,    126,  -1916, -5419 },
    {   -249,  -3755,  -1422, -5594 },
    {    575,  -2683,  -1926, -4566 },
    {   -762,   1885,    192, -5880 },
    {   -811,  -2562,  -1068, -6013 },
    {  -2264,  -3086,   -976, -4775 },
    {     70,  -1215,   2880, -4410 },
    {    714,  -3760,   2916, -4691 },
    {   -244,  -3404,   1740, -4493 },
    {    684,  -5137,   -328, -5608 },
    {   -529,  -3825,  -1786, -4535 },
    {   -713,  -4743,  -1118, -5546 },
    {   2718,  -3788,   1798, -5708 },
    {  -1639,  -3679,  -1564, -6095 },
    {   1693,  -2642,  -1389, -4539 },
    {    505,  -1573,  -1651, -4878 },
    {   -835,  -2256,  -1941, -5352 },
    {   1464,   -411,   1993, -6441 },
    {    493,  -3184,   -145, -6148 },
    {  -1413,    499,  -1617, -6479 },
    {   -294,   1722,  -1419, -5725 },
    {  -2937,  -1528,   -175, -4624 },
    {   -594,  -5911,    -56, -6146 },
    {   -300,  -4275,   1156, -5947 },
    {    552,  -2643,   2669, -3959 },
    {    905,  -4158,   1789, -5809 },
    {   1336,  -2009,   2108, -5903 },
    {   1555,  -3600,   1110, -6759 },
    {  -1294,  -3464,     77, -6084 },
    {  -1139,  -4006,  -1270, -4181 },
    {  -5094,  -3296,   1092, -2847 },
    {  -5503,  -2883,   1984, -2067 },
    {  -4671,  -4218,  -1417, -4132 },
    {  -3763,  -3818,   1262, -3082 },
    {  -5132,  -3430,   2928,  -728 },
    {  -5957,  -2877,   1251, -2446 },
    {  -4425,  -2319,   -212, -4276 },
    {  -6201,  -1993,   1774, -2182 },
    {  -5500,  -3836,   2201, -1396 },
    {  -6934,  -2334,   2366, -1293 },
    {  -6124,  -4140,   1337, -1977 },
    {  -6553,  -4186,   1756, -1325 },
    {  -5126,  -1258,    744, -3656 },
    {  -5167,  -1390,   1581, -2895 },
    {  -4525,  -3398,   2429, -1865 },
    {  -4076,  -3183,   2027, -2510 },
    {  -6191,  -3274,   1838, -1814 },
    {  -4454,  -2753,   2723, -1185 },
    {  -6655,  -4797,    251, -2595 },
    {  -6332,  -2232,   1832,   217 },
    {  -5869,  -1698,    134,   340 },
    {  -6614,  -1045,   2126, -1932 },
    {  -4859,  -2107,   2010, -2435 },
    {  -6274,  -1622,   2808, -1374 },
    {  -3119,  -3209,    521, -3988 },
    {  -5676,  -2082,   -420, -2711 },
    {  -7073,  -3623,    696, -2343 },
    {  -5986,  -4224,    572, -2454 },
    {  -4340,  -4521,    882, -2771 },
    {  -6178,  -1933,    535, -1444 },
    {  -4923,  -4163,   1744, -2066 },
    {  -6410,  -1519,   1058, -2683 },
    {  -5077,  -1185,    856, -2216 },
    {  -7091,  -2444,    687, -2597 },
    {  -5284,  -2165,   3239,  -993 },
    {  -4763,  -1497,    197, -3179 },
    {  -4128,  -4958,   -396, -3578 },
    {  -5054,  -3878,   -647, -2672 },
    {  -7005,  -3348,   1679, -1579 },
    {  -5767,  -1017,   2582, -1915 },
    {  -7069,  -2787,   1331, -2070 },
    {  -5532,  -2296,    706, -2950 },
    {  -5059,  -3543,   -821, -3637 },
    {  -6639,  -1835,   1016,  -696 },
    {  -5611,  -5220,   -694, -3371 },
    {  -5994,  -2803,   2933,  -729 },
    {  -5948,   -619,   1596, -2676 },
    {  -5486,  -4419,    153, -3265 },
    {  -4329,  -3440,   1646, -1439 },
    {  -4083,  -3978,    177, -3569 },
    {  -4289,  -2599,   1224, -3075 },
    {  -5707,  -3253,   1912,  -759 },
    {  -6606,  -3437,   2562,  -571 },
    {  -5254,  -2444,    769,  -352 },
    {  -6545,  -3154,    582, -1103 },
    {  -5328,  -2241,   2566, -1775 },
    {  -7216,  -1936,   1538, -1983 },
    {  -3730,  -2451,    426, -3869 },
    {  -5110,  -1385,   2031, -1169 },
    {  -6470,  -2715,    269, -3123 },
    {  -5806,  -2480,    -97, -3832 },
    {  -3683,  -4916,   -490, -4330 },
    {  -6341,  -2083,   -669,  -115 },
    {  -4913,  -4079,   -837, -4673 },
    {  -3274,  -2497,   2334, -2652 },
    {  -1286,  -1731,   2550, -3756 },
    {  -3375,   -877,    926, -3977 },
    {  -2525,  -2079,   2879, -2625 },
    {  -5308,   -504,   3111, -1607 },
    {  -4904,    460,   4093, -1232 },
    {  -1993,   1616,   4656, -1913 },
    {  -3481,  -1176,   3119, -2236 },
    {  -4132,  -1502,   2339, -2545 },
    {  -2542,   1151,   3569, -2550 },
    {  -4381,    430,   3147, -2082 },
    {  -3888,    867,   3899, -1657 },
    {  -2861,   1290,   4202, -1979 },
    {  -3893,   -253,   2363, -2764 },
    {  -1705,    688,   3827, -2923 },
    {  -2223,   2312,   3700, -3148 },
    {  -1986,   -720,   5021,  -795 },
    {  -3177,    242,   1952, -3352 },
    {  -1854,   1509,   2528, -3815 },
    {  -3173,     97,   5019,  -706 },
    {  -2689,   -145,   1375, -3915 },
    {  -4838,   -385,   2488, -2427 },
    {  -4557,   -355,   1603, -3060 },
    {  -3522,   1832,   3292, -2674 },
    {  -3769,    780,   2378, -2704 },
    {  -4323,  -1932,   3414, -1169 },
    {  -2740,   1158,   2729, -3273 },
    {  -3647,    210,   1464, -2892 },
    {  -2342,  -2097,   1513, -3727 },
    {  -4422,  -1242,   3130, -1833 },
    {  -1308,  -1039,   4290, -1875 },
    {  -1754,  -2535,   3298, -2314 },
    {  -4102,   -186,   4037, -1094 },
    {  -1008,   1570,   3290,   171 },
    {  -3322,  -2621,   2791, -1536 },
    {  -2539,  -2597,   3442, -1672 },
    {  -3411,  -2015,   3670, -1174 },
    {  -2097,    730,   5581, -1399 },
    {  -1510,    -74,   4820, -2004 },
    {  -4086,   -868,   4425,  -771 },
    {   -956,   -986,   3640, -2925 },
    {  -2087,  -1250,   3464, -2458 },
    {  -3308,  -2411,   1334, -3667 },
    {  -2264,   -389,   4004, -1854 },
    {   -680,    239,   4058, -3388 },
    {  -1357,     30,   2993, -3658 },
    {  -3601,   -552,   1177, -1136 },
    {  -2641,    442,   4374, -1625 },
    {  -2525,    770,   1640, -3895 },
    {  -3172,   -891,   3893, -1608 },
    {  -2996,     13,   3277, -2414 },
    {   -899,   1055,   4470, -2501 },
    {   -422,   -584,   3475, -3787 },
    {  -1978,   -593,   2566, -3415 },
    {  -3150,  -1280,   2362, -3047 },
    {  -3592,    224,   1026, -3932 },
    {  -4840,  -1189,   3633,  -879 },
    {  -3952,  -2255,   2916, -1826 },
    {  -1695,     28,   1810,  -349 },
    {   -745,  -2484,   3308, -3293 },
    {  -1016,   1563,   5365, -1823 },
    {  -2172,  -1787,   4266, -1287 },
    {  -1241,  -1951,   3982, -2413 },
    {  -2009,  -2639,   2330, -3480 },
    {   5105,  -1618,  -2588, -2015 },
    {   6497,  -1523,  -3218,  -910 },
    {   6526,  -2305,  -2029, -1790 },
    {   5289,    -99,  -3436,  -400 },
    {   5781,  -1623,  -1577, -2617 },
    {   5259,   -670,  -3125, -1700 },
    {   6343,  -1256,   -331, -3222 },
    {   7967,   -678,  -2195, -1462 },
    {   6119,   -695,  -2988, -1538 },
    {   6108,    494,  -3359, -1548 },
    {   5067,    969,  -2328, -2707 },
    {   7595,   -435,  -1497, -2056 },
    {   6929,   -719,  -2420, -1665 },
    {   5190,    584,  -2982, -2103 },
    {   6106,   -444,  -1411, -2739 },
    {   5584,    289,  -1804, -2803 },
    {   5276,    227,  -1180, -3361 },
    {   7544,  -1525,  -1834, -1725 },
    {   5986,  -1470,  -2606, -1701 },
    {   5096,   -765,  -1712, -3006 },
    {   5423,   -149,  -3933, -1157 },
    {   7651,     26,  -2445, -1507 },
    {   4745,   -464,  -1735, -2362 },
    {   5352,  -1011,  -1094, -1999 },
    {   6300,   -672,   -542, -1950 },
    {   6675,  -1020,  -1318, -1059 },
    {   7218,  -2036,   -603, -2462 },
    {   7755,  -1514,  -2430, -1229 },
    {   5041,    449,  -1056, -2405 },
    {   6710,  -2277,  -1344, -2284 },
    {   6824,  -1347,  -2254,   251 },
    {   6068,  -1857,   -983, -1316 },
    {   5603,  -2177,  -2730, -1477 },
    {   5838,  -1059,  -3604,  -970 },
    {   5076,   -789,   -335, -2413 },
    {   6191,  -1634,  -2000, -2129 },
    {   5092,  -1292,  -2543, -1034 },
    {   5305,    435,  -1710, -1850 },
    {   6140,    561,  -2176, -2380 },
    {   6752,    348,  -2496, -1890 },
    {   6405,    273,  -1098, -2778 },
    {   6942,  -1340,   -496, -1381 },
    {   5238,   -687,  -2454, -2349 },
    {   6959,   -882,  -1833, -2061 },
    {   6292,   -253,  -2125, -2199 },
    {   5838,   -574,   -759, -3215 },
    {   6954,  -1484,   -640, -2771 },
    {   7498,  -1706,  -1210, -2154 },
    {   6772,  -1003,  -1235, -2532 },
    {   6014,    228,  -2154, -1108 },
    {   6943,  -2178,  -2644, -1122 },
    {   7262,   -763,  -3056, -1090 },
    {   6273,  -1478,  -1072,   177 },
    {   4734,    425,  -2912,   357 },
    {   7129,    168,  -1537, -2327 },
    {   7204,   -434,   -746, -2660 },
    {   6879,     57,  -3087, -1310 },
    {   4623,   -610,   -718, -3459 },
    {   6565,   -543,  -1998,  -339 },
    {   4752,   -277,  -2066, -1405 },
    {   7435,  -1416,  -1904,  -505 },
    {   4076,    150,  -1222, -3556 },
    {   7082,    -28,  -1456, -1174 },
    {   5941,   -446,  -1326, -1158 },
    {   3870,  -1648,  -2474, -2589 },
    {    858,     37,  -3387, -3721 },
    {   3557,  -1503,  -1664, -3383 },
    {   3336,  -1972,  -3079, -2216 },
    {   3186,     60,  -4185,  -863 },
    {   3456,   -773,  -3066, -2457 },
    {   4131,   -913,  -2060, -2601 },
    {   4431,   -691,  -4114,  -972 },
    {   3461,   -334,  -3680, -1751 },
    {   2006,   -459,  -2214, -3827 },
    {   1322,     32,  -2816, -3203 },
    {   4425,  -1897,  -2791, -1946 },
    {   4504,     23,  -3421, -1909 },
    {   3090,   -885,  -2366, -3264 },
    {   3209,  -2363,  -3730,  -834 },
    {   3312,  -1471,  -3641, -1579 },
    {   4184,  -1669,  -3323, -1248 },
    {   2190,   -931,  -3302, -2944 },
    {   2947,   -229,  -4791, -1195 },
    {   2020,  -1626,  -2700, -3125 },
    {   2214,   -326,  -4352, -1683 },
    {   3286,  -2619,  -2412, -2458 },
    {   1000,  -2571,  -4129, -2158 },
    {   2496,  -2627,  -3611, -1433 },
    {   2043,  -2191,  -2167, -3827 },
    {   2571,  -2544,  -1915, -3222 },
    {   2022,  -1501,  -3856, -2165 },
    {   2685,  -1180,  -1461, -4038 },
    {   1610,  -2313,  -4391, -1173 },
    {   2340,  -2490,  -4215,  -516 },
    {   1742,  -2615,  -3632, -2146 },
    {    523,  -1293,  -4246, -2442 },
    {   3725,  -2723,  -3014, -1576 },
    {   3554,  -1381,  -4200,  -824 },
    {   1291,  -1594,  -4777, -1430 },
    {   1452,    515,  -2960, -3830 },
    {   4264,   -894,  -3305, -1826 },
    {   2606,  -1452,  -4522,  -966 },
    {   1196,   -830,  -4807, -1816 },
    {   1054,   -775,  -2616, -4071 },
    {   4206,    415,  -4344, -1132 },
    {   3044,    491,  -4126, -1934 },
    {    988,   -901,  -3353, -3443 },
    {   1729,  -3063,  -2267, -3370 },
    {   3915,    912,  -2989, -2387 },
    {   3781,    300,  -2457, -3050 },
    {   2712,    924,  -1350, -1206 },
    {   4230,    405,  -2343,   665 },
    {   1878,   -873,   -225,   -29 },
    {   3510,     56,  -1334, -3420 },
    {   2850,   1447,  -2651, -3150 },
    {   1510,   -706,  -4125, -2483 },
    {   3115,    793,  -1692, -3894 },
    {   2667,    213,  -2973, -2786 },
    {   1184,  -2384,  -3051, -3173 },
    {   2139,    796,  -2079, -3697 },
    {   1464,  -1483,  -3726, -2754 },
    {   2407,  -1148,  -3915, -1569 },
    {   2612,  -1779,  -3217, -2271 },
    {   2406,  -2870,  -2937, -2496 },
    {   2140,    126,  -3646, -2758 },
    {   2952,  -1036,    268, -1423 },
    {     93,  -1931,  -3841, -3535 },
    {    389,  -2953,  -3383, -3343 },
    {   8652,  -5511,  -1662,   565 },
    {   7427,  -2791,  -2535,  -842 },
    {   8541,  -4253,  -1407,  -988 },
    {   8018,  -3203,  -2998,   105 },
    {   7231,  -3926,   -958,  1308 },
    {   7331,  -3690,   -363,  2586 },
    {   6803,  -3646,  -2226,  -903 },
    {   8163,  -2811,   -477, -2235 },
    {   9356,  -3818,  -1685,  -684 },
    {   8466,  -2854,   -302,  -698 },
    {   8458,  -3224,    517,   279 },
    {   8074,  -2619,  -1326,  2596 },
    {   8779,  -2761,  -2527,  -441 },
    {   6533,  -2887,   -899,  -696 },
    {   7394,  -2305,  -1642,  -120 },
    {   8281,  -3780,    -22,  1305 },
    {   9158,  -4413,   -779,   901 },
    {   9031,  -5240,  -1109,  1678 },
    {   8717,  -3650,    410, -1075 },
    {   7317,  -3197,   -818, -2264 },
    {   7934,  -2385,  -1214, -1886 },
    {   8256,  -4441,   -291,  -587 },
    {   7358,  -3395,   1090,  -270 },
    {   9446,  -4910,  -1343,  -473 },
    {   8187,  -4726,   -808,  1166 },
    {   7504,  -3845,    -47,   267 },
    {   8029,  -2146,  -1283,  -383 },
    {   7461,  -2705,   -853,   783 },
    {   9367,  -3636,   -645,  -354 },
    {   8955,  -3473,   -308, -1947 },
    {   8676,  -2683,  -2099,  1485 },
    {   7481,  -3003,   -871,  -444 },
    {   8015,  -2839,  -1673,  1175 },
    {   6947,  -4643,  -1527, -1047 },
    {   7622,  -2575,   -137,  -960 },
    {   9388,  -4279,   -707, -1322 },
    {   8382,  -5259,  -1283,  -565 },
    {   6856,  -4138,  -1030,   630 },
    {   8659,  -2571,  -1124, -1666 },
    {   8763,  -3807,   -537,  2543 },
    {   8049,  -3578,  -2186,  -604 },
    {   8272,  -2351,  -1985, -1214 },
    {   6855,  -3796,  -1527, -1631 },
    {   7178,  -2896,  -1600, -1756 },
    {   7040,  -2888,    -89, -1586 },
    {   6261,  -3403,   -264,   998 },
    {   7756,  -4699,  -1543,  -834 },
    {   7682,  -4622,   -758, -1721 },
    {   8839,  -4232,  -2932,  1959 },
    {   9363,  -4679,  -1956,    39 },
    {   7883,  -3616,  -1414, -1432 },
    {   8828,  -3188,  -1356, -1312 },
    {   7746,  -3987,   -121, -2424 },
    {   9262,  -3256,   -693,   818 },
    {   7670,  -3420,   -148,  3504 },
    {   7344,  -3183,    608,  1595 },
    {   8976,  -4139,  -1848,  1304 },
    {   6708,  -4131,     33,  -852 },
    {   7840,  -4429,  -2275,    79 },
    {   8980,  -3858,  -2838,   453 },
    {   7815,  -4604,  -2563,   944 },
    {   8372,  -4422,  -1783,  3071 },
    {   8623,  -5128,  -1754,  2888 },
    {   7462,  -3281,    889,   920 },
    {   8416,    -59,  -1320, -1825 },
    {   7928,  -1488,   -414, -2499 },
    {   8110,   -977,  -1047, -2042 },
    {   8278,   -687,  -1597, -1550 },
    {   7988,   -174,   -977, -2106 },
    {   8609,  -1547,  -1628, -1527 },
    {   9000,  -1798,   -946, -1761 },
    {   8954,   -872,  -1404, -1594 },
    {   8939,    466,   -748, -1212 },
    {   9549,   -329,   -177, -1360 },
    {   9411,    -18,  -1126, -1568 },
    {   8859,   -782,   -488, -1338 },
    {   8955,   -218,    -43, -1209 },
    {   9131,    -69,   -453, -1001 },
    {   9069,  -1519,  -1091, -1199 },
    {   9247,  -1309,   -566, -1146 },
    {   8528,  -1617,   -287, -1313 },
    {   7763,   -745,   -149, -2040 },
    {   8294,   -343,    257, -2633 },
    {  10149,   -893,   -552, -1649 },
    {   9398,   -915,    218, -2042 },
    {   9703,  -1194,   -675, -1592 },
    {   9586,   -700,   -427, -1710 },
    {   8930,    497,  -1445, -1218 },
    {   9285,  -1323,   -163, -1552 },
    {   8431,  -1289,   -985, -1404 },
    {   8965,   -655,    653, -1483 },
    {   9542,  -1001,   -951, -1128 },
    {   9205,   -647,    -37,  -882 },
    {   8603,    -56,    514, -1793 },
    {   9300,    -12,  -1324,  -567 },
    {   8773,    238,   -184, -1456 },
    {   9941,  -1306,    -69, -1792 },
    {   9360,    279,   -376, -1919 },
    {   9180,   -285,     95, -2170 },
    {   9922,   -501,   -970, -1570 },
    {   8341,  -1493,   -856, -2092 },
    {   8780,   -981,   -850, -1014 },
    {   9721,   -548,  -1504, -1094 },
    {   9973,  -1493,    482, -2105 },
    {   8707,   -333,  -1027, -1087 },
    {   9098,   -469,   -315, -1723 },
    {   8879,  -1050,   -661, -2020 },
    {   8857,    602,   -866, -1918 },
    {   8945,  -1025,  -2154, -1071 },
    {   8484,  -1930,   -468, -2179 },
    {   9177,  -1903,   -224, -2112 },
    {   8652,   -137,  -2097, -1214 },
    {   9063,   -973,  -1405,  -772 },
    {   9328,   -456,    662, -2469 },
    {  10101,   -697,    127, -2113 },
    {   9685,    811,  -2359, -1024 },
    {   8586,    -94,   -460, -1982 },
    {   7924,   -141,   -509, -2513 },
    {   7773,   -669,   -107, -2835 },
    {   8636,  -1064,    -46, -2409 },
    {   9748,    596,  -1815, -1349 },
    {   8924,    304,    547, -2614 },
    {   9442,    746,  -1153, -1679 },
    {   9454,   -278,   -529, -1976 },
    {   8488,    561,    -32, -2160 },
    {  10083,    -63,  -1544, -1364 },
    {   9390,  -1278,    568, -1131 },
    {   9740,    -49,  -2253,  -910 },
    {   3636,  -2391,  -1115, -3614 },
    {   6014,  -3204,  -1902, -1808 },
    {   5787,  -3497,  -1116, -2590 },
    {   4365,  -3046,  -1632, -2668 },
    {   4733,  -2192,  -2029, -2468 },
    {   5412,  -2753,  -1633, -2464 },
    {   4455,  -3375,   -767, -3399 },
    {   4456,  -1644,   -983, -2841 },
    {   4039,  -2523,     38, -3967 },
    {   3406,  -2662,     72, -4757 },
    {   4279,  -2005,   1055, -4399 },
    {   4321,  -1377,   -860, -3786 },
    {   3743,  -5739,   -651, -3047 },
    {   3528,  -5510,    361, -4060 },
    {   6496,  -4886,   -136, -2689 },
    {   4513,  -5254,    551, -4010 },
    {   6557,  -3413,    -92, -3063 },
    {   4186,  -2059,    187,    47 },
    {   6210,  -4117,  -1256, -1985 },
    {   6038,  -4343,    351, -2124 },
    {   4305,  -4780,  -2077, -1897 },
    {   4480,  -3815,  -2228, -1533 },
    {   5582,  -3689,   1221, -3429 },
    {   5532,  -4874,   1195, -2765 },
    {   6518,  -2853,   -905, -2568 },
    {   5467,  -2192,    470, -4115 },
    {   4139,  -1577,    240, -3493 },
    {   5281,  -1926,   -729, -3340 },
    {   5214,  -2870,   1359, -4289 },
    {   3046,  -3510,  -1536, -3214 },
    {   5433,  -2881,  -1230, -1184 },
    {   4861,  -3932,  -1071, -2791 },
    {   5693,  -4234,  -1906, -1502 },
    {   4004,  -3935,  -1804, -2383 },
    {   3728,  -3792,    681, -4773 },
    {   3621,  -3030,  -1951, -2598 },
    {   5133,  -3903,     44, -3700 },
    {   3561,  -3451,   1183, -5301 },
    {   5026,  -2762,  -2341, -1780 },
    {   5841,  -2492,   -467, -3210 },
    {   5591,  -1791,    497, -2472 },
    {   5054,  -3898,  -1822, -2097 },
    {   5813,  -2792,     83, -1469 },
    {   4432,  -4497,   1670, -5193 },
    {   5338,  -4653,  -1109, -2200 },
    {   3239,  -4401,   -648, -3655 },
    {   2147,  -3598,  -1200, -4242 },
    {   4417,  -2271,  -1552, -3210 },
    {   6494,  -4360,    852, -3565 },
    {   2393,  -6358,   -856, -4524 },
    {   4959,  -4196,   -847, -1403 },
    {   4924,  -5438,   -226, -3026 },
    {   4254,  -5303,  -1306, -2424 },
    {   4121,  -3126,  -2334, -1981 },
    {   3437,  -4443,  -1464, -2953 },
    {   3203,  -3459,   -529, -4339 },
    {   5896,  -5945,    543, -3246 },
    {   1987,  -4733,   -220, -4863 },
    {   4358,  -4431,   -514, -3081 },
    {   4583,  -2416,   -492, -2287 },
    {   2943,  -5035,    419, -4927 },
    {   5358,  -5129,    987, -4309 },
    {   4460,  -3392,   1752, -5634 },
    {   3415,  -4633,   1507, -5945 },
    {    811,  -4692,   -445,  2333 },
    {   1009,  -5613,  -1857,  1360 },
    {   1338,  -2712,  -2720,  3036 },
    {   1002,  -3754,  -2582,  2344 },
    {    750,  -4608,  -2334,   714 },
    {   2043,  -3207,  -2822,  2173 },
    {   -140,  -4654,  -2953,   357 },
    {    -54,  -4026,  -2376,  2695 },
    {   1858,  -5022,   -717,  2287 },
    {   2064,  -3894,   -722,  3255 },
    {   2727,  -4558,   -332,  2603 },
    {   1810,  -5378,    283,  1826 },
    {   3935,  -4326,    762,  3383 },
    {   -767,  -4697,  -2510,  1922 },
    {   2146,  -4312,  -3090,  1641 },
    {     54,  -5881,  -2114,   921 },
    {   1992,  -5766,   -640,  1574 },
    {   1200,  -5371,  -1114,  1828 },
    {   2973,  -5337,     34,  2266 },
    {   1531,  -5018,  -2817,  1192 },
    {   3078,  -4570,    117,  1990 },
    {    924,  -4286,  -1388,  2713 },
    {    142,  -5058,  -2848,  1487 },
    {   -106,  -6180,   -881,   842 },
    {    673,  -5433,   -229,  1596 },
    {    783,  -5710,  -2784,   562 },
    {   1935,  -5729,  -2009,   856 },
    {   -410,  -3375,  -3326,  2734 },
    {    234,  -3000,  -2628,  3260 },
    {    733,  -3405,  -3806,  1589 },
    {    771,  -4285,  -3544,  1314 },
    {   1192,  -3563,  -3960,  2178 },
    {    206,  -5555,  -1250,  1546 },
    {   -130,  -3815,  -1210,  3041 },
    {    646,  -3940,   -393,  2992 },
    {   -184,  -4931,  -1767,  1925 },
    {   2746,  -5120,  -2275,  1464 },
    {   2440,  -3731,  -3352,  2729 },
    {   -490,  -4942,  -3779,   997 },
    {     68,  -2636,  -4167,  3778 },
    {     48,  -3986,  -4118,  2106 },
    {   -978,  -5486,  -1336,  1390 },
    {   1126,  -5297,   -855,   640 },
    {   -472,  -3975,  -3622,  1557 },
    {   2456,  -5344,  -1523,  1648 },
    {   -774,  -5652,  -2417,  1147 },
    {    995,  -6122,   -812,  1132 },
    {   3282,  -4571,  -1763,  2175 },
    {   3655,  -3862,   -676,  3568 },
    {   3038,  -3647,  -1672,  3381 },
    {   2595,  -2964,  -2772,  3263 },
    {   4176,  -3353,  -1148,  4354 },
    {   1603,  -3442,  -1500,  3444 },
    {    828,  -6226,  -1783,   678 },
    {   1421,  -3333,  -3080,  3403 },
    {   1121,  -4727,  -1924,  1984 },
    {   -186,  -5083,   -682,  1796 },
    {    819,  -2778,  -3488,   530 },
    {    421,  -2873,  -3832,  2596 },
    {   2164,  -4263,  -1605,  2282 },
    {    585,  -4437,   -682,  -491 },
    {   -644,  -4452,  -1157,  2325 },
    {   1991,  -4299,    210,  2834 },
    {   2135,  -3632,  -2113,   665 },
    {  -7482,  -2724,  -2662, -1380 },
    {  -6983,  -2166,  -3756, -3509 },
    {  -7085,  -1439,  -2397, -3112 },
    {  -7760,  -3049,  -3319, -2822 },
    {  -8413,  -2760,  -4406, -3298 },
    {  -5995,  -3943,  -1260, -3750 },
    {  -7879,  -1554,  -3464, -2606 },
    {  -6314,  -2034,  -3878, -1681 },
    {  -8849,  -2084,  -1399, -1231 },
    {  -7153,  -2602,  -1384,  -817 },
    {  -8041,  -2571,   -407, -2785 },
    {  -7246,  -2233,  -1578,   260 },
    {  -7336,  -3883,  -4061, -1342 },
    {  -7619,  -3908,  -2342,   382 },
    {  -8684,  -3724,  -1662,  -727 },
    {  -7850,  -2922,  -1770, -3449 },
    {  -6766,  -2034,  -1293, -1988 },
    {  -6895,  -2116,   -968, -3744 },
    {  -7136,  -5147,  -2618, -2809 },
    {  -8224,  -3724,  -2519, -1589 },
    {  -6711,  -2750,  -3021,  -219 },
    {  -8059,  -1638,  -1102, -3175 },
    {  -8710,  -4839,  -3963, -3143 },
    {  -9363,  -4965,  -3257, -1002 },
    {  -6099,  -1751,  -3157,  -395 },
    {  -6453,  -3216,  -4597,  -483 },
    {  -7879,  -5477,   -839, -2638 },
    {  -7202,  -4038,   -526, -2856 },
    {  -8022,  -1228,  -1910, -1646 },
    {  -9117,  -1393,  -1582, -2535 },
    {  -9095,  -2693,   -636, -2605 },
    {  -9076,  -2580,  -3481, -2519 },
    {  -8327,  -4859,  -2422,    83 },
    {  -8368,  -2129,  -2324, -2173 },
    {  -8554,  -4563,  -3842, -2007 },
    { -10462,  -4261,  -1934, -2084 },
    {  -9717,  -3187,  -2294, -1896 },
    {  -9625,  -3889,  -3020, -3224 },
    {  -9857,  -4955,  -4239, -2184 },
    {  -9752,  -2351,  -2277, -3129 },
    {  -7219,  -1302,  -2639, -1603 },
    {  -7477,  -4360,  -3718,  -559 },
    {  -5680,  -2033,  -2326, -3078 },
    { -10190,  -5548,  -4643, -3601 },
    {  -9431,  -4121,   -879, -2479 },
    {  -8365,  -5450,  -2020, -1439 },
    {  -6289,  -5178,  -1605, -3845 },
    {  -8319,  -3866,   -687, -2792 },
    {  -8131,  -1031,  -3608, -3947 },
    { -10510,  -2560,  -1199, -2082 },
    { -11015,  -3640,  -2748, -3041 },
    {  -8762,  -5022,  -5231, -1162 },
    { -10153,  -2715,  -4648, -4859 },
    {  -7930,  -5205,  -1900, -3600 },
    {  -9561,  -3548,  -4812, -3722 },
    {  -7663,  -4709,  -1180, -1475 },
    {  -9073,  -5707,  -1815, -2980 },
    {  -8602,  -2363,  -2675, -3770 },
    {  -9967,  -5614,  -3575, -3838 },
    {  -8324,  -1005,  -2131, -3254 },
    { -10331,  -5737,  -2550, -2940 },
    {  -8234,  -3354,  -3361, -4479 },
    {  -8140,  -1951,  -4526, -4545 },
    {  -6679,  -2662,  -2284, -4182 },
    {  -1122,  -1514,  -6427,  -212 },
    {     54,  -1660,  -5424, -1404 },
    {    254,  -2778,  -5222,   846 },
    {   -267,  -1661,  -6577,   814 },
    {   -305,  -2021,  -5759,  1484 },
    {  -1791,  -2446,  -6867,   -86 },
    {  -2929,  -3158,  -6603, -1799 },
    {  -1391,  -3189,  -5557, -1053 },
    {  -1602,   -884,  -6767, -1213 },
    {   -361,   -318,  -6219,   -44 },
    {  -4078,  -2635,  -5523,  -433 },
    {   -956,    478,  -4382,  1470 },
    {  -3300,  -2462,  -6021, -2721 },
    {    708,  -2434,  -5085,  -540 },
    {  -2435,  -3607,  -5647, -2110 },
    {   -491,  -1134,  -4681, -2886 },
    {     87,  -3435,  -4641, -1194 },
    {   -586,  -2927,  -4784,   366 },
    {  -1394,  -2326,  -6021,   350 },
    {     97,  -2519,  -4678, -2120 },
    {  -1547,  -1907,  -5069, -2993 },
    {    268,  -3724,  -4719,   127 },
    {   -827,  -1190,  -5912,  1144 },
    {  -3959,  -2322,  -6898, -1974 },
    {  -2728,  -2228,  -6426,  -562 },
    {   -456,   -666,  -5785, -1609 },
    {    531,  -1096,  -5731,  -656 },
    {  -3569,   -688,  -3915,   110 },
    {  -4752,  -1725,  -4393,  -377 },
    {  -3210,  -3315,  -6960,  -840 },
    {   -688,  -3416,  -4971,  1221 },
    {  -1833,     77,  -6491, -2434 },
    {   -239,   -255,  -6850,  -886 },
    {  -2112,  -1490,  -6291, -2689 },
    {  -1544,  -4579,  -5198, -1261 },
    {  -2771,  -4014,  -5520,   683 },
    {  -1635,  -2829,  -5512,  1214 },
    {   -958,  -2582,  -4823,  2360 },
    {  -2077,  -4566,  -4642,   365 },
    {  -3112,  -4214,  -5960,  -823 },
    {  -2467,  -2510,  -4858,  1467 },
    {  -1561,  -3399,  -5822,   211 },
    {   -775,  -1081,  -4424,  2636 },
    {  -1263,     25,  -6378, -1392 },
    {  -3476,   -366,  -5417, -1393 },
    {  -3176,  -1476,  -4149,  1466 },
    {  -2479,    518,  -4448,  -257 },
    {  -2992,    158,  -4660, -1279 },
    {  -1320,  -3872,  -4479,  1147 },
    {  -1475,   -312,  -5318,   539 },
    {  -3527,  -1679,  -5860, -1681 },
    {  -3397,  -3438,  -5593,  1866 },
    {  -4089,  -2439,  -4763,  1275 },
    {   -748,  -4513,  -4687,   -48 },
    {  -2166,  -4531,  -4691, -2856 },
    {  -2385,   -853,  -6035,  -627 },
    {  -1194,  -4091,  -4472, -1963 },
    {   -682,  -3234,  -4084, -3033 },
    {  -3255,  -5015,  -5328,   -12 },
    {  -2313,  -3436,  -4601,  -155 },
    {  -2792,  -1038,  -6947, -2019 },
    {  -1244,  -1526,  -5771, -1882 },
    {  -4679,  -3731,  -5506,   283 },
    {  -3062,    -66,  -3558,  -758 },
    {  -4895,  -1187,   4751,  3728 },
    {  -7600,  -2752,   3320,  4613 },
    {  -5703,  -2975,   3944,  2659 },
    {  -4972,  -1257,   -246,  2952 },
    {  -4221,  -2487,   1702,  4295 },
    {  -2900,  -1529,   2458,  4935 },
    {  -5061,    407,   2416,  4050 },
    {  -6931,  -3478,   2761,  2213 },
    {  -6037,  -3921,   3192,  1866 },
    {  -6113,   -811,   2407,  3782 },
    {  -5878,  -1716,   1207,  3478 },
    {  -5953,  -2853,   2207,  2712 },
    {  -6807,  -3223,   2749,  3595 },
    {  -3272,  -3157,   1389,  3788 },
    {  -5368,  -1904,   1980,  5077 },
    {  -7235,  -1398,   3075,  4548 },
    {  -4765,  -3487,   2755,  2796 },
    {  -7658,  -4435,   2694,  2582 },
    {  -6997,  -4282,    456,  3832 },
    {  -5563,  -3115,    -63,  3713 },
    {  -4244,  -4220,   1450,  2767 },
    {  -3801,  -2194,    190,  4303 },
    {  -5458,  -4119,   1958,  2274 },
    {  -7300,  -3469,   3514,  3193 },
    {  -4594,  -2067,    775,  4752 },
    {  -3389,  -1654,   1464,  5412 },
    {  -4845,  -3483,    964,  3437 },
    {  -6007,  -2818,   1666,  4659 },
    {  -8709,  -5007,   1757,  3287 },
    {  -5833,  -4389,   1025,  3171 },
    {  -5788,  -1780,   3944,  3661 },
    {  -4430,   -920,   1938,  4753 },
    {  -7066,  -1857,   4591,  4538 },
    {  -3549,   -513,   1427,  5317 },
    {  -7517,  -1220,   2883,  3049 },
    {  -7605,  -2687,   1874,  2735 },
    {  -8718,  -4035,   2676,  3730 },
    {  -7990,  -3907,   1185,  2607 },
    {  -6058,  -1744,   3349,  5157 },
    {  -5954,    565,   3161,  3250 },
    {  -6478,   -612,   1930,  2271 },
    {  -6535,  -1445,     -2,  1618 },
    {  -8963,  -4151,   1192,  4044 },
    {  -7227,  -3570,   1600,  4234 },
    {  -4674,     79,    595,  3015 },
    {  -3974,    430,   2727,  5137 },
    {  -5299,      9,   3714,  4779 },
    {  -6779,  -2699,     -8,  2436 },
    {  -7016,  -1145,   1293,  2310 },
    {  -6955,  -3312,   1534,  1801 },
    {  -4025,    740,   1850,  4054 },
    {  -9589,  -3460,   4154,  5270 },
    {  -4404,  -1181,   4298,  5173 },
    {  -7356,  -4583,    -18,  2644 },
    {  -6516,  -1235,   4439,  6234 },
    {  -3453,   -301,   4344,  4464 },
    {  -4643,   1530,   3315,  4340 },
    {  -4575,  -2557,   3754,  3682 },
    {  -3643,  -3501,   2051,  2997 },
    {  -5412,  -2475,   2301,  1579 },
    {  -5846,    259,   1360,  2348 },
    {  -5258,  -1358,   1050,   838 },
    {  -5542,   -219,   6377,  5750 },
    {  -5713,  -2952,    922,   899 },
    {  -2049,  -1135,   5206,  1033 },
    {  -1693,  -1886,   4835,  -106 },
    {  -2344,  -3504,   4232,   -13 },
    {  -2475,  -2334,   5043,  1126 },
    {   -787,  -2549,   3880,  2138 },
    {  -3159,  -2341,   4830,  2887 },
    {  -1780,  -1009,   6240,  2061 },
    {  -4327,  -3363,   2818,   886 },
    {  -3376,  -2743,   4104,   207 },
    {  -3250,  -4640,   2718,  1498 },
    {   -382,  -1075,   4382,  3460 },
    {  -2416,  -4168,   3530,   816 },
    {  -1756,  -2708,   4861,   622 },
    {  -1879,  -2097,   5156,  2889 },
    {  -2496,  -2418,   3722,  2671 },
    {  -2717,  -3252,   3341,  1944 },
    {  -4063,  -4091,   3306,   267 },
    {  -3549,  -3808,   3747,   842 },
    {  -2635,    546,   5794,  1894 },
    {  -1857,  -1121,   4383,  3964 },
    {  -2226,  -2166,   3489,  3678 },
    {  -3492,   -660,   5323,  1063 },
    {  -3033,  -3130,   4382,  1828 },
    {  -2703,   -625,   6369,  2851 },
    {  -1656,  -2842,   4584,  -528 },
    {  -4781,  -2622,   4390,  2097 },
    {   -413,  -2045,   5081,  3035 },
    {  -3810,  -2662,   4532,  1095 },
    {  -3144,  -1858,   5215,  1880 },
    {  -3562,  -1795,   4928,   670 },
    {  -4800,  -1509,   5189,  1859 },
    {  -1085,  -3832,   4169,   900 },
    {  -1969,  -3270,   2857,  2878 },
    {  -4267,  -4140,   3176,  1805 },
    {  -5145,  -3727,   3524,  1168 },
    {  -1346,  -1876,   5501,  1748 },
    {  -4998,  -2945,   3699,   338 },
    {  -3458,  -3096,   3406,  -635 },
    {  -1751,  -3209,   3508,   395 },
    {  -2507,    170,   5987,   705 },
    {  -3756,  -1072,   5647,  3536 },
    {  -2870,  -1439,   5026,  3212 },
    {  -3913,  -3225,   3669,  2144 },
    {  -3739,    226,   5747,   764 },
    {  -2052,   -820,   5266,  3093 },
    {  -3214,  -3820,   2409,  2391 },
    {  -4398,  -2588,   3501,  -218 },
    {  -4484,  -1763,   4180,  -198 },
    {  -3368,  -1525,   4362,  -134 },
    {  -2407,    224,   4905,  3533 },
    {  -1369,  -2937,   4728,  1788 },
    {  -4848,  -1707,   4159,   851 },
    {  -3454,  -1749,   4281,  3230 },
    {  -1990,  -3853,   3487,  1735 },
    {  -3117,     92,   6155,  4075 },
    {  -2676,  -2472,   4078,  -589 },
    {  -1547,  -2012,   2626,  1835 },
    {  -4275,   -588,   4824,   725 },
    {   -601,  -2249,   3736,  3548 },
    {  -4060,    -61,   5333,  3097 },
    {  -4303,      7,   6551,  3054 },
    {  -5003,  -1029,   5786,  3319 },
    {  -2810,   -728,   5392,   199 },
    {  -1232,   -200,   5228,  3121 },
    {   2621,    165,  -6255,   298 },
    {   3669,    537,  -6844,  1564 },
    {   1598,  -1190,  -6235,  2523 },
    {   2164,    -32,  -6894,  1383 },
    {    853,  -1597,  -6069,  1449 },
    {   1377,  -1661,  -5266,   108 },
    {   2660,     48,  -5172,  -517 },
    {   1903,   -391,  -5677,  1010 },
    {   3792,    206,  -5274,   -11 },
    {   1239,   2776,  -2929,  2721 },
    {   4071,    149,  -7259,  3125 },
    {   1436,   -480,  -6156,  -196 },
    {   1373,  -1960,  -5005,  3122 },
    {   3413,  -1271,  -5176,  3283 },
    {   3060,    -68,  -6495,  2238 },
    {   2700,  -2075,  -4681,    91 },
    {   2928,  -1728,  -5168,  1858 },
    {   4424,    828,  -4471,    88 },
    {   2672,  -2604,  -4038,  2753 },
    {   5223,   -123,  -6749,  2295 },
    {   4237,   -420,  -5538,  1353 },
    {   4744,  -1281,  -4097,  4708 },
    {   1103,  -2764,  -4751,  2024 },
    {   3747,  -1913,  -3911,  3960 },
    {   2470,  -1416,  -5542,   615 },
    {   4847,  -1354,  -5334,  1733 },
    {   5336,     88,  -7593,  4007 },
    {   2388,  -2880,  -4807,  1037 },
    {   4495,   1391,  -5685,  -139 },
    {   5253,   1637,  -6450,  1533 },
    {   1199,    795,  -5515,  1261 },
    {   1397,  -1259,  -4252,  3838 },
    {    746,     70,  -6640,   604 },
    {   1584,    166,  -4972,  3072 },
    {    380,   -999,  -5397,  2267 },
    {   2974,   1707,  -3242,  5360 },
    {   5202,   -403,  -5453,  2832 },
    {   3718,  -1731,  -4760,   714 },
    {   4150,   -975,  -4792,    61 },
    {   2925,   -818,  -4841,    15 },
    {   5301,    577,  -4006,  3259 },
    {   5265,   1986,  -5679,  3028 },
    {   3752,   1928,  -4509,  3729 },
    {   3278,   1925,  -6370,  1247 },
    {   5107,   1721,  -4853,  3127 },
    {   3279,   2982,  -2515,  4005 },
    {   4622,    668,  -6204,   759 },
    {   6034,    317,  -5763,  4818 },
    {   -558,     57,  -3785,  2817 },
    {   4476,   1616,  -3965,  4536 },
    {   5953,   2056,  -8215,  2715 },
    {   4387,   2613,  -7463,   868 },
    {   5834,   1088,  -4736,  4924 },
    {   6473,   -856,  -6991,  4172 },
    {   4959,   -293,  -5162,    76 },
    {   2731,   -843,  -6119,  3847 },
    {   3245,   1202,  -6833,   616 },
    {   2553,   1383,  -3829,  3859 },
    {   4332,   2099,  -3480,  3622 },
    {   2110,   2683,  -2728,  3990 },
    {    876,   1167,  -3290,  3466 },
    {   3991,   1709,  -2410,  4077 },
    {   5105,    939,  -2584,  3256 },
    {   4719,    688,  -1566,  3040 },
    {  -3632,   4335,   1266, -3303 },
    {  -4956,   3207,   1312, -2806 },
    {  -4669,   2627,   2663, -2435 },
    {  -4282,   3708,   2303, -3038 },
    {  -4536,   2297,   -175, -3350 },
    {  -5234,   2503,   -139,  -880 },
    {  -3978,   1512,   1092, -3619 },
    {  -4519,   4649,   1363, -2455 },
    {  -5118,   3132,   1961, -1577 },
    {  -5196,   3379,   -182, -1378 },
    {  -6420,   4486,   2397, -1993 },
    {  -5030,   5046,   1292, -1118 },
    {  -4559,   2573,   -927, -1406 },
    {  -3501,   3730,    691, -4930 },
    {  -4364,   2758,   1007, -3909 },
    {  -4026,   2839,  -1559, -2340 },
    {  -5037,   4053,    836, -1571 },
    {  -4727,   5136,   1110, -3588 },
    {  -5245,   2799,   -999, -2164 },
    {  -4954,   1501,    422, -3963 },
    {  -5994,   2726,   1462, -2833 },
    {  -5621,   5159,   2038, -2512 },
    {  -4991,   2291,   1917, -3151 },
    {  -5469,   4382,   -148, -2978 },
    {  -5858,   1983,    807, -2720 },
    {  -4709,   3556,    952,  -467 },
    {  -2489,   2362,   1714, -4230 },
    {  -4717,   5004,  -1180, -3672 },
    {  -5914,   3653,   1359, -1317 },
    {  -5506,   2995,    780, -1059 },
    {  -5287,   3945,   2480, -2293 },
    {  -3849,   4358,    322, -1770 },
    {  -3911,   3570,    252, -3185 },
    {  -3660,   5128,    158, -3719 },
    {  -4599,   3277,   -503, -2727 },
    {  -3673,   3760,  -1252, -3339 },
    {  -5161,   2337,    388, -1943 },
    {  -3529,   2216,   2156, -3080 },
    {  -4309,   4331,   1808, -1460 },
    {  -4782,   3820,    480, -2504 },
    {  -4166,   3544,   -378, -1567 },
    {  -5572,   2466,   -418, -2909 },
    {  -6096,   2930,    119, -1878 },
    {  -5963,   3554,   1011, -2233 },
    {  -6433,   4335,    935, -2930 },
    {  -5004,   3314,  -1352, -3430 },
    {  -6042,   3463,  -1008, -3940 },
    {  -4671,   2214,   -640, -5040 },
    {  -2795,   3759,   1412, -3803 },
    {  -3647,   4436,    729,  -515 },
    {  -3594,   1033,     56, -4148 },
    {  -2908,   3027,   2889, -3485 },
    {  -3338,   2234,    313, -4285 },
    {  -3825,   4497,   -561, -2634 },
    {  -6167,   3012,    -48, -3149 },
    {  -4828,   3515,   -969, -4475 },
    {  -5789,   2757,   -539, -4173 },
    {  -2452,   3067,    564, -4249 },
    {  -4921,   1358,   1331, -2889 },
    {  -3127,   4239,  -1045, -1523 },
    {  -4780,   2326,  -1118, -3446 },
    {  -3908,   5546,    152, -2622 },
    {  -6972,   2976,    337, -2809 },
    {  -4839,   4613,    -35, -4077 },
    {  -1408,   4822,  -1149, -4997 },
    {   -981,   4979,   -912, -6304 },
    {  -2098,   5689,   -888, -2878 },
    {  -3343,   4814,   -657, -4434 },
    {  -2461,   3601,   -967, -4869 },
    {  -2652,   3944,     87, -5520 },
    {  -1104,   6076,    174, -6407 },
    {    355,   5370,  -1721, -5869 },
    {   1242,   4497,  -1107, -5091 },
    {    -89,   4002,  -1491, -5182 },
    {   1059,   5693,  -1591, -4905 },
    {   1323,   4682,  -2078, -4768 },
    {    818,   3996,   -549, -5468 },
    {   -287,   4529,    929, -5543 },
    {   -919,   5519,  -2791, -2844 },
    {  -1407,   5679,  -3289, -3974 },
    {   -189,   6530,  -3547, -4002 },
    {   -900,   7039,  -3371, -4855 },
    {  -2983,   7211,   -363, -4835 },
    {   -814,   6503,   -104, -5106 },
    {  -2386,   6896,    809, -4919 },
    {    845,   4492,    352, -6621 },
    {  -1998,   7237,  -1646, -4231 },
    {  -3380,   6251,    471, -4577 },
    {  -1908,   7059,     84, -5726 },
    {   -340,   6346,   -803, -6265 },
    {  -2279,   5834,    -47, -4633 },
    {  -1532,   5286,  -1748, -1901 },
    {  -2757,   6188,   -453, -3415 },
    {  -1255,   6405,  -2043, -6357 },
    {    918,   5581,   -121, -5667 },
    {   1840,   5336,   -821, -5034 },
    {  -2475,   4992,  -1825, -3104 },
    {  -2413,   5606,  -1789, -4298 },
    {    132,   5128,  -2389, -4442 },
    {    223,   6400,  -2653, -4742 },
    {   -673,   5012,    680, -4582 },
    {  -1657,   6624,   -349, -3596 },
    {   -755,   6289,  -1860, -3978 },
    {   -572,   6894,  -1946, -5207 },
    {  -1141,   4756,  -2665, -5586 },
    {  -1073,   4269,   -431, -4030 },
    {    186,   5761,    916, -5868 },
    {  -1907,   4836,   1017, -5106 },
    {   -963,   3363,  -1248, -6348 },
    {  -3262,   4774,  -1818, -5858 },
    {    847,   3812,  -2538, -4302 },
    {  -1223,   5903,   1360, -5479 },
    {  -1094,   6923,  -1244, -2381 },
    {    267,   6276,   -709, -2846 },
    {   -157,   5840,   1124, -4266 },
    {    889,   3206,   -910, -5305 },
    {  -1736,   3344,    582, -4838 },
    {  -2357,   5676,  -2695, -6277 },
    {  -1916,   6901,   -986, -5397 },
    {  -3062,   6028,   -695, -5687 },
    {   1836,   3566,  -1357, -5226 },
    {  -2176,   4938,    646, -3872 },
    {  -2199,   3055,   -208, -6124 },
    {   -236,   3032,   -821, -5325 },
    {  -3989,   7277,   -565, -3899 },
    {   -595,   4362,     74, -5975 },
    {    684,   5874,   -841, -4424 },
    {  -2731,   6305,  -2389, -5465 },
    {  -5775,   1325,    -56, -2528 },
    {  -7029,   -534,  -1890, -3278 },
    {  -5798,    -15,  -2734, -2210 },
    {  -5504,  -1198,   -353, -3659 },
    {  -5079,    960,   -894, -4336 },
    {  -6073,    -36,   -133, -3014 },
    {  -5782,   -259,  -1025, -3986 },
    {  -6843,   1262,   -807, -1639 },
    {  -5263,   -918,  -3290,  -579 },
    {  -4840,    461,  -2158,  -533 },
    {  -6014,    -50,   -620,   504 },
    {  -5843,    241,  -1359,  -282 },
    {  -5898,    577,    769, -3271 },
    {  -6833,   -946,   -466, -3347 },
    {  -6026,   1459,   -512,  -729 },
    {  -7361,    747,   -388, -1110 },
    {  -6391,   2142,  -1160, -2513 },
    {  -6995,    304,    498, -2673 },
    {  -6757,    679,   -386,  -433 },
    {  -5222,   1688,  -1093, -1032 },
    {  -5019,    575,    184, -3627 },
    {  -4237,    628,  -3507, -1243 },
    {  -7479,   -456,  -1722, -1486 },
    {  -6464,    713,  -1273, -1153 },
    {  -6255,   1682,   -606, -3607 },
    {  -7033,   1497,    -71, -1955 },
    {  -6694,   1556,  -1721, -3214 },
    {  -6114,   -356,    813, -2575 },
    {  -5308,    632,  -1851, -1636 },
    {  -5742,   -911,  -1733,   383 },
    {  -6083,   -387,  -2313,  -879 },
    {  -6535,   -530,  -1505, -2083 },
    {  -4896,   1223,  -2750, -1816 },
    {  -6392,   -463,  -3247, -2093 },
    {  -5373,   1264,  -2706, -3042 },
    {  -3894,  -1390,  -1020,  -891 },
    {  -6179,   1168,  -1966, -1922 },
    {  -5162,   1668,  -1617, -1916 },
    {  -6453,    920,  -1169, -2432 },
    {  -6130,   2005,   -536, -1519 },
    {  -6552,    -98,   -518, -1938 },
    {  -7528,    355,  -1101, -1772 },
    {  -5745,    610,   -247, -1360 },
    {  -7003,    177,  -2064, -1958 },
    {  -6956,   -570,  -2220, -4225 },
    {  -7830,    791,  -1394, -2774 },
    {  -7634,    480,  -3171, -4224 },
    {  -7913,   1154,   -350, -2381 },
    {  -5063,   1704,  -1804, -2977 },
    {  -4887,   -524,  -2703,   188 },
    {  -5551,    406,  -1620, -3063 },
    {  -7109,   1342,    381, -3021 },
    {  -6846,    631,   -458, -3398 },
    {  -4606,   -605,     11, -3930 },
    {  -8134,   -225,  -1738, -2648 },
    {  -7043,    402,  -2734, -3059 },
    {  -7417,   1825,  -2545, -4389 },
    {  -6971,   -236,  -1031,  -665 },
    {  -5752,   2111,  -1632, -3808 },
    {  -7660,    -78,   -624, -3135 },
    {  -6358,    619,  -1951, -3911 },
    {  -8134,    408,  -1935, -3695 },
    {  -6335,   1911,  -2368, -4505 },
    {  -7116,   2163,   -344, -2753 },
    {   2357,   4488,   2220, -5682 },
    {   1385,   3206,   2300, -5305 },
    {   1419,   2557,   5203, -3516 },
    {    262,   4315,   3920, -1847 },
    {   3316,   3187,   1612, -5609 },
    {   1729,   2350,   1673, -6068 },
    {   1603,   6126,   1467, -2839 },
    {  -1339,   3316,   3691, -3530 },
    {   -563,   4618,   3180, -4548 },
    {    463,   4624,   3111, -5614 },
    {   1246,   5455,   3356, -5720 },
    {    480,   2149,   5422, -2893 },
    {   1768,   4827,    913, -5579 },
    {   -149,   5381,   4366, -3297 },
    {    985,   3672,   2644,   -92 },
    {   -258,   2911,   5817, -2213 },
    {   3428,   3289,   3351, -3541 },
    {   -666,   3295,   4727, -2869 },
    {     35,   6641,   4160, -4052 },
    {    623,   6787,   3156, -4560 },
    {   2654,   4360,   4676, -4632 },
    {   1386,   5246,   4834, -4497 },
    {   3488,   4574,   3856, -5946 },
    {    383,   4481,   4168, -4110 },
    {   1753,   3652,   4288, -3326 },
    {   1344,   4905,   2508, -4660 },
    {   1580,   4106,   3104, -2224 },
    {   2027,   5038,   1683, -1554 },
    {    446,   3699,   5872, -3013 },
    {   4637,   4087,   3578, -5018 },
    {   2629,   3560,   5331, -4900 },
    {   1527,   6674,   2523, -4131 },
    {  -1437,   2804,   2528, -4464 },
    {   -229,   3355,   2016, -5537 },
    {   3666,   3418,   4374, -4581 },
    {   1192,   3799,    923, -6596 },
    {   2040,   2956,    448, -5322 },
    {   2468,   5768,   4029, -5869 },
    {   3438,   6516,   3529, -6667 },
    {   2737,   5495,    680, -5535 },
    {   3896,   5727,   1801, -4958 },
    {   4988,   4957,   3592, -6518 },
    {   -542,   4416,   5794, -2787 },
    {   4136,   4354,   2064, -4696 },
    {   3067,   5936,   1207, -3396 },
    {   2789,   4966,   2405, -3854 },
    {   1731,   3270,   3251, -1063 },
    {   1767,   5537,   2084, -2349 },
    {    465,   3116,   4532,  -837 },
    {   1499,   2627,   4610, -2212 },
    {    122,   3095,   3642, -3552 },
    {   2542,   2866,   2705, -6402 },
    {   3134,   4323,    698, -4785 },
    {    731,   1859,   3112, -5242 },
    {   2553,   2980,   3241, -4846 },
    {   1329,   5310,   1607, -6624 },
    {   2468,   1858,   3476, -1034 },
    {   -172,   4996,   2000, -5562 },
    {   2621,   4220,   1574, -3386 },
    {   -333,   1832,   3362, -4117 },
    {   2169,   6762,   3065, -6225 },
    {   2844,   5528,   3223, -4765 },
    {    526,   5175,   1644, -4267 },
    {   2922,   4426,   2414, -2610 },
    {    452,   1399,  -4516, -2636 },
    {   2872,   1720,  -4667, -1435 },
    {   1279,    702,  -5424, -1984 },
    {   2187,    870,  -5021, -1341 },
    {    583,   -144,  -4628, -2464 },
    {      3,   2237,  -5284, -2827 },
    {    -19,   1005,  -5460, -1819 },
    {   2897,   2084,  -5885,  -515 },
    {   -400,   3370,  -5527, -2947 },
    {   1505,   2593,  -5518, -1802 },
    {   1341,   4534,  -5094, -1899 },
    {   3241,   3670,  -5493, -1252 },
    {  -1287,    921,  -5994, -1675 },
    {    627,    408,  -6652,  -364 },
    {   -260,   1127,  -4849, -3247 },
    {    371,   3400,  -5976, -2285 },
    {   1533,   1566,  -6373,  -610 },
    {   2462,   4274,  -6184, -1254 },
    {   1782,   3363,  -6222, -1381 },
    {    572,   4650,  -5673, -2754 },
    {   2674,   3414,  -4460, -2154 },
    {   3614,   3820,  -6883,  -398 },
    {   1136,     -1,  -5511, -1112 },
    {  -1773,   1137,  -5647, -2377 },
    {   -753,   2104,  -6085, -2565 },
    {   -204,   3025,  -4731, -1418 },
    {  -1486,   1438,  -4380,  -216 },
    {    302,    858,  -5786,  -264 },
    {   3486,   1495,  -5234,  -783 },
    {    888,   2327,  -3423, -3720 },
    {   -259,    772,  -6596, -1311 },
    {  -1197,   2073,  -5174, -1826 },
    {   1500,   3470,  -4462, -2645 },
    {   3072,   1960,  -3277, -2264 },
    {   1841,    952,  -4324, -2340 },
    {   1994,   2200,  -3940, -2923 },
    {  -1782,   1699,  -4667, -1075 },
    {  -1464,   2906,  -3468,  -375 },
    {    366,   2380,  -3747,  1467 },
    {   -545,   1645,  -4619,   376 },
    {   1724,   2350,  -2374, -3512 },
    {   3184,   2628,  -2996, -3275 },
    {    734,   2010,  -6239, -1479 },
    {    524,   3756,  -4496, -3263 },
    {   1492,   3570,  -3494, -3600 },
    {   -932,    618,  -5389, -2894 },
    {   -133,   2161,  -4083, -3267 },
    {    786,    774,  -3279, -3731 },
    {   1078,    803,  -3843, -3007 },
    {   -332,   3405,  -3347,    40 },
    {    -17,      6,  -4005, -3690 },
    {   -189,   4372,  -4488, -2561 },
    {   -450,   3846,  -3790, -1370 },
    {    362,   2212,  -5272,   -15 },
    {  -1529,    791,  -6802, -2296 },
    {   2145,   4241,  -4474,   376 },
    {   1813,   2426,  -2932, -2726 },
    {   -542,   4557,  -3140, -1080 },
    {   1192,   3784,  -4371,   -20 },
    {   2784,   5188,  -6399, -1394 },
    {    431,   4561,  -3673, -1398 },
    {   1382,   3096,  -4083,  1253 },
    {   1209,   4224,  -2930,  1500 },
    {   2798,   2684,  -6676,  -606 },
    {  -2396,   1510,  -5381, -2713 },
    {  -2625,   2542,  -4032, -2880 },
    {  -1231,   3967,  -4098, -2886 },
    {  -1393,   2374,  -3862, -4525 },
    {  -2495,   1665,  -1637, -5445 },
    {  -3854,   1759,  -1750, -4944 },
    {  -2373,   1668,  -2856, -6251 },
    {  -2668,   1981,   -886, -4557 },
    {  -2927,   4427,  -3451, -6172 },
    {  -1925,   2596,  -4696, -2527 },
    {  -3202,   2847,  -3928, -5896 },
    {  -3332,   1665,  -5025, -3412 },
    {  -3212,   3115,  -4155, -4062 },
    {  -1013,   3205,  -5133, -3751 },
    {  -2022,   4595,  -3947, -5611 },
    {  -3556,   1755,  -3715, -2300 },
    {  -1784,   4114,  -2723, -1773 },
    {  -3586,   4081,  -2733, -4942 },
    {  -1608,   3685,  -4154, -4573 },
    {  -3368,   4042,  -4452, -6227 },
    {  -1407,   3881,  -5729, -3719 },
    {  -2751,   3281,  -5077, -4999 },
    {  -3791,   2410,  -4906, -5288 },
    {   -730,   2303,  -4217, -3755 },
    {  -1812,   2311,  -5492, -3709 },
    {   -610,   4336,  -3915, -3783 },
    {  -2841,   4337,  -4278, -4430 },
    {  -1662,   4666,  -4661, -3964 },
    {   -589,   5209,  -4923, -3682 },
    {  -4155,   2234,  -4076, -4218 },
    {  -3951,   2770,  -2665, -2805 },
    {  -2302,   3228,  -3717, -1908 },
    {  -3129,   4373,  -2264, -2851 },
    {   -447,   1363,  -3578, -4323 },
    {  -2648,   4237,  -3159, -3071 },
    {  -4072,   3241,  -3541, -4605 },
    {  -4507,   3458,  -2339, -3838 },
    {  -1646,    997,  -4926, -3970 },
    {  -3025,   1614,  -3940, -1242 },
    {  -1337,   1756,  -3163, -5529 },
    {  -3203,   1865,  -3282, -4354 },
    {  -1646,   2118,  -2203, -6018 },
    {    174,   1871,  -2707, -4639 },
    {  -2607,   1485,  -4778, -4750 },
    {  -2199,   3991,  -3134, -4879 },
    {  -2962,   3323,  -2816, -2419 },
    {  -5286,   2495,  -4548, -5395 },
    {  -2810,   3710,  -2274, -4211 },
    {   -330,   3006,  -2993, -4678 },
    {  -1187,   2411,  -2743, -5196 },
    {   -664,   4033,  -3101, -5641 },
    {  -1458,   3602,  -2816, -5371 },
    {  -4116,   4923,  -3321, -5630 },
    {  -4165,   2528,  -2592, -4798 },
    {  -2759,   3080,  -2333, -5719 },
    {  -5157,   3011,  -5526, -6348 },
    {  -3095,   2126,  -5881, -4234 },
    {  -4377,   3849,  -3600, -6099 },
    {  -1994,   4947,  -5235, -4753 },
    {  -1067,    600,  -3258, -5133 },
    {  -4992,   3302,  -2208, -5051 },
    {  -3377,   2981,  -1655, -4815 },
    {  -3325,   2446,  -1787, -6116 },
    {  -2341,   2737,  -3240, -6347 },
    {  -2258,  -3732,   3710, -1235 },
    {  -1558,  -3849,   2694, -3012 },
    {   -599,  -4837,   3050, -2951 },
    {  -2246,  -5433,   2798, -1910 },
    {  -2255,  -4989,   3260,   270 },
    {  -3026,  -5353,   2693, -1036 },
    {  -1151,  -6097,   1097, -3782 },
    {  -3391,  -6012,   2130, -1303 },
    {  -2850,  -4422,   3375,  -480 },
    {  -1138,  -3779,   1491, -4162 },
    {   -551,  -3892,   3787, -2082 },
    {  -3221,  -3676,   3144, -1202 },
    {  -3023,  -5196,   2650,   605 },
    {  -1756,  -5729,   2646,   321 },
    {  -2693,  -4409,    494, -4797 },
    {  -1913,  -4573,   3372, -1730 },
    {  -1277,  -3604,   4061,  -993 },
    {   -420,  -4993,   1351, -4796 },
    {  -3052,  -5333,   1435, -1242 },
    {   -602,  -5034,   3869, -1141 },
    {  -2436,  -4680,   1665, -3019 },
    {  -2657,  -3658,   1459, -3391 },
    {  -1220,  -6246,   2749,  -525 },
    {  -3838,  -4844,   2265, -1735 },
    {  -1247,  -5679,   3356, -1417 },
    {   -917,  -5448,   3342,   105 },
    {  -1756,  -6839,   2276, -2350 },
    {   -412,  -5206,   1764, -3539 },
    {  -1439,  -6915,   1442, -3750 },
    {  -1381,  -4439,   3863,  -282 },
    {  -3482,  -4953,   2726,  -336 },
    {  -1376,  -5931,   1714, -1987 },
    {  -1716,  -4405,   2608,   105 },
    {  -1590,  -5191,   2652, -2704 },
    {  -2149,  -6442,   2453, -1263 },
    {  -3426,  -3832,   2334, -1829 },
    {  -2747,  -5948,   2362,  -173 },
    {  -2435,  -3267,   2966, -1710 },
    {  -3979,  -4282,   2705,  -775 },
    {   -356,  -4238,   2544, -4343 },
    {  -1363,  -6471,   2817, -1836 },
    {  -2878,  -5117,    218, -3149 },
    {  -3539,  -5196,   1710, -2356 },
    {  -2888,  -4537,   2746, -1701 },
    {  -1870,  -4439,   1496, -4121 },
    {  -1486,  -3388,   3349, -2145 },
    {  -3333,  -4138,   1467, -2876 },
    {   -345,  -5340,   1012, -1190 },
    {  -1672,  -4992,   2289, -1029 },
    {  -2146,  -5528,   3038,  -635 },
    {   -316,  -3656,   3426, -3152 },
    {  -2695,  -5812,   2336, -2050 },
    {  -2067,  -6052,    737, -3258 },
    {  -2664,  -4205,   -350, -1266 },
    {   -617,  -5406,     80, -4853 },
    {  -2418,  -3825,   1853, -1326 },
    {  -1961,  -4339,    583, -4315 },
    {  -1495,  -5141,   -133, -5205 },
    {  -3208,  -6440,   1691, -2069 },
    {  -2632,  -3633,   2325, -2761 },
    {  -2624,  -5670,   1252, -3676 },
    {  -3687,  -5608,    687, -2833 },
    {  -3320,  -5707,     16, -3877 },
    {  -2738,  -6112,     84, -5135 },
    {   2277,  -5661,   3076,   843 },
    {   1555,  -5769,   2821, -5236 },
    {    536,  -6381,    603, -4910 },
    {    734,  -4609,   3314, -4092 },
    {   1836,  -4547,   3267, -4322 },
    {    -13,  -5976,   3752, -1607 },
    {   1423,  -6318,   2336,   398 },
    {    365,  -7779,   1498,  -534 },
    {   2104,  -8366,   2946, -1345 },
    {    143,  -5545,   1898, -3756 },
    {    655,  -6852,   1430,   148 },
    {      4,  -6653,   2397,   -59 },
    {   2346,  -5996,   4562,  -934 },
    {   1229,  -7104,   2963,  -598 },
    {   -528,  -7048,   2887, -1790 },
    {   1451,  -6857,   3900, -1637 },
    {    554,  -6018,   3336,     9 },
    {   3278,  -5758,   4034,   129 },
    {   3541,  -7145,   4905, -1575 },
    {   2339,  -6907,   3464,  -301 },
    {   2775,  -7301,   1667, -3894 },
    {    539,  -7887,    991, -4156 },
    {   2115,  -7421,   3131, -3075 },
    {   2803,  -8546,   2564, -5836 },
    {   2869,  -5833,   1620, -4561 },
    {   2591,  -7281,   3215, -4719 },
    {  -1228,  -8477,    706, -4782 },
    {   1967,  -5243,   4813, -1940 },
    {    701,  -7010,   2273, -3893 },
    {    915,  -8470,   1918, -5620 },
    {    -94,  -6715,    156, -3873 },
    {   1074,  -5607,   4389, -1017 },
    {   2739,  -6551,   1227, -3521 },
    {    725,  -7835,   2701, -1291 },
    {   -493,  -7475,   2263, -1075 },
    {   -412,  -6508,   2984,  -744 },
    {    665,  -5451,   3725, -2692 },
    {   1499,  -8129,   3564, -2072 },
    {   2870,  -6333,   4487, -2108 },
    {    706,  -5007,   3911,  -152 },
    {   -482,  -8660,   1483, -2900 },
    {   2481,  -6596,   2518, -1715 },
    {   1403,  -6414,   1398, -5387 },
    {    652,  -6267,    583, -5942 },
    {    694,  -7540,    646, -6272 },
    {   2275,  -7614,    256, -5015 },
    {   1416,  -9727,   1900, -3153 },
    {   2760,  -6433,   3875, -3771 },
    {   2325, -11196,   2182, -5155 },
    {   1223, -11061,   1377, -5097 },
    {    108, -10603,    307, -4952 },
    {   -118,  -8268,   1650, -1572 },
    {   1839,  -7943,   1755,  -612 },
    {   2501,  -9056,    981, -2969 },
    {   2902,  -8476,   1491, -5780 },
    {   1995, -11175,   1585, -3643 },
    {    696,  -8212,    828, -2474 },
    {   1526,  -8649,   1380, -1210 },
    {    461,  -7253,   3222, -2229 },
    {   2966,  -8641,   4121, -3271 },
    {    833,  -6039,   2361, -1086 },
    {   3565,  -7312,   1980, -5427 },
    {   2850,  -8671,   3760, -1846 },
    {   2643,  -7281,   2163,  -173 },
    {   3463,  -3706,  -3132,  -923 },
    {   1315,  -3825,  -3443,     2 },
    {   2594,  -4083,  -3815,   670 },
    {   1826,  -4291,  -2741,  -155 },
    {    868,  -3749,  -4175,  -298 },
    {   2008,  -4237,  -3897,  -517 },
    {   1242,  -3493,  -4335, -1335 },
    {    -88,  -4142,  -3390, -1529 },
    {   2176,  -3488,  -3822,  -975 },
    {   1706,  -5188,  -3415,  -637 },
    {   2717,  -6159,  -2333,  -882 },
    {   1276,  -3978,  -4361,   537 },
    {   2471,  -5556,  -2866,  -208 },
    {    799,  -4673,  -4086,    56 },
    {   1901,  -4786,  -3533,   270 },
    {   3036,  -3902,  -3606,  -333 },
    {   2249,  -3317,  -4319,  -144 },
    {   2594,  -4207,  -2105, -2930 },
    {   4008,  -4774,  -2626,  -902 },
    {   1038,  -3659,  -3496, -2454 },
    {   2725,  -3597,  -3298, -1535 },
    {   1662,  -5803,  -2813,   175 },
    {    705,  -3757,  -3441, -1484 },
    {   1860,  -5987,  -2821,  -886 },
    {   3786,  -4918,  -2199, -1929 },
    {   3683,  -4235,  -2547, -1287 },
    {   2531,  -4896,  -2956, -1593 },
    {   1005,  -5585,  -3324,  -180 },
    {   1625,  -5229,  -1756, -3642 },
    {   1494,  -5041,  -2989, -2685 },
    {   2718,  -4655,  -3224,  -867 },
    {   2374,  -6640,  -1745, -2975 },
    {   2133,  -6436,  -2477, -1499 },
    {   1833,  -4418,  -3523, -1512 },
    {   1128,  -4910,  -2658, -1106 },
    {    689,  -4777,  -2831, -2085 },
    {   3593,  -5280,  -2627,  -315 },
    {   3264,  -3771,  -2673, -1861 },
    {   3202,  -5602,  -2409,   402 },
    {    552,  -4618,  -2221, -3002 },
    {   3095,  -5356,  -2666, -1083 },
    {   3401,  -4609,  -3146,    45 },
    {   3051,  -4662,  -2192, -2232 },
    {   2798,  -5552,  -2462, -1941 },
    {   2354,  -5815,  -2223, -2619 },
    {    192,  -3708,  -2807, -2658 },
    {   1886,  -4226,  -1862, -3529 },
    {   2526,  -3976,  -2819, -2332 },
    {   1577,  -3870,  -2711, -2806 },
    {   1288,  -5588,  -3382, -1403 },
    {   2711,  -5399,  -1564, -3253 },
    {   1459,  -5492,  -2222,  -322 },
    {   2823,  -5091,  -2886,   776 },
    {   3559,  -5821,  -2109, -1360 },
    {   1587,  -6331,  -2760, -1909 },
    {   2139,  -5213,  -2874, -2120 },
    {   1318,  -4337,  -3695, -2098 },
    {    821,  -4471,  -1849,  -565 },
    {   3329,  -4782,  -1725,   -89 },
    {    582,  -4914,  -4105, -1119 },
    {    417,  -4144,  -4072, -2529 },
    {   -199,  -3803,  -2765, -4042 },
    {   2731,  -4283,  -2143,     1 },
    {   2911,  -6187,  -1951, -2116 },
    {   1573,  -6094,   -493, -2838 },
    {   2081,  -6927,   -864, -3211 },
    {   1058,  -7826,     79,  -364 },
    {   3147,  -5570,   -684,  -978 },
    {   3572,  -5856,   1060,  1824 },
    {   1143,  -6702,  -1478,   338 },
    {   2341,  -7220,    -88,   260 },
    {   3639,  -6861,    668,   815 },
    {   2227,  -6268,  -1706,   446 },
    {   3390,  -6082,   -353,  1302 },
    {   1123,  -7556,  -1237,  -430 },
    {   1729,  -7742,    729,  -218 },
    {   1457,  -6774,    587,   579 },
    {    505,  -6919,   -569,   371 },
    {   1106,  -7245,     78,   158 },
    {   2755,  -6745,  -1122,   338 },
    {   3069,  -6040,  -1415,   986 },
    {   2174,  -7064,  -1430,  -283 },
    {   1390,  -8626,   -446, -3031 },
    {   3534,  -6890,   -431,   547 },
    {   2267,  -9618,    475, -2994 },
    {   3672,  -7673,     75,  -115 },
    {   2131,  -7560,  -1206,  -750 },
    {   2972,  -7477,   -685,  -262 },
    {   1604,  -6637,   -672,   699 },
    {   1666,  -7577,   -577,  -240 },
    {   1591,  -6554,  -2158,   -94 },
    {   2348,  -6286,   -353,  1123 },
    {   2017,  -8810,   -412, -1805 },
    {   2892,  -6713,  -1765,  -554 },
    {   2500,  -6828,  -1995, -1197 },
    {   3877,  -6639,   -224, -1655 },
    {   2392,  -7872,    -91,  -333 },
    {   3562,  -7370,   -532, -2836 },
    {   2552,  -7614,    164, -1805 },
    {    990,  -6104,    218,   438 },
    {    910,  -7861,    312, -1195 },
    {   1472,  -6327,    372,  -640 },
    {   1576,  -7143,  -1983,  -843 },
    {    422,  -7625,   -457,  -278 },
    {   1797,  -8532,    405, -1011 },
    {   1088,  -7396,   -238, -2277 },
    {   3209,  -6753,  -1431, -2072 },
    {   2617,  -6839,    100, -2573 },
    {   2575,  -8573,   -387, -3188 },
    {   3618,  -6971,  -1190,  -321 },
    {   2205,  -7361,  -1695, -2008 },
    {   2985,  -6297,   1464,  1179 },
    {   2804,  -7310,   1053,   338 },
    {   1362,  -6074,  -1163,  -840 },
    {   3336,  -6325,  -1794,    21 },
    {   2836,  -8109,    818,  -329 },
    {   2791,  -5879,    560,  1546 },
    {   2392,  -6064,    135,   100 },
    {   1838,  -6194,    596,  1085 },
    {   1926,  -7515,   -414, -4901 },
    {   3225,  -7298,  -1202, -1189 },
    {   3960,  -7558,   -659,  -719 },
    {   3442,  -6647,  -1692, -1095 },
    {   3381,  -6441,    262,  -886 },
    {   1431,  -8150,  -1186, -1406 },
    {    340,  -8498,   -150,  -899 },
    {   3004,  -8149,   -260,  -953 },
    {   2749,  -6611,    563,   873 },
    {  -6647,  -1325,  -4517, -4691 },
    {  -6005,  -1657,  -4089, -3797 },
    {  -3157,    588,  -5213, -3068 },
    {  -3311,  -1425,  -6329, -3726 },
    {  -5866,   -819,  -3857, -2744 },
    {  -5001,  -1799,  -1075, -4621 },
    {  -5330,  -2650,  -2672, -4664 },
    {  -4930,   -539,  -2363, -4010 },
    {  -2984,     10,  -3863, -5749 },
    {  -1055,  -2106,  -3713, -4267 },
    {  -5476,   -502,  -4279, -6504 },
    {  -5231,  -1543,  -5018, -6425 },
    {  -5134,   -363,  -3165, -5109 },
    {  -3953,   -771,  -4107, -6393 },
    {  -2159,   -563,  -3652, -5342 },
    {  -3888,  -2321,   -919, -5057 },
    {  -1236,   -597,  -4235, -4193 },
    {  -4053,    675,  -3083, -6174 },
    {  -2793,  -1089,  -5396, -3460 },
    {  -3000,    -44,  -2209, -6575 },
    {  -3336,  -1531,  -4313, -5160 },
    {  -2127,    128,  -4851, -3692 },
    {  -3321,    136,  -2067, -5660 },
    {  -5215,   1404,  -4374, -4356 },
    {  -2747,    400,  -6340, -3691 },
    {  -3926,   -599,  -5361, -5006 },
    {  -2875,  -2592,  -5143, -4092 },
    {  -4991,  -1958,  -5322, -4891 },
    {  -4965,  -1318,  -6652, -5333 },
    {  -4920,  -1691,  -3388, -5561 },
    {  -3644,  -3354,  -2688, -5982 },
    {  -5076,   -919,  -4563, -2984 },
    {  -6114,    250,  -3884, -3915 },
    {  -4014,    744,  -3973, -1924 },
    {  -5543,  -1041,  -5557, -3847 },
    {  -4711,  -1352,  -5649, -2603 },
    {  -3362,    775,  -5305, -4879 },
    {  -5001,    107,  -3554, -2888 },
    {  -6258,  -1651,  -6356, -6566 },
    {  -4529,    407,  -5003, -3865 },
    {  -5154,    550,  -5278, -5465 },
    {  -4195,   -467,  -1894, -3129 },
    {  -5022,   1127,  -3349, -3314 },
    {  -6075,   1250,  -4313, -5641 },
    {  -2677,  -2283,  -2312, -5903 },
    {  -4113,    193,  -1195, -4833 },
    {  -3940,  -1048,  -1389, -5079 },
    {  -3703,    917,  -4043, -4451 },
    {  -3366,  -4231,  -1534, -5488 },
    {  -3326,  -3583,  -2091, -4903 },
    {  -5144,   1254,  -2532, -4949 },
    {  -5982,   -870,  -2545, -4555 },
    {  -3925,   -157,  -5367, -2281 },
    {  -6419,   -746,  -5668, -4371 },
    {  -5787,    518,  -7096, -5805 },
    {  -4258,    954,  -6453, -4321 },
    {  -4771,   -695,  -4158, -1639 },
    {  -7078,   -760,  -5195, -5877 },
    {  -7348,     83,  -4101, -4586 },
    {  -2430,    184,  -2874, -1679 },
    {  -2284,  -3943,  -2924, -5034 },
    {  -1804,  -1785,  -3002, -4710 },
    {  -4399,  -2772,  -1815, -4637 },
    {  -6340,  -2626,  -2824, -5191 },
    {  -4998,  -5168,  -3480,  1905 },
    {  -3958,  -5492,  -1599,  1579 },
    {  -2471,  -3755,   -276,  3182 },
    {  -3033,  -5779,  -1063,  1554 },
    {  -2936,  -4829,  -1290,  2386 },
    {  -1835,  -5073,  -3051,  1299 },
    {  -1724,  -3771,  -3935,  2324 },
    {  -5070,  -2550,  -3692,   768 },
    {  -4326,  -5333,   -297,  1878 },
    {  -3472,  -5619,  -3094,   992 },
    {  -3027,  -4384,  -3038,  2265 },
    {  -3201,  -5332,     67,  2200 },
    {  -1681,  -4373,  -1947,  2461 },
    {  -3221,  -3329,  -4238,  2564 },
    {  -1262,  -2968,  -2915,  3227 },
    {  -3419,  -1878,  -3373,  2110 },
    {  -2244,  -5583,  -2012,  1288 },
    {  -1971,  -5266,   -990,  1812 },
    {  -2975,  -2778,   -452,  4063 },
    {  -2198,  -1165,  -3298,  2965 },
    {  -4782,  -4894,  -4767,   664 },
    {  -6002,  -3950,  -2806,  2025 },
    {  -3142,  -3162,  -2859,  3295 },
    {  -3262,  -3340,  -4123,  1596 },
    {  -4014,  -3918,  -1955,  3361 },
    {  -1700,  -3463,  -1346,  3449 },
    {  -4245,  -4445,  -4743,  1644 },
    {  -4180,  -3969,   -401,  3281 },
    {  -2782,  -5240,  -4117,  1156 },
    {  -5744,  -4040,  -1439,  3470 },
    {  -5063,  -4663,   -323,  3172 },
    {  -4531,  -3319,   -844,  3988 },
    {  -6226,  -5125,  -2064,  2976 },
    {  -3115,  -3267,  -1531,  3898 },
    {  -4628,  -4421,  -2864,  2808 },
    {  -4559,  -2989,  -3442,  2024 },
    {  -1775,  -4487,   -656,  2477 },
    {  -2664,  -1865,  -1884,  4081 },
    {  -1828,  -2575,  -3894,  3378 },
    {  -6441,  -3677,  -2025,  1677 },
    {  -4141,  -2156,  -1191,  3474 },
    {  -4802,  -1623,  -1727,  2160 },
    {  -5474,  -2745,  -1475,  2498 },
    {  -3664,  -1056,  -1975,  2491 },
    {  -4672,  -3062,  -2235,  2933 },
    {  -4205,  -5960,  -2849,  1517 },
    {  -4995,  -5708,  -1739,  1805 },
    {  -4892,  -6080,  -4793,   872 },
    {  -4270,  -4172,  -4263,  2185 },
    {  -4687,  -1470,  -2905,  1023 },
    {  -6446,  -5017,  -3919,  1000 },
    {  -6046,  -5538,  -3943,  2006 },
    {  -6028,  -3750,  -3953,   771 },
    {  -5959,  -4582,  -5024,   824 },
    {  -5818,  -2576,  -2249,  1326 },
    {  -5659,  -5345,  -1119,  2500 },
    {  -3346,  -4155,    606,  2749 },
    {  -5680,  -4827,  -2501,  1838 },
    {  -6193,  -2543,  -1295,   840 },
    {  -6871,  -4925,  -3512,  1801 },
    {  -5605,  -1788,  -1895,   779 },
    {  -3922,  -5712,  -4644,   510 },
    {  -4745,  -3869,  -4533,    99 },
    {  -2984,  -4907,   -399,  1497 },
    {   1847,   -478,   3061, -5812 },
    {   4450,  -1116,   3609, -6570 },
    {   3139,     99,   3007, -5532 },
    {   2590,  -3782,   3138, -4770 },
    {   1881,   1204,   5778, -3404 },
    {   3631,   2060,   5566, -5038 },
    {   3461,   1961,   5167, -3800 },
    {   2947,    273,   4536, -4389 },
    {   4453,  -1730,   5788, -4370 },
    {   4032,   1805,   2666, -4534 },
    {   3487,   -944,   2313, -6028 },
    {   1313,     34,   4210, -4067 },
    {   5632,  -1502,   5825, -5855 },
    {   7736,   -547,   4879, -5476 },
    {   4906,  -1512,   4760, -5760 },
    {   3843,    447,   1091, -4958 },
    {   2982,  -1135,   5442, -4386 },
    {   3579,    271,   3031, -6770 },
    {   3932,   -211,   4688, -5507 },
    {   4411,   1720,   2387, -5584 },
    {   5379,   -479,   4575, -6280 },
    {   3613,   -362,   2012, -4885 },
    {   3744,  -2013,   4493, -5073 },
    {   5693,    109,   4379, -3362 },
    {   5475,   -621,   5317, -3985 },
    {   6411,   -673,   5708, -4752 },
    {   4933,   -796,   7262, -4290 },
    {   2804,    444,   6276, -3655 },
    {   4120,   -517,   6078, -4531 },
    {   5119,    841,   3486, -3910 },
    {   4738,   1539,   3525, -2970 },
    {   5086,    370,   5895, -5640 },
    {   4235,   2716,   4589, -5044 },
    {   3691,    682,   6199, -4700 },
    {   6111,   -570,   6271, -6528 },
    {   2611,   1277,   3756, -4802 },
    {   4395,    970,   3807, -5879 },
    {   5225,   2299,   3242, -4333 },
    {   5144,   1778,   4946, -5545 },
    {   2989,  -3016,   3247, -5495 },
    {   2983,    920,   2071, -6059 },
    {   5270,   -903,   4434, -2350 },
    {   6415,   -585,   3970, -3554 },
    {   3866,   -197,   5216, -2884 },
    {   3767,  -1298,   6702, -3315 },
    {   6299,   2620,   5284, -6824 },
    {   6654,    646,   3653, -4927 },
    {   4770,   3047,   5160, -6287 },
    {   5364,    434,   2919, -5207 },
    {   2998,   1344,   4801, -2456 },
    {   3896,   1013,   3773, -1864 },
    {   2115,    655,   2999, -6344 },
    {   5170,   -981,   2849, -4464 },
    {   2735,  -2159,   2717, -5776 },
    {   2430,  -1952,   4392, -4559 },
    {   6143,  -1180,   3659, -4746 },
    {   4978,  -1483,   1726, -4875 },
    {   3486,  -2383,   3306, -4301 },
    {   1434,  -1372,   4171, -4770 },
    {   3354,  -2627,   1525, -5093 },
    {   6790,   2386,   3995, -5909 },
    {   1475,  -2674,   3451, -4204 },
    {   1999,  -3494,   3693, -5556 },
    {   4764,  -2848,   2856, -5589 },
    {  -3677,   5131,   2827, -2934 },
    {  -2844,   7078,   2852, -3580 },
    {  -3902,   6434,   4118, -1911 },
    {  -1769,   7530,   3492, -3541 },
    {  -1937,   5679,   -447, -1127 },
    {  -2456,   4680,   4196, -2407 },
    {  -2778,   8241,   1698, -4288 },
    {  -2876,   6104,   5182, -2387 },
    {  -2802,   7341,   4463, -2938 },
    {  -1025,   6267,   4752, -3201 },
    {  -2349,   5413,   2041, -3794 },
    {  -2252,   8225,   2856, -4269 },
    {  -1465,   4967,   4976, -2500 },
    {   -636,   7565,   3517, -4233 },
    {  -1905,   5618,   3904, -2942 },
    {   -302,   6816,   3343, -3316 },
    {  -2210,   4156,   2817, -3511 },
    {   -717,   6568,   1863, -2951 },
    {  -3873,   5682,   2164,  -575 },
    {  -2878,   5835,    440, -2597 },
    {  -3228,   7701,   2610, -2514 },
    {  -3608,   8888,   3377, -2468 },
    {  -2582,   9717,   2519, -3126 },
    {  -5238,   6202,   2866, -2831 },
    {  -3428,   7370,   3056,  -335 },
    {  -1681,   8836,   1210, -2010 },
    {  -3276,   6724,   1156, -3930 },
    {   -894,   8149,    827, -1258 },
    {  -2965,   8631,   2549, -1320 },
    {  -3961,   6902,   3581,    55 },
    {  -1894,   7745,   1750,  -841 },
    {   -821,   6844,    850,  -676 },
    {   -608,   6948,     -4, -1376 },
    {    615,   6524,   1089, -1147 },
    {  -2972,   5668,   1091,  -489 },
    {   -157,   4649,   2904,  -413 },
    {    673,   5121,   1498,   -66 },
    {   -390,   5902,   1611,  -245 },
    {  -2349,   5478,   4772, -1320 },
    {     88,   6798,   1972, -1859 },
    {  -1213,   5120,   2991,   200 },
    {  -2347,   6040,   2839,   376 },
    {   -578,   5976,   3364, -1796 },
    {  -1391,   5872,   3002,  -965 },
    {   -564,   4496,   3946, -1186 },
    {  -2299,   6386,   3135, -2176 },
    {  -2131,   5641,   2011,  1223 },
    {   -772,   5807,   1124,   895 },
    {  -2837,   6758,   2297,  -740 },
    {  -3091,   6298,   1415, -2126 },
    {  -4197,   6036,   1843, -3022 },
    {    -41,   6459,     92,   344 },
    {  -2241,   6860,   2095, -4396 },
    {  -1931,   7088,   2117, -2135 },
    {  -2375,   4422,   1688, -3169 },
    {  -1742,   6674,   1538,  -119 },
    {  -4818,   7749,   4192, -1577 },
    {  -2004,   5672,    193,  -430 },
    {  -3825,   6042,   2128, -1898 },
    {  -1108,   8033,   2119, -3013 },
    {  -2370,   5453,   1721,   266 },
    {  -1570,   7134,    614, -2638 },
    {  -1519,   8752,   3503, -4330 },
    {  -2050,   3845,   2907, -1126 },
    {   5085,   4412,   -335, -1923 },
    {   3618,   1423,   -613, -4012 },
    {   4481,   3729,    589, -4631 },
    {   4270,   3216,  -1763, -3168 },
    {   4241,   1796,  -1701, -2796 },
    {   4787,   2338,   -487, -3639 },
    {   2915,   3429,   -621, -4753 },
    {   5175,   1660,  -1265, -3223 },
    {   4280,   4057,   -684, -4079 },
    {   4980,   4419,  -1455, -2719 },
    {   5436,   2464,    387, -4197 },
    {   4507,   4018,   1121, -3314 },
    {   6020,   2401,   -413, -3201 },
    {   4200,   3789,   -333, -2813 },
    {   5229,   2493,  -1194, -1878 },
    {   5851,   2695,   -492, -2292 },
    {   5743,   3288,   -697, -1221 },
    {   5692,   2612,    979, -2227 },
    {   5085,   2067,   1046, -1214 },
    {   3163,   2240,  -2098, -3435 },
    {   5228,   1898,    145, -2397 },
    {   5860,   3976,   -418, -2872 },
    {   6008,   3399,   1027, -3506 },
    {   4126,   2035,   1865,  -893 },
    {   5375,   3596,    511, -2362 },
    {   1937,   1493,   -852,  -122 },
    {   3473,   4849,    547, -2603 },
    {   4631,   2977,   1141, -1768 },
    {   6149,   3050,    -71, -1886 },
    {   4069,   4353,   -289, -1429 },
    {   2884,   1225,  -1388,   365 },
    {   5485,   2518,   -235,  -571 },
    {   1216,   4375,   1443,   398 },
    {   4988,   3106,    107, -1435 },
    {   4511,   2801,    307,  -444 },
    {   3235,   4386,    327,  -676 },
    {   2055,   3708,   1657,  -305 },
    {   5839,   2374,    290, -1385 },
    {   5110,   3305,   1936, -4206 },
    {   6416,   2920,    338, -2736 },
    {   3350,   2824,  -1269, -3881 },
    {   4840,   1815,    464,   186 },
    {   2399,   3332,    238,  1238 },
    {   3516,   1363,   1582,   688 },
    {   3582,   1874,    154, -4770 },
    {   3261,   2878,    886,   283 },
    {   3877,   2658,   -327,   884 },
    {   4151,   3436,   2173, -2923 },
    {   3592,   3674,   1281, -1295 },
    {   4561,   3730,  -1114, -1747 },
    {   4595,   3625,   -558,  -575 },
    {   2577,   2348,   2267,   120 },
    {   5242,   3299,     32, -3412 },
    {   4264,   3637,    709, -2320 },
    {   6556,   3570,   -838, -2472 },
    {   5745,   4014,   -940, -1973 },
    {   5629,   4475,    477, -3328 },
    {   5269,   3199,   1682, -3085 },
    {   4432,   2416,   1145, -3299 },
    {   4465,   2505,   2162, -2186 },
    {   4643,   4941,    -88, -2885 },
    {   4568,   5231,    552, -3915 },
    {   5667,   3075,  -1406, -2963 },
    {   5418,   5259,   -771, -2818 },
    {   -256,  -7875,    511,  -471 },
    {  -1813,  -7971,   -424,  -396 },
    {   -306,  -7006,    862,   282 },
    {  -2306,  -6422,  -1440,   508 },
    {   -245,  -6787,    375,  -100 },
    {  -1309,  -6065,    -20,   779 },
    {  -1656,  -6047,   -641,  1307 },
    {  -1496,  -6522,    964,   726 },
    {  -2291,  -6588,   -202,   795 },
    {   -762,  -7522,   1454,  -558 },
    {  -2270,  -7004,   -834,  -580 },
    {  -1139,  -7078,    259,   362 },
    {  -2535,  -7568,  -1040,    49 },
    {  -3786,  -7280,    934,  -476 },
    {  -3336,  -6368,    606,  1056 },
    {  -3602,  -6924,     52,   714 },
    {  -2278,  -6550,   1674,   204 },
    {  -2855,  -5765,    930,  1530 },
    {  -2889,  -7325,   -215,   305 },
    {  -2749,  -6080,   -237,  1452 },
    {   -985,  -6667,   1577,   400 },
    {  -2036,  -6083,    380,  1267 },
    {  -2077,  -7460,    380,   -30 },
    {  -1775,  -7175,   1540,  -386 },
    {  -3065,  -6927,    989,   168 },
    {  -2836,  -7602,    117, -3392 },
    {  -1058,  -6396,    593, -3078 },
    {   -844,  -6062,    999,  -236 },
    {  -3261,  -6951,   1491,  -720 },
    {  -2186,  -8484,     75, -1287 },
    {  -2882,  -7756,    456,  -510 },
    {  -1800,  -6879,    960, -1183 },
    {  -2554,  -7241,   1614, -1474 },
    {  -2608,  -5305,    392,   851 },
    {  -2973,  -6562,   -859,   858 },
    {  -2640,  -5989,   1031,  -416 },
    {   -977,  -8366,    705, -1434 },
    {  -1213,  -7409,    -77, -1390 },
    {  -1335,  -6657,   2125,  -123 },
    {  -2544,  -6862,   1852,  -737 },
    {  -3235,  -6422,   1752,  -103 },
    {  -1300,  -7557,    939,  -348 },
    {  -3476,  -7579,    202,  -109 },
    {  -2482,  -6572,    753,   619 },
    {  -2554,  -8136,   -648,  -429 },
    {  -1012,  -7870,     -3,  -421 },
    {  -3604,  -6247,     32, -3102 },
    {  -1486,  -7271,   2013, -1021 },
    {   -578,  -6799,   -523,   405 },
    {  -2841,  -5948,   1644,   911 },
    {  -2411,  -7473,   1084,  -484 },
    {  -2238,  -6033,    294, -1059 },
    {  -3459,  -6470,   -201,  -790 },
    {  -2027,  -6009,   1833,   805 },
    {  -1433,  -8047,   1531, -1754 },
    {  -3258,  -7884,    763, -1422 },
    {  -1544,  -6928,   -729,   478 },
    {  -2314,  -8415,     74, -3757 },
    {  -3201,  -5684,     95, -2214 },
    {  -2423,  -8694,    725, -3631 },
    {  -3545,  -7071,   1162, -1798 },
    {   -294,  -9662,    403, -2274 },
    {  -2290,  -5460,   1196,   402 },
    {  -1603,  -6713,    903, -2363 },
    {   4121,   2491,  -3142, -2482 },
    {   4500,   3305,  -3671, -1567 },
    {   5973,   3172,  -1348,  -534 },
    {   4830,   3379,  -1549,   643 },
    {   5214,   3938,  -2641, -2302 },
    {   4639,   4826,  -5532,  -847 },
    {   5639,   2731,  -2170,  -963 },
    {   6084,   3487,  -3525, -1346 },
    {   5971,   3154,  -2190, -2316 },
    {   5618,   4865,  -6927,   116 },
    {   5345,   3568,  -7391,   709 },
    {   5429,   5078,  -3811, -1524 },
    {   6960,   2037,  -3515, -1096 },
    {   7092,   2531,  -4557,  -588 },
    {   6061,   4247,  -5651,  -478 },
    {   4595,   3684,  -4907,  -827 },
    {   7497,   3213,  -3048,  -424 },
    {   5996,   2137,  -3098, -1745 },
    {   6198,   5199,  -2223, -2274 },
    {   6888,   2851,  -2768, -1675 },
    {   6114,   4210,  -2316,  -954 },
    {   7127,   4242,  -3041, -1408 },
    {   6126,   3668,  -1517, -1427 },
    {   6245,   6129,  -4225, -1186 },
    {   6816,   3213,  -2101,  -964 },
    {   5345,   5276,  -2643,  -847 },
    {   6592,   4665,  -4338,   484 },
    {   6746,   3751,  -3443,   124 },
    {   5453,   1980,  -2738,  2606 },
    {   4662,   2179,  -4226, -1059 },
    {   5571,   3208,  -3554,   174 },
    {   5256,   4447,  -1815, -1481 },
    {   5400,   2570,  -1210,   235 },
    {   7056,   2549,  -2674,   318 },
    {   4574,   4340,  -2892,  -130 },
    {   6203,   4587,  -3273,  -305 },
    {   5103,   1925,  -2715, -2137 },
    {   3905,   4296,  -1700,   247 },
    {   4421,   4605,  -3299,   811 },
    {   5671,   1273,  -3870,  -924 },
    {   5486,   1805,  -4901,   133 },
    {   6437,   2578,  -1828,  -106 },
    {   5530,   5253,  -5058,  1223 },
    {   4816,   2025,  -1215,  1443 },
    {   3457,   3525,  -2456,  3217 },
    {   3316,   2595,  -1108,  2459 },
    {   3068,   3810,  -2207,  1926 },
    {   6351,   5436,  -6470,   600 },
    {   6324,   4240,  -5365,  2416 },
    {   4851,   4774,  -4075,  1878 },
    {   4900,   3679,  -5198,  1078 },
    {   8347,   3633,  -4565,  -171 },
    {   5244,   5718,  -3853,   173 },
    {   3960,   3492,  -2939,  2105 },
    {   6070,   3473,  -2351,   161 },
    {   8228,   3034,  -3360,  -901 },
    {   7006,   3985,  -1940, -1926 },
    {   7123,   4681,  -4301,  -878 },
    {   5122,   4097,  -1851,  -449 },
    {   6200,   2060,  -2251,  1049 },
    {   7106,   3844,  -7209,  2625 },
    {   7108,   3370,  -6734,   533 },
    {   6859,   2849,  -3992,  1360 },
    {   5458,   2278,  -3253,  1131 },
    {  -1072,  -2109,   4783, -1073 },
    {   -319,  -2604,   4257, -2418 },
    {   2466,   1300,   3476,  -314 },
    {   2847,  -1502,   5296,  -141 },
    {   1667,  -1273,   5559, -2725 },
    {   2877,  -3402,   6434,   204 },
    {     53,  -2637,   5275, -1181 },
    {   1091,  -2215,   5803, -1549 },
    {   2397,   -922,   4327,  1182 },
    {    219,  -3747,   4647, -1564 },
    {    -29,  -2705,   4812,  1277 },
    {   1499,  -2608,   5648,  1407 },
    {   2139,  -2399,   4202,  2791 },
    {   -426,  -2064,   5528,   151 },
    {   2560,  -2803,   6179, -2806 },
    {   4537,  -2479,   3797,  1095 },
    {    888,  -3357,   5341,  -415 },
    {   4460,  -1814,   5388, -1227 },
    {   3920,  -3268,   6364,  -703 },
    {   3343,  -4698,   4410,   784 },
    {    309,  -1897,   6306,  1223 },
    {    958,  -3318,   4254, -3167 },
    {    -99,   1596,   6018, -1983 },
    {   -429,   -853,   6407,   878 },
    {   1170,  -1322,   6290,  -417 },
    {   2288,   -505,   6303, -1999 },
    {   3312,  -1674,   6749, -2494 },
    {   -415,  -3401,   4721,  -371 },
    {   -189,  -1210,   4844, -2002 },
    {    888,  -4142,   4377,   130 },
    {   2469,  -4381,   5398, -2492 },
    {   2879,  -2912,   5094, -2598 },
    {   -717,   -617,   5650,  -685 },
    {   1470,  -3863,   5352, -1684 },
    {   3935,    -96,   3823,  -730 },
    {   3769,   -430,   3168,   694 },
    {   2556,    385,   3539,   512 },
    {     77,  -1415,   5111,  2655 },
    {   2724,  -2158,   6715,  -822 },
    {   1832,   1001,   5385, -1900 },
    {    900,   2198,   4464,  -559 },
    {    441,     69,   5921, -1743 },
    {  -1161,    738,   6732,  -308 },
    {    257,   2035,   4091,   736 },
    {   1607,   1288,   4355,   -23 },
    {    -13,   1316,   4180,  1672 },
    {   1511,   1336,   3057,  1435 },
    {   2189,  -3813,   4530,   939 },
    {   3632,   -706,   2646,  1375 },
    {   4266,  -3761,   4241,  1077 },
    {   3101,   -427,   5273, -1202 },
    {   2293,    276,   4810,  -313 },
    {   3430,  -1851,   3101,  2045 },
    {   3453,  -2979,   5142,   942 },
    {   1683,  -3281,   4802,  2002 },
    {   3954,  -4715,   5611,   578 },
    {   1272,   -155,   5085,   454 },
    {    128,   -194,   5095,  1409 },
    {    820,    880,   5797, -2658 },
    {  -1095,    656,   5774,  1095 },
    {    813,  -1669,   4320, -3251 },
    {   -119,    518,   6372,  -651 },
    {   2922,  -4299,   6115,  -877 },
    {   4205,  -4273,   4004,  2642 },
    {  -1211,  -3892,    224,  3127 },
    {    -34,  -4371,   1321,  2318 },
    {     77,  -6326,   1201,   828 },
    {   3995,  -3775,   1958,  3233 },
    {    178,  -3301,   1985,  3318 },
    {   2330,  -3801,   1033,  3195 },
    {   1413,  -5536,    826,  1709 },
    {   2468,  -3499,   3653,  3631 },
    {    741,  -4617,   1723,  2008 },
    {   1246,  -3043,   2978,  3949 },
    {   -343,  -4308,   2258,  2189 },
    {   -682,  -4640,    454,  2272 },
    {   1236,  -4829,   2491,  1642 },
    {   -512,  -3766,   1182,  3052 },
    {    119,  -3939,   3712,   971 },
    {  -1145,  -4624,   1360,  2281 },
    {    101,  -4746,   2866,  1255 },
    {  -1500,  -5455,    539,  1637 },
    {   -969,  -5909,   1414,  1128 },
    {  -1261,  -4939,   -231,  2022 },
    {   -226,  -5345,   1207,   705 },
    {   2712,  -5109,   3205,  1866 },
    {   -476,  -5913,    273,  1208 },
    {  -2039,  -4464,    624,  2545 },
    {  -2351,  -3930,   2019,  2673 },
    {  -2675,  -4849,   1522,  1990 },
    {  -1524,  -3461,   1446,  3204 },
    {    477,  -5314,   1710,  1577 },
    {    656,  -3729,   2346,  2511 },
    {    550,  -5917,   1975,  1040 },
    {   1728,  -4704,   3067,  1058 },
    {     -9,  -5247,    506,  1760 },
    {   -574,  -5135,   1675,  1672 },
    {   2129,  -3781,   3444,  2313 },
    {   1144,  -4439,   2214,  2529 },
    {   1292,  -4160,   3185,  1833 },
    {   2445,  -3262,   2534,  3227 },
    {   2266,  -4401,   2023,  2400 },
    {   -587,  -3602,   3408,  2067 },
    {   -885,  -4951,   3228,  1174 },
    {   -728,  -2711,   2807,  3552 },
    {   1019,  -3043,   3195,  2954 },
    {   1888,  -4615,   1140,  2454 },
    {    660,  -5616,    754,   800 },
    {  -1975,  -5371,   1649,  1585 },
    {  -1544,  -5436,   2422,  1081 },
    {   -422,  -5882,   2390,   750 },
    {   1336,  -5557,   2441,  1230 },
    {    136,  -4001,    267,  2854 },
    {   -522,  -3289,   2226,  2728 },
    {   -971,  -4580,   2471,   708 },
    {    704,  -5306,   3300,  1001 },
    {    325,  -3464,   3555,  2398 },
    {    794,  -3686,    848,  3169 },
    {    660,  -3017,   4584,  3242 },
    {  -1486,  -3978,   2170,  1644 },
    {  -1615,  -4650,   2688,  1844 },
    {    750,  -4578,    538,  2239 },
    {   1668,  -5849,   1455,  1031 },
    {   3486,  -4681,   2030,  2183 },
    {   2642,  -5429,   1696,  1761 },
    {   4491,  -4502,   3538,  2767 },
    {   3545,  -4528,   3514,  2982 },
    {   3269,  -3676,   2758,  3966 },
    {   5572,   1146,    209, -3379 },
    {   7459,   1053,    593, -1896 },
    {   4480,    200,   -310, -4259 },
    {   5577,   -939,    242, -3992 },
    {   8142,    442,   1257, -3083 },
    {   5442,   1261,   1424, -3236 },
    {   6260,   -183,   3125, -2532 },
    {   7179,    889,   1618, -2548 },
    {   6416,    932,   2379, -2487 },
    {   7094,   2560,    961, -3392 },
    {   7322,    463,   2732, -3735 },
    {   6632,   1577,   1912, -3272 },
    {   6312,   1349,   3028, -3460 },
    {   6105,    386,   1213,  -977 },
    {   5478,   1158,   1114,  -486 },
    {   6493,    410,   1686, -2180 },
    {   6378,   1881,   1333, -2240 },
    {   5711,    812,   1958, -1300 },
    {   6844,    877,    730, -1189 },
    {   6824,   -245,   2249, -2000 },
    {   7515,   1521,   1251, -3058 },
    {   6697,   1051,   1300, -1749 },
    {   6476,   1425,    811, -2773 },
    {   7350,    465,    -76, -2849 },
    {   6975,   2095,    567, -2492 },
    {   4691,   1736,   2660, -2289 },
    {   7837,   1456,    340, -2767 },
    {   7930,    507,    838, -2074 },
    {   6106,   1502,    766, -1110 },
    {   4891,   -659,    835, -3954 },
    {   7250,    141,   1369, -1523 },
    {   7651,     67,   1651, -2298 },
    {   7364,   -305,    601, -3132 },
    {   7179,    193,   2491, -2871 },
    {   6504,   -272,   2167, -1322 },
    {   4456,    983,   2300,  -421 },
    {   4817,    457,   1695,   371 },
    {   6914,    555,    850, -3159 },
    {   5904,   1030,    202, -1959 },
    {   6258,    880,   2233, -4503 },
    {   6029,     10,   2130, -3600 },
    {   6449,    985,   1129, -3963 },
    {   6616,    -18,   -111, -3285 },
    {   4496,    775,    817, -4276 },
    {   6134,   2338,   1470, -2973 },
    {   6911,    152,    430, -1946 },
    {   4053,    991,   3218, -1193 },
    {   5435,   1285,   3124, -2412 },
    {   5507,   1836,   1935, -1988 },
    {   5240,    689,   2189, -2670 },
    {   6638,   1719,    606, -1799 },
    {   5556,   -180,    129, -2595 },
    {   5644,   1918,   1281, -4316 },
    {   6410,   1088,   -282, -3117 },
    {   6503,   1841,    312, -3514 },
    {   6947,     20,   1358, -3886 },
    {   5464,   2109,   2398, -3194 },
    {   5616,   -407,   2140,  -498 },
    {   6121,   2707,   2379, -4096 },
    {   7303,   1846,   2266, -4095 },
    {   5444,    470,   2718, -1553 },
    {   5817,   -645,   3285, -1349 },
    {   5625,   1427,   1103, -1991 },
    {   6041,   -806,   1196, -2943 },
    {   3050,  -5722,   4070, -5460 },
    {   3420,  -4386,   4078, -5155 },
    {   6020,  -3982,   7268, -2689 },
    {   7502,  -4317,   7894, -3973 },
    {   4156,  -3558,   5247, -4316 },
    {   4725,  -4401,   7290, -1540 },
    {   6688,  -5122,   8216, -3210 },
    {   9176,  -6576,   9276, -4963 },
    {   8706,  -5708,   7987, -4621 },
    {   7060,  -3535,   6532, -3308 },
    {   5600,  -2719,   5363, -1568 },
    {   4661,  -2803,   6263, -4716 },
    {   3673,  -3636,   6147, -3433 },
    {   5305,  -2585,   6073, -2638 },
    {   7614,  -1962,   6079, -5266 },
    {   6760,  -3366,   7382, -4322 },
    {   6385,  -3883,   4797, -1353 },
    {   8182,  -5120,   4298, -4641 },
    {   9130,  -6198,   4975, -3063 },
    {   7421,  -5436,   5576, -3713 },
    {   3483,  -4898,   5443, -2745 },
    {   4907,  -5643,   6390, -4105 },
    {   8119,  -7008,   7992, -6764 },
    {   6528,  -6122,   6967, -5590 },
    {   5890,  -4190,   6624, -5688 },
    {   6815,  -7934,   7275, -5456 },
    {   5434,  -4306,   5169, -5378 },
    {   4364,  -6436,   5376, -2604 },
    {   8152,  -3404,   5913, -5048 },
    {   7983,  -4863,   4262, -2461 },
    {   8023,  -6188,   6238, -5062 },
    {   6753,  -3692,   3935, -3723 },
    {   6826,  -4760,   3284, -4051 },
    {   7224,  -7423,   4492, -3875 },
    {   6904,  -2590,   6587, -6248 },
    {   6106,  -1944,   7345, -5506 },
    {   4956,  -2990,   7808, -3146 },
    {   6908,  -6885,   5949, -1288 },
    {   7162,  -6058,   3419, -3401 },
    {   7015,  -7080,   6907, -3018 },
    {   6971,  -6832,   5646, -3273 },
    {   8014,  -5546,   5471, -1544 },
    {   6792,  -2220,   5105, -2879 },
    {   8494,  -3974,   4408, -3999 },
    {   9591,  -4866,   6027, -4558 },
    {   5264,  -5161,   6101,  -738 },
    {   5803,  -6141,   5197, -5231 },
    {   4657,  -6822,   3232, -5189 },
    {   4791,  -5135,   3809, -4665 },
    {   6108,  -5103,   2379, -3873 },
    {   4680,  -3909,   3234, -5093 },
    {   5802,  -3853,   3795, -4984 },
    {   4360,  -7483,   4802, -3877 },
    {   5429,  -7517,   5911, -3717 },
    {   6866,  -2280,   4880, -4634 },
    {  10131,  -4628,   4414, -4092 },
    {  10811,  -5189,   7746, -5337 },
    {   5663,  -8941,   5287, -5680 },
    {   8023,  -5991,   7403, -2796 },
    {   9669,  -6919,   6525, -4932 },
    {   7275,  -3796,   4962, -2547 },
    {   8848,  -4806,   5677, -3080 },
    {   8128,  -4308,   7749, -6569 },
    {   4032,  -5196,   2282, -6239 },
    {   6593,    700,   -229,   304 },
    {   8260,    539,    -66, -1259 },
    {   6605,    176,   -814,  -109 },
    {   8057,      0,     -1,  -136 },
    {   7382,    -38,   -484, -1129 },
    {   8373,   -929,    682,  -454 },
    {   7674,    690,  -1278,   546 },
    {   7326,   -517,    406, -1283 },
    {   7612,  -1715,  -1167,  1175 },
    {   8590,    441,   -782,  -710 },
    {   8572,  -1202,   -291,   260 },
    {   7308,   -147,  -1785,   414 },
    {   6787,   -353,   -672,   934 },
    {   5177,   -133,    179,    82 },
    {   4161,    -34,    447,  1497 },
    {   5997,   -902,   1533,  -121 },
    {   5727,   -871,  -1370,   945 },
    {   8386,   -252,    293,  -823 },
    {   6573,  -1354,    682,   616 },
    {   7650,  -2096,    725,   457 },
    {   8122,     78,    636, -1400 },
    {   8421,    428,  -1620,   131 },
    {   7341,  -1292,   -717,   186 },
    {   7998,    -49,   -720,   266 },
    {   5987,   -351,    669,   844 },
    {   7314,  -1620,    250,  -603 },
    {   7219,  -1562,   -572,  1994 },
    {   8682,   -358,   -290,  -388 },
    {   5810,    155,   -178,  1199 },
    {   7246,    -12,   1042,  -786 },
    {   7357,   -923,   1468,  -475 },
    {   7801,    621,   -212,  -724 },
    {   5346,   -514,   1210,  1356 },
    {   8459,     36,   -127,  -779 },
    {   6878,  -2429,    854,  1750 },
    {   7280,  -1401,  -1353,  2845 },
    {   7579,  -2148,  -1463,  2087 },
    {   6637,    946,   -872,   750 },
    {   4807,  -1100,   1289,  2602 },
    {   4495,    219,   1551,  1128 },
    {   7639,    506,    446, -1107 },
    {   6359,    188,   1009,  -115 },
    {   6641,  -1820,   1655,   723 },
    {   5394,  -2382,   1604,  2542 },
    {   6021,  -2644,   2396,  1407 },
    {   4698,    882,    245,  1525 },
    {   8103,    573,   -798,  -349 },
    {   8045,   -519,    997, -1092 },
    {   7571,   -122,    227,  -338 },
    {   5347,  -1200,    630,  1718 },
    {   7070,    790,    218,  -544 },
    {   7440,    728,   -527,   -20 },
    {   6402,   -355,    197,  -736 },
    {   4031,    771,    866,  1895 },
    {   6009,    896,    445,   -31 },
    {   5160,   1098,   -856,  1784 },
    {   7980,   -886,  -1293,  1396 },
    {   6318,  -1361,   2423,   252 },
    {   7547,   -699,    133,   506 },
    {   8562,  -2344,    940,   264 },
    {   5890,   1187,  -1425,  2194 },
    {   6558,   -645,  -1311,  2621 },
    {   4634,  -1671,   2075,  1623 },
    {   5614,    105,   -816,  2376 },
    {   6646,   1558,  -1365,   630 },
    {   6998,   1150,  -2117,  -990 },
    {   6555,   2311,  -1093, -1783 },
    {   6682,   1430,  -2391, -1940 },
    {   7861,   1555,  -2977, -1188 },
    {   6745,   1723,   -459, -2085 },
    {   7504,   1229,  -1666, -2060 },
    {   7937,    671,  -2128, -1529 },
    {   7139,    991,   -735, -2632 },
    {   6867,   1592,  -1303, -2324 },
    {   6401,   2230,  -1732, -2508 },
    {   7201,   2184,  -2169, -1988 },
    {   6636,   2190,   -995, -2840 },
    {   7620,   2306,  -2089,  -651 },
    {   7584,   1875,  -1438,  -631 },
    {   9214,   1561,  -2464, -1139 },
    {   6154,   1318,  -1237, -2917 },
    {   7917,   2847,  -1797, -1599 },
    {   8309,   2029,  -2555,  -465 },
    {   8204,   1282,   -584, -2405 },
    {   8440,   1035,  -1147, -1137 },
    {   7107,   1858,    -60, -1568 },
    {   6781,   2912,   -873, -1463 },
    {   7603,   1316,   -319, -1249 },
    {   7833,   1335,    -78, -1849 },
    {   7930,   1141,  -1016,  -695 },
    {   7883,   1610,  -1017, -1314 },
    {   8069,   1409,  -1811,  -196 },
    {   8319,   1031,   -582, -1590 },
    {   5948,   1537,  -2153, -2373 },
    {   8684,   1171,  -1871,  -850 },
    {   8357,   2484,  -2411, -1292 },
    {   6516,   2092,   -193, -1167 },
    {   6112,   1697,     22,  -525 },
    {   7161,    703,   -602, -1879 },
    {   6047,   2351,   -807,  -219 },
    {   8072,   1854,  -1817, -1553 },
    {   6956,   1304,     76, -1011 },
    {   6607,   1481,   -544,  -162 },
    {   6958,   2541,   -265, -1938 },
    {   6416,   2514,   -777,  -850 },
    {   7272,   2110,   -899, -1171 },
    {   7741,   2153,   -283, -2614 },
    {   6482,   2041,  -1758, -1221 },
    {   6762,    940,  -1862, -2281 },
    {   5610,   1194,  -1691, -1561 },
    {   7833,   2164,   -823, -1952 },
    {   5460,   1438,   -848,  1189 },
    {   6011,   1377,   -771, -1557 },
    {   7679,    544,  -1134, -2214 },
    {   7209,   1292,  -2714, -1564 },
    {   5567,   1200,   -404,  -169 },
    {   5853,   1461,  -1465,  -518 },
    {   6782,    689,   -844,  -860 },
    {   7330,   1337,  -1152,   -71 },
    {   7189,   1506,   -653,  -685 },
    {   6860,   2116,  -1403,  -240 },
    {   8804,   1516,  -1391, -1760 },
    {   7210,   2689,  -1498,  -989 },
    {   7030,   3022,  -1441, -2083 },
    {   5649,   1836,   -407,   525 },
    {   7451,   3099,   -717, -2464 },
    {   7384,   1656,  -2007,   398 },
    {   6504,    707,  -1919,  -134 },
    {  -1851,   3639,  -2279,  -695 },
    {  -4037,   1644,    -77,  1329 },
    {  -4025,   1960,  -1565,  -567 },
    {  -3430,   2495,   -795,   368 },
    {  -4771,   2480,    993,   756 },
    {  -3431,   2058,  -2539,  -971 },
    {  -3802,   3418,    380,   217 },
    {  -3074,   3350,  -1652, -1056 },
    {  -3705,    326,  -1650,  1535 },
    {  -3122,   1281,  -1192,  1607 },
    {  -4601,   1367,   -968,    53 },
    {  -3808,    958,     44,  2560 },
    {  -2079,   2530,  -1485,  1166 },
    {  -3707,    343,  -2889,   180 },
    {  -5249,   1431,    -31,   688 },
    {  -4990,    125,   -704,  1270 },
    {  -2771,   1334,  -2446,   746 },
    {  -2292,    994,  -1527,  2630 },
    {  -1261,   3070,  -2519,   268 },
    {  -2544,   3890,  -1057,  -552 },
    {  -4421,    255,  -1980,   530 },
    {  -2951,    454,    -13,  3643 },
    {  -2262,   1815,   -370,  2880 },
    {  -2383,   3657,   -649,   576 },
    {  -3541,   -161,  -1389,  2550 },
    {  -4241,   1575,   1325,  2561 },
    {  -2767,   4037,   1221,  1578 },
    {  -3748,   2697,   1148,  1801 },
    {  -4686,   2385,   -220,     0 },
    {  -1531,   1645,  -2751,  1327 },
    {    -45,   4032,   -799,  2298 },
    {  -2915,   2280,    709,  2495 },
    {  -1199,   3278,   -406,  2346 },
    {  -2471,    116,  -2706,  2060 },
    {  -2440,   2173,  -2894,  -344 },
    {  -3375,   2287,   1781,  3226 },
    {  -2153,   3568,   1827,  2918 },
    {   -862,   2267,  -1626,  2527 },
    {  -2698,   1135,    301,  4239 },
    {  -2364,   2123,   1010,  3710 },
    {  -2447,   3281,    -81,  1408 },
    {  -2660,   4735,    472,   258 },
    {  -1053,   3097,   2682,  2398 },
    {  -3366,  -1037,  -1152,  -868 },
    {   -643,   4242,   2212,  1259 },
    {    971,   3991,    934,   643 },
    {  -1617,   2002,   2139,  2195 },
    {  -4897,    972,    784,  1719 },
    {  -1275,   2992,   1039,  3821 },
    {   -392,   4973,   -209,  1821 },
    {  -1028,   4718,  -1479,  -137 },
    {     50,   3914,    553,  2210 },
    {    678,   4364,    359,  1303 },
    {   -582,   4911,    514,  1671 },
    {   1276,   3914,  -1252,  2934 },
    {  -1496,   3984,    857,  2330 },
    {    772,   4744,   -655,  2332 },
    {   -799,   5283,   -439,   624 },
    {   1341,   2937,    650,  2027 },
    {  -1739,   4892,   1275,  1702 },
    {   -892,   2596,   -151,  3951 },
    {  -3532,   1090,   1292,    32 },
    {    321,   3146,   2647,  1475 },
    {    264,   4199,  -1591,  1317 },
    {   -452,  -2357,   2266,  4192 },
    {   3022,  -1033,  -2389,  5678 },
    {  -1162,  -1342,   3543,  4990 },
    {   -474,  -1477,  -1223,  5016 },
    {   -699,  -2857,    900,  3835 },
    {   -461,  -2255,   -117,  4626 },
    {   1204,  -2062,  -1211,  4403 },
    {   2192,  -3035,   -337,  3966 },
    {    108,   -831,    279,  5643 },
    {   1457,   -620,  -2908,  5276 },
    {  -2527,    -78,   1085,  5460 },
    {  -1978,  -1918,   -949,  4733 },
    {     32,    367,  -1904,  5166 },
    {   1890,  -1665,    440,  4752 },
    {   -518,   -348,   2816,  4891 },
    {   3695,  -2490,  -1374,  4603 },
    {    246,  -1965,   3549,  3969 },
    {   1100,  -3111,    656,  3737 },
    {  -1379,    870,   -414,  4575 },
    {    628,   -357,  -1227,  6179 },
    {  -1129,  -1318,  -2457,  4576 },
    {   -425,    -98,    -73,  6336 },
    {    367,   -887,   2990,  4207 },
    {   2091,  -1251,   2444,  3557 },
    {  -1759,  -1610,   2046,  5273 },
    {   3210,   1414,    -20,  2616 },
    {   3303,  -2636,   1005,  4237 },
    {   -327,  -3107,   -640,  3687 },
    {   -197,    764,    572,  5486 },
    {    646,   -767,   1388,  5464 },
    {    104,   2742,   -228,  3907 },
    {   -236,   1829,   -579,  4585 },
    {  -2150,   -474,  -1525,  4006 },
    {    -23,  -2632,  -2400,  3892 },
    {    -12,  -1739,  -2910,  4867 },
    {  -2310,   -368,   -102,  4583 },
    {  -1991,  -2061,    533,  4531 },
    {   3884,  -1446,   -153,  4393 },
    {   1568,     14,   -289,  5268 },
    {  -1376,   -253,  -2797,  3417 },
    {   3193,  -2577,   2475,  3566 },
    {   3418,    617,   1350,  1857 },
    {   3792,    -24,   -272,  3370 },
    {    153,   1159,   2906,  2877 },
    {    511,   2162,   1548,  2741 },
    {    262,    819,  -2791,  3734 },
    {   4232,  -2015,   1486,  3477 },
    {   2943,  -1110,  -1014,  5480 },
    {   2842,    369,    703,  3476 },
    {   3011,   1634,   -933,  3553 },
    {   4412,  -1548,   -942,  5021 },
    {  -1405,    593,   2372,  5267 },
    {   2093,   2129,    896,  2365 },
    {   4845,  -1980,      0,  3823 },
    {  -2140,     81,   3278,  5637 },
    {   1484,   2665,   -324,  3653 },
    {     10,    192,   1620,  5291 },
    {   2152,    738,  -2269,  5000 },
    {   2102,   2748,  -1652,  4707 },
    {   2855,  -2131,   -387,  5188 },
    {   1173,    676,   1338,  3277 },
    {   2340,  -2329,  -2064,  4095 },
    {    861,  -2024,   1296,  5055 },
    {   2189,   3225,   -695,  2626 },
    {   6196,  -7079,   1943,  -822 },
    {   4547,  -4813,   3261,  1856 },
    {   4243,  -6904,   3443,   448 },
    {   4581,  -7503,    946,   506 },
    {   6626,  -7754,   3427,   470 },
    {   3407,  -9088,   3269, -1496 },
    {   4079,  -6464,   2304,   777 },
    {   5621,  -9336,   2684,  -768 },
    {   5351,  -6464,   5238,  -214 },
    {   5961,  -8007,   1724, -3091 },
    {   4213,  -8067,    603,  -246 },
    {   7208,  -7403,   3168, -1738 },
    {   6098,  -7700,    329, -1379 },
    {   6525,  -6735,   4248, -1072 },
    {   6073,  -6241,   2167, -2378 },
    {   4609,  -9218,   3051, -1033 },
    {   6813,  -7283,   1581, -1897 },
    {   6126,  -6275,   2789,   681 },
    {   4423,  -6538,   1621, -1692 },
    {   6272,  -8298,   3167, -1855 },
    {   6172,  -8558,   4498, -1169 },
    {   4844,  -8588,   1647,  -366 },
    {   6209,  -8807,   1581,  -369 },
    {   5389,  -8059,    550,  -192 },
    {   6654,  -9775,   2504, -1063 },
    {   7103,  -7998,    806,   530 },
    {   5662,  -6736,   1565, -3620 },
    {   4165,  -9564,   4191, -2131 },
    {   4526,  -7181,    576, -2875 },
    {   4633,  -8623,   2807, -4742 },
    {   3709,  -7794,   1815,    34 },
    {   3634,  -8622,   2313,  -826 },
    {   6991,  -8447,   2063, -3198 },
    {   7757,  -9486,   2255,  -558 },
    {   4149,  -7778,   4728, -1696 },
    {   5767,  -7427,   1113,   707 },
    {   4592,  -6261,   2329,  1864 },
    {   3159, -10498,   1677, -4273 },
    {   3534,  -9010,   2437, -3565 },
    {   4479, -10821,   2715, -4942 },
    {   3207,  -9805,   3054, -3886 },
    {   4627,  -8189,   3018, -2354 },
    {   5527, -10566,   3244, -2749 },
    {   4346, -10127,   3335, -3084 },
    {   6132, -10085,   3316, -1308 },
    {   5629,  -9704,   2178, -3058 },
    {   3603,  -8538,   1246,  -624 },
    {   3737,  -8488,    395, -3167 },
    {   5465, -11414,   2810, -4640 },
    {   5306,  -7745,   2721, -3988 },
    {   7000,  -9111,   1695, -1409 },
    {   6663,  -7741,   2466, -4079 },
    {   4083,  -7175,   1836, -4831 },
    {   3613,  -9926,   1342, -3455 },
    {   6588,  -8033,    457,  -258 },
    {   4720,  -8102,     17, -1209 },
    {   7414,  -8709,   1294,  -344 },
    {   5437, -10030,   4043, -1704 },
    {   4862,  -9281,   1558, -1431 },
    {   6800,  -6403,   5113,   862 },
    {   4623,  -8242,   2667,  -228 },
    {   5919,  -5083,   3348,  2135 },
    {   5985,  -8889,   2733, -5105 },
    {   5029,  -5767,   4407,   719 },
    {    354,  -6158,   -838, -3001 },
    {    351,  -5943,  -2104, -1534 },
    {   -633,  -7190,    -25, -4798 },
    {  -1595,  -7235,  -3812, -1400 },
    {    103,  -6197,  -2933,   -78 },
    {  -1722,  -5020,  -3441, -4333 },
    {  -1963,  -5644,  -4365,  -270 },
    {   -846,  -5743,  -3477,   196 },
    {   -191,  -5348,  -4054,  -469 },
    {  -2515,  -7754,  -3495,  -818 },
    {  -2090,  -6710,  -2701,   117 },
    {   -546,  -7036,  -1398,   163 },
    {   -278,  -7091,  -2662,  -536 },
    {   -622,  -7962,  -2731, -1464 },
    {  -1555,  -8118,  -3612, -2057 },
    {  -1094,  -6280,  -2314,   505 },
    {  -2556,  -8538,  -4024, -2247 },
    {    109,  -7134,  -3107, -1823 },
    {   -900,  -6954,  -3340,  -717 },
    {   -605,  -7113,  -3656, -2154 },
    {    837,  -6263,  -3211, -2177 },
    {   -417,  -5810,  -3871, -1469 },
    {  -1318,  -5649,  -4207, -3198 },
    {    413,  -6765,  -2082,   -33 },
    {  -3101,  -6450,  -4362,  -766 },
    {    755,  -6489,  -2967,  -846 },
    {   1117,  -7106,  -2452, -1352 },
    {  -1202,  -8387,  -3072, -2897 },
    {   -365,  -4894,  -3561, -2937 },
    {  -2372,  -8776,   -265, -4441 },
    {  -1224,  -8678,   -896, -5074 },
    {   -755, -10096,   -600, -6623 },
    {    300,  -8206,   -225, -4568 },
    {  -1176,  -6824,  -2633, -3527 },
    {  -2006,  -5443,  -1526, -5849 },
    {  -1115,  -5540,  -2363, -4785 },
    {   1059,  -6812,  -2543, -2654 },
    {  -1976,  -6861,  -3062, -5508 },
    {   -379,  -5328,  -2321, -3624 },
    {  -2108,  -5860,  -4518, -1915 },
    {   -379,  -7885,  -1329,  -594 },
    {    774,  -5389,   -581, -5213 },
    {  -2601,  -5083,  -1849, -4921 },
    {   -176,  -5580,     74, -5075 },
    {   -204,  -6780,   -190, -6232 },
    {    418,  -7594,  -1987,  -820 },
    {  -1873,  -8529,  -2926, -1609 },
    {   1340,  -6362,   -919, -4975 },
    {    577,  -7990,  -2044, -1873 },
    {  -2572,  -7413,  -1745, -2224 },
    {  -2037,  -7030,  -1461, -7138 },
    {  -2559,  -8756,  -2039, -5836 },
    {  -2079,  -6764,  -1209, -5669 },
    {  -1613,  -7801,  -2006,  -685 },
    {  -1865,  -6583,   -722, -3529 },
    {   -589,  -6358,  -1377, -1003 },
    {   -540,  -7514,  -1331, -3542 },
    {    419,  -6192,  -1677, -4927 },
    {  -2786,  -8763,  -2966, -5065 },
    {  -2172,  -8411,  -1726, -4675 },
    {  -3382,  -9833,  -3497, -5722 },
    {  -2433, -10169,  -2077, -5775 },
    {   -424,  -9451,  -1096, -3658 },
    {   -537,  -8522,   -910, -1897 },
    {  -5550,   2807,   1683,  -693 },
    {  -6395,    635,   3573, -1246 },
    {  -7544,   2280,   2140,    44 },
    {  -8751,   1136,   2951,  -794 },
    {  -5605,   2709,   2052,   916 },
    {  -7650,    654,    869,   135 },
    {  -6939,    967,   1409,   870 },
    {  -7834,   2123,   3310,   974 },
    {  -6935,   2818,   1274, -1678 },
    {  -5605,   2233,   1013,   471 },
    {  -7095,   1849,   1648,   198 },
    {  -6636,   1634,    712,   -37 },
    {  -7279,    978,    296,  -315 },
    {  -7664,   3504,   3292,  -216 },
    {  -7836,   1209,   1221,  -257 },
    {  -7913,   2201,   1765, -1529 },
    {  -7077,   3783,   2632, -1407 },
    {  -5565,   1645,   1410,  -622 },
    {  -6494,   2879,   1181,  -759 },
    {  -7073,   3137,   3010,   550 },
    {  -7249,   1839,    847,  -805 },
    {  -6630,   2197,    282, -1096 },
    {  -8836,   1573,   1988, -1090 },
    {  -7809,   1274,    836, -1198 },
    {  -7895,   2970,   3511, -1097 },
    {  -6960,   1664,   1356, -2442 },
    {  -6582,   2866,   2273,   307 },
    {  -7221,    821,   2851, -1435 },
    {  -6015,   1703,   2001, -2367 },
    {  -8082,   1034,   2103,   239 },
    {  -5952,   1912,    301,  -465 },
    {  -6099,    841,    379,   567 },
    {  -6343,     50,    494,   658 },
    {  -6586,    983,    591,  -893 },
    {  -5500,    869,   2187, -2479 },
    {  -6482,     60,   1545,  -979 },
    {  -6705,    515,   1974,   -53 },
    {  -6460,   1755,   1325, -1275 },
    {  -6093,   2617,   2465,  -623 },
    {  -7330,   2161,    594, -2115 },
    {  -7324,    762,   1593, -2004 },
    {  -6385,    679,   1510, -2514 },
    {  -6159,    241,   2976, -1631 },
    {  -8583,   3030,   4045,  -162 },
    {  -6299,     66,   2209, -2103 },
    {  -5428,   1279,   3267, -1846 },
    {  -6438,   1335,   2728, -1631 },
    {  -8012,   1070,   2428, -1151 },
    {  -6201,   2781,   2349, -1918 },
    {  -5918,   1139,   3121,  -148 },
    {  -6314,   2481,   3137, -1808 },
    {  -7180,   1722,   2435, -1602 },
    {  -6750,   1829,   3763, -1145 },
    {  -6713,   1777,   2221,  1212 },
    {  -7479,   1835,   3627,  -479 },
    {  -7299,     10,   2406, -1593 },
    {  -8249,   3129,    996, -2870 },
    {  -8374,   1534,   1333, -1882 },
    {  -7507,   3353,   1598, -2299 },
    {  -7379,   2701,   2326, -1167 },
    {  -8440,   2276,   2796,  -542 },
    { -10348,   1527,   2649, -1165 },
    {  -8184,   3614,   2574, -1738 },
    {  -5539,   1574,   1733,  1138 },
    {   9404,  -7652,     67,    79 },
    {   8654,  -3972,   1358,   -60 },
    {   8617,  -4794,    117,  2318 },
    {   7886,  -4505,   1784,  1200 },
    {   8636,  -6125,   3879, -1003 },
    {   9654,  -6836,   1816,   205 },
    {   9374,  -6553,    913,  1875 },
    {   8020,  -6150,   1134,  2390 },
    {   7786,  -4970,   2078, -1857 },
    {   8691,  -6119,    711,   708 },
    {   9039,  -5568,   2944, -1902 },
    {   9955,  -5048,   1433,  -601 },
    {   8089,  -6927,   3093, -2846 },
    {   8487,  -7024,   2415,    19 },
    {   9388,  -5287,   3577, -2655 },
    {   8591,  -7371,   2300,  -996 },
    {   9104,  -4763,   1453, -2558 },
    {   7615,  -5457,    596,   164 },
    {   9860,  -7047,   3433,  -614 },
    {   8756,  -4404,   2235,  -964 },
    {   9462,  -4660,    299, -1822 },
    {  10119,  -5550,   2689, -1273 },
    {  10915,  -7471,   2705, -1007 },
    {  11433,  -7090,   1410, -1198 },
    {   9882,  -7431,   2965, -1895 },
    {   7628,  -5219,    769, -2661 },
    {   8169,  -5318,   2262,    70 },
    {   8846,  -6320,   1939,  -754 },
    {   7147,  -5593,   1248,  -971 },
    {  10652,  -5485,    935,   137 },
    {   7778,  -6533,   2564, -1932 },
    {   8878,  -5173,   1214,  -361 },
    {   9828,  -4943,    282,   510 },
    {  10042,  -6134,   3895, -1914 },
    {   7965,  -6630,   3566,  -433 },
    {   8573,  -4502,   3574, -1209 },
    {   8398,  -4801,   1031, -1347 },
    {  10136,  -7772,   2612,  1547 },
    {   9890,  -7280,   1768, -1083 },
    {   8407,  -6585,   -706,   -58 },
    {   7976,  -7582,    229,  -131 },
    {  10481,  -8866,   1166,  -147 },
    {  10914,  -4342,   3189, -2412 },
    {  10440,  -5198,   -104, -1109 },
    {  11227,  -6530,   2381, -2449 },
    {   8487,  -8064,   1086,   230 },
    {   9975,  -6123,   -857,  -134 },
    {   8339,  -6498,   1232, -2337 },
    {  11042,  -4506,   1119, -2098 },
    {  12563,  -5592,   1837, -2062 },
    {  11801,  -5590,    632, -1296 },
    {  10152,  -5617,   1511, -1917 },
    {   7800,  -6473,     51, -1337 },
    {   7941,  -5560,   2438, -3270 },
    {   6554,  -3834,   2100,  1476 },
    {   9065,  -5520,   -226, -1120 },
    {  10794,  -7120,   -243,   122 },
    {  10429,  -6968,    272,  -806 },
    {   8942,  -8914,   1442,  -392 },
    {   9969,  -5051,   2033, -2953 },
    {   7275,  -4152,   3058,   -64 },
    {  11127,  -5488,   4589, -3227 },
    {   9626,  -6666,   2739, -2958 },
    {   6943,  -5362,   4470,  1008 },
    {  -7456,   -967,   2936, -1002 },
    {  -8622,   -333,   6962,  2606 },
    {  -7486,  -3392,   3668,  1287 },
    {  -8053,   -827,   5148,  1097 },
    {  -6610,    454,   4952,    96 },
    {  -7701,  -1982,   3161,  -468 },
    {  -7307,  -1132,   4071,   -36 },
    {  -8125,   -271,   5199,  3862 },
    {  -9182,  -1950,   2813,  1878 },
    {  -9855,   -952,   4794,  3010 },
    {  -7241,   1431,   4202,  2468 },
    {  -9646,    157,   4766,  1046 },
    {  -9371,   1230,   6009,  2958 },
    { -11514,    -64,   8630,  5248 },
    {  -6766,    565,   2766,  2140 },
    {  -8426,     -9,   2852,  1271 },
    { -11291,  -1113,   5087,  2937 },
    {  -8297,   2092,   4495,  1264 },
    {  -9983,    735,   3809,   -51 },
    {  -9048,  -1000,   3191,  -308 },
    {  -7331,  -1987,   2655,  1391 },
    {  -7144,    -21,   4333,  2161 },
    {  -6032,  -1540,   3543,   896 },
    {  -7987,  -1036,   1985,  1529 },
    {  -9264,   2004,   5194,   290 },
    { -11308,   -840,   5754,  1654 },
    {  -9130,  -2398,   4292,  2973 },
    {  -6248,    838,   3563,  1223 },
    {  -6819,  -2760,   3511,   119 },
    {  -7213,  -2006,   4364,   762 },
    {  -5431,  -1047,   4533,   166 },
    {  -7098,   -641,   2021,   639 },
    {  -8628,  -2249,   3588,   399 },
    {  -6352,  -1498,   3560,  -648 },
    {  -7033,  -2190,   4870,  2562 },
    {  -7405,    -46,   3772,  -581 },
    {  -6104,    796,   5143,  1965 },
    {  -5787,    943,   5784,  3030 },
    {  -8367,   1465,   7192,  4097 },
    {  -8259,    789,   5694,  1963 },
    { -10614,  -1899,   5748,  2645 },
    {  -8258,   -805,   3698,  2275 },
    {  -6877,   -972,   6431,  3160 },
    {  -6483,    363,   7018,  3129 },
    {  -6283,  -1358,   5191,  1524 },
    {  -8853,  -3157,   4119,  1741 },
    {  -6086,   -267,   3883,  -835 },
    {  -7254,   1032,   6613,  4017 },
    { -11470,  -3350,   4649,  3426 },
    {  -6743,    481,   6148,  1239 },
    {  -5394,   -166,   5309,  3165 },
    {  -7958,   1068,   4268,  -240 },
    { -10520,   2256,   7916,  2828 },
    {  -5132,     -4,   5739,  1176 },
    {  -8643,    120,   3255,  -629 },
    {  -9631,   1974,   8870,  4362 },
    { -10663,  -1221,   3733,   589 },
    {  -8224,  -1843,   5806,  2655 },
    {  -8282,   1255,   8647,  3478 },
    { -12311,  -1505,   9043,  6256 },
    { -11312,   -856,   7136,  4681 },
    { -11944,   -722,   7941,  3309 },
    {  -7868,   -463,   6846,  4196 },
    {  -8679,   -241,   7410,  5347 },
    {   6759,  -4680,   -508,  1220 },
    {   5176,  -6111,    944,   121 },
    {   6843,  -5667,  -1368,  -533 },
    {   5616,  -5884,  -1471,  -695 },
    {   6030,  -5089,  -1808,  -940 },
    {   7444,  -5463,    -52,  1881 },
    {   4207,  -6079,   -506,  1571 },
    {   6785,  -4410,   -649,  3084 },
    {   4838,  -5214,   2026,  2998 },
    {   4201,  -5790,    645,  1811 },
    {   6930,  -5129,  -1940,  1698 },
    {   6332,  -4627,    692,  3027 },
    {   6285,  -4314,   -106,  3644 },
    {   6255,  -5450,  -1975,   742 },
    {   4199,  -4676,   -459,  1796 },
    {   5592,  -5500,   1345,  1300 },
    {   4358,  -5556,  -2236,   114 },
    {   4620,  -5875,  -1563,   888 },
    {   4892,  -7550,   -327,  -419 },
    {   4734,  -7085,      7,   613 },
    {   3883,  -5562,  -1969,  1080 },
    {   5610,  -4990,   -204,   834 },
    {   4117,  -6482,  -1271,   341 },
    {   6585,  -5107,    892,  1169 },
    {   6632,  -3683,    302,  3002 },
    {   6326,  -5351,   -983, -1250 },
    {   4382,  -7192,   -730,  -158 },
    {   5227,  -6540,   -451,  1123 },
    {   5468,  -6472,   -870, -1471 },
    {   5191,  -6402,  -1365,  -127 },
    {   7407,  -6317,   -973,  -336 },
    {   4611,  -6530,   -820, -1980 },
    {   4963,  -5159,  -2050,  -966 },
    {   4414,  -5691,   -211,  -998 },
    {   5954,  -5873,    750, -1749 },
    {   4394,  -4796,  -1268,   254 },
    {   7161,  -6214,  -1010,   689 },
    {   4965,  -3598,   2372,  1711 },
    {   6248,  -6180,    981,   864 },
    {   6473,  -5336,    525,  -600 },
    {   4591,  -6864,  -1131,  -900 },
    {   6314,  -6440,  -1021,  -375 },
    {   5838,  -6209,  -1199,   944 },
    {   5308,  -5283,  -2100,  1267 },
    {   4342,  -5860,  -1637, -1356 },
    {   5680,  -4388,  -1227,  -104 },
    {   4900,  -4098,   1449,  4046 },
    {   4677,  -4284,   -106,  3190 },
    {   7574,  -6173,   -848,  1859 },
    {   6493,  -7207,   -131,   726 },
    {   5513,  -5261,  -2117,     4 },
    {   6191,  -7352,   -193,  -505 },
    {   5885,  -4333,    324,  -134 },
    {   6162,  -6081,   -312, -2044 },
    {   4216,  -6200,  -1810,  -572 },
    {   5652,  -7035,   -696,  -197 },
    {   7131,  -7189,   -366,   -60 },
    {   5032,  -4803,  -1514,  2832 },
    {   7386,  -4610,   -606,  3489 },
    {   4211,  -5031,   1221,  3047 },
    {   4050,  -4653,   1584,  1469 },
    {   6852,  -5302,  -1861,   206 },
    {   7736,  -4816,  -1794,  3359 },
    {   6290,  -3439,   1522,  2454 },
    {   1768,   5990,  -5560, -2594 },
    {   3903,   5326,  -1530, -1501 },
    {   2472,   3738,  -2117, -4240 },
    {   3260,   5448,   -904, -4733 },
    {   1435,   7297,  -3676, -4102 },
    {   4096,   5951,   -656, -3312 },
    {   2178,   6009,  -3146, -3724 },
    {   3787,   5493,  -5473, -1633 },
    {   2998,   7286,  -3334, -3571 },
    {   2894,   6576,  -4708, -2804 },
    {    830,   6163,  -4286, -3348 },
    {   4755,   5569,  -1730, -2739 },
    {   4604,   6065,  -3562, -2605 },
    {   2749,   5141,  -3986, -2775 },
    {   3942,   4875,  -2143, -3340 },
    {   2819,   8517,  -2004, -2724 },
    {   2146,   6298,   -689, -3093 },
    {   5196,   6504,  -3393, -1475 },
    {   1851,   8386,  -1748, -1420 },
    {   3474,   8572,  -3534, -2688 },
    {   4503,   7560,  -3561, -2245 },
    {   4433,   6219,  -2393, -1575 },
    {   3506,   7248,  -2275, -1977 },
    {   3490,   7409,  -3147,  -604 },
    {   4214,   6447,  -3520,   516 },
    {    619,   7034,   -829, -1705 },
    {   1732,   7395,   -356, -2208 },
    {   1226,   5204,  -3294, -3732 },
    {   2027,   5619,  -1813, -4146 },
    {   3078,   5877,     47, -2651 },
    {   1654,   5458,    424,  -682 },
    {   3163,   5464,  -2026,  -270 },
    {   2884,   5375,   -685,  -530 },
    {   2950,   7286,    -35, -2967 },
    {   1986,   5066,   -597,   482 },
    {   3459,   4308,  -3845, -2333 },
    {   3155,   7037,  -1346, -4345 },
    {   2193,   6696,   -717, -1319 },
    {   3677,   5089,  -3892,  -487 },
    {   2186,   5136,  -4186, -1492 },
    {    773,   5796,   -917,   817 },
    {   2489,   6546,  -3570, -2117 },
    {   1223,   6469,  -1362,   -33 },
    {    271,   6061,  -1466, -1725 },
    {   2540,   5171,  -1847,  1032 },
    {   2548,   5251,  -2697,  1677 },
    {    771,   7600,   -768,  -632 },
    {   4710,   6647,  -4736, -1275 },
    {   1369,   5917,  -2971, -1056 },
    {    163,   5239,  -3499, -2275 },
    {   2104,   4285,  -3211, -3286 },
    {   1107,   7411,  -1972, -1671 },
    {   2196,   7262,  -2310, -1926 },
    {   -244,   6439,  -1745,  -839 },
    {   3293,   3832,  -2890, -3000 },
    {    419,   6443,   -379,  -407 },
    {   3077,   4930,  -1156, -2869 },
    {   2131,   5874,  -2330,   224 },
    {    690,   6538,  -2212, -2841 },
    {   1602,   4421,  -2515,  1542 },
    {   3318,   9373,  -3032, -3477 },
    {   5646,   7462,  -5153, -1463 },
    {   4139,   7137,  -1539, -3321 },
    {   3481,   9077,  -1645, -3653 },
    {  -7747,    375,   -106,  -543 },
    {  -8587,  -1379,   -586,  -461 },
    { -10146,   -892,   2094,   694 },
    {  -8103,    382,    504,  -325 },
    {  -8548,    -92,     94,  -656 },
    {  -7460,     38,    152,   388 },
    {  -8266,   -271,   -459,  -883 },
    {  -7935,   -664,  -1026,  -802 },
    {  -8341,   -109,    853,   161 },
    {  -8802,  -1355,   1099,   630 },
    {  -8957,     -6,   1108,  -669 },
    {  -7260,  -1520,    -43,  -407 },
    {  -7555,   -174,    668, -2562 },
    {  -9014,   -126,    227, -1191 },
    {  -8184,    769,    290, -1375 },
    {  -9476,     55,    962, -1528 },
    {  -8679,    541,    755, -1030 },
    {  -9842,  -1626,    838, -1588 },
    {  -8513,   -702,    788, -1998 },
    { -10101,  -1558,   -366, -1841 },
    {  -8135,     78,   1479, -1813 },
    {  -9128,   -454,    313, -1786 },
    {  -7554,  -1084,    831, -2442 },
    {  -7576,   -701,   2068, -1665 },
    {  -7791,  -1481,   1587, -1808 },
    {  -6701,   -596,    -97,   802 },
    {  -7418,    -15,    684,  -963 },
    {  -7127,   -477,   -139,  -426 },
    {  -8097,   -110,    -36,  -264 },
    {  -7620,  -1922,   -590,  -101 },
    {  -7647,  -1201,    279,   660 },
    {  -7856,  -1974,    758, -2271 },
    {  -8496,   -167,   2232, -1143 },
    {  -8506,  -1359,    624,  -740 },
    {  -7274,  -1052,   1062,  -139 },
    {  -7800,   -217,     91, -1794 },
    {  -7030,  -1694,   -955,   615 },
    {  -9020,  -1864,    101, -2182 },
    {  -9400,   -740,    598,  -667 },
    {  -8448,  -1184,   2024, -1272 },
    {  -8812,   -570,   -897, -2384 },
    { -10559,  -1286,    538, -1536 },
    {  -8728,   -888,  -1089, -1397 },
    {  -7080,  -1185,    636, -1252 },
    {  -9880,    233,   2344,  -782 },
    {  -7952,  -1326,   -378, -1947 },
    {  -7207,   -378,   1408, -2237 },
    {  -8467,  -1545,    902, -1987 },
    {  -9163,  -1474,    924, -1739 },
    {  -8159,   -992,    -77, -2744 },
    {  -8343,    148,   -423, -1573 },
    {  -9105,   -649,   -254, -1214 },
    {  -8939,    456,    281, -1905 },
    {  -8837,    179,   -394, -2634 },
    {  -9145,    757,   1547, -1319 },
    {  -9775,   -723,    441, -1680 },
    {  -8910,   -686,   1529, -1525 },
    {  -9492,  -1134,   2064,  -938 },
    {  -6111,   -943,    677,   -31 },
    {  -7411,   -613,   -814,    46 },
    {  -9479,   -922,   -430, -2061 },
    { -11298,  -1268,   1318, -1117 },
    {  -8190,    832,    671, -2214 },
    { -10453,   -550,   1672,  -886 },
    {   1044,   9353,  -1651, -5423 },
    {   1034,   8149,   -455, -6166 },
    {    761,   8293,  -3214, -4838 },
    {    938,   8077,    164, -5130 },
    {   1295,   8673,   2582, -5490 },
    {   -314,   7973,  -2395, -5231 },
    {   -507,   9012,  -2497, -5775 },
    {   2396,   8314,  -1022, -4673 },
    {  -1516,   8501,   1950, -4969 },
    {   -308,   7401,   1549, -4866 },
    {   -112,   8340,   3003, -4920 },
    {    -50,   9315,   1371, -5666 },
    {   -659,   9449,   2496, -5547 },
    {   2573,   9148,  -2270, -4783 },
    {    830,   7104,   -438, -3907 },
    {    522,  10672,   -677, -6483 },
    {  -1190,  10108,   -510, -6518 },
    {   -427,   8271,   -579, -6315 },
    {   1602,   8113,  -1927, -4418 },
    {  -2266,   8180,    448, -5190 },
    {  -1633,   8816,   -226, -5771 },
    {    759,   9481,   -105, -5813 },
    {   2254,   6679,   -466, -5662 },
    {    -88,   6946,    895, -5958 },
    {  -1705,  10009,   1394, -5574 },
    {    748,   7943,    540, -6692 },
    {   1411,   7009,    232, -6145 },
    {    697,   7290,  -1221, -5342 },
    {  -1764,  10580,   1944, -3981 },
    {  -1334,   9124,   1195, -3903 },
    {   -905,  10067,    635, -5039 },
    {    664,  10680,     49, -4625 },
    {   1374,   9536,   -777, -3591 },
    {    252,   9698,   -597, -2931 },
    {    824,   9164,  -1014, -2144 },
    {   2438,  10569,  -2289, -4424 },
    {   2101,   7102,    507, -3614 },
    {    294,   8051,   -432, -1518 },
    {   -665,  10337,    547, -2852 },
    {   1168,  11989,   -492, -5427 },
    {   1344,   6416,    302, -5061 },
    {  -1727,  12264,   1507, -4543 },
    {    674,  10889,   -902, -3605 },
    {   -582,   9504,    300, -3618 },
    {    641,   7654,    689, -2109 },
    {   2065,   9243,    508, -4367 },
    {   1055,   8373,    688, -3144 },
    {   -641,   8185,    986, -3307 },
    {   1120,   7426,   1785, -3757 },
    {   1660,   8070,   -593, -3104 },
    {   2002,   9467,  -1722, -3475 },
    {   2361,   8368,    100, -3709 },
    {   -772,   7845,   -613, -4988 },
    {   1485,   7430,   1896, -6127 },
    {   -432,   7823,   -947, -2882 },
    {    313,  11122,   -760, -4871 },
    {    412,   8412,   -283, -4231 },
    {   1585,  10402,  -1884, -3267 },
    {    321,   6952,    773, -3016 },
    {   -105,   9014,    121, -2249 },
    {   1585,  10313,   -977, -4812 },
    {   1619,  11869,   1306, -6876 },
    {  -1168,   8886,    -81, -2500 },
    {   -395,  10886,    733, -6490 },
    {  -4949,   4274,   3992, -1054 },
    {  -4241,   5299,   4262, -1584 },
    {  -2710,   3862,   4552, -1673 },
    {  -4608,   2472,   3672, -1715 },
    {  -2843,   2816,   4003, -2326 },
    {  -5229,   2964,   5636,    90 },
    {  -4924,   3442,   5015, -1096 },
    {  -1281,   3313,   5537, -2066 },
    {  -3808,   1939,   4351,  -919 },
    {  -1915,   2585,   4939, -1614 },
    {  -3470,   1843,   5562,  -682 },
    {  -3800,    870,   5827,   144 },
    {  -4985,   1452,   4728,  -709 },
    {  -3745,   2750,   7220,   259 },
    {  -1875,   1900,   6514,  -826 },
    {  -4329,   1574,   7192,  1304 },
    {  -5408,   1444,   6208,   631 },
    {  -3327,   5312,   5707, -1541 },
    {  -6966,   3334,   4034,  1028 },
    {  -7484,   4245,   4218,  -212 },
    {  -6567,   5839,   4539,  -512 },
    {  -5715,   5935,   3747, -1186 },
    {  -6410,   4881,   3356, -1610 },
    {  -5146,   2590,   2850,  2172 },
    {  -5196,   4095,   2569,  -373 },
    {  -5043,   6025,   4318,   692 },
    {  -5525,   4884,   3513,   370 },
    {  -6804,   7533,   5812,  -488 },
    {  -5657,   2480,   4061,  1234 },
    {  -3155,   1472,   6071,  1188 },
    {  -3427,   5217,   3442,   858 },
    {  -4698,   3013,   5517,  2586 },
    {  -4449,   2226,   5418,  3580 },
    {  -6395,   3547,   5487,  2028 },
    {  -3500,   5019,   4787,     1 },
    {  -4038,   2578,   3073,  3151 },
    {  -2750,   1955,   4469,  3856 },
    {  -5696,   1659,   6118,  2469 },
    {  -4350,   1241,   6840,  3126 },
    {  -5565,   5058,   5196,  1314 },
    {  -1642,   4190,   3948,   607 },
    {  -1233,   4108,   4850,  -640 },
    {   -997,   3428,   3239,  1378 },
    {  -6488,   2741,   6926,  2792 },
    {  -4188,   3763,   4235,  2018 },
    {  -3210,   3224,   5646,  1427 },
    {  -5526,   6909,   5070,  -627 },
    {  -2815,   3994,   3425,  1903 },
    {  -2163,   2734,   5423,   145 },
    {  -4149,   4247,   2355,   734 },
    {   -410,   2521,   4138,   -16 },
    {  -2411,   2385,   4927,  2105 },
    {  -6077,   3591,   3114,   594 },
    {  -4186,   4834,   5926, -1004 },
    {  -7315,   3369,   5966,   448 },
    {  -7042,   5721,   5771,   238 },
    {  -4466,   3907,   3535, -1751 },
    {  -2116,   3970,   6163, -1392 },
    {  -7239,   2143,   8407,  3630 },
    {  -5431,   4486,   6486,   -42 },
    {  -1874,   1617,   6333,   519 },
    {  -6478,   2629,   4634,  -505 },
    {  -7784,   2342,   7216,  1365 },
    {  -1154,   1432,   4831,  1544 },
    {  -4964,  -5801,   1797,   506 },
    {  -4436,  -6905,   1059, -1237 },
    {  -5400,  -6886,    884,  -290 },
    {  -6259,  -7103,    523,  -227 },
    {  -4819,  -6450,   1412,  -450 },
    {  -4056,  -6213,   1725,  -943 },
    {  -5642,  -6091,   1357,   605 },
    {  -4196,  -5678,   2187,  -173 },
    {  -4726,  -5126,   2470,   321 },
    {  -6642,  -5091,   1507, -1005 },
    {  -5304,  -5250,   1944,  1579 },
    {  -7179,  -5520,   1468,  -425 },
    {  -6033,  -4895,   1876,  -955 },
    {  -6595,  -5143,   2207,  1291 },
    {  -4224,  -4943,   1846,  1792 },
    {  -7128,  -6950,    539,   724 },
    {  -4369,  -4901,   2590,  1103 },
    {  -7413,  -5696,   1712,  1440 },
    {  -5885,  -6821,    418,   871 },
    {  -6828,  -5599,    710, -1563 },
    {  -6123,  -5817,   1358,  1631 },
    {  -5291,  -5622,    578,  2138 },
    {  -7171,  -6004,    347,  2208 },
    {  -6083,  -5251,   2132,   425 },
    {  -4329,  -5721,    407, -2993 },
    {  -5326,  -5056,   1119, -1837 },
    {  -5485,  -5856,    185, -2389 },
    {  -6529,  -5178,    403,  -697 },
    {  -6719,  -4412,   2726,   871 },
    {  -5126,  -5629,   1835,  -771 },
    {  -5622,  -4361,   2973,   858 },
    {  -5282,  -5895,     45,  -335 },
    {  -4357,  -5656,   1696, -1558 },
    {  -7139,  -6659,    627,  -409 },
    {  -4415,  -6328,     35,  1306 },
    {  -7639,  -6110,   1134,   197 },
    {  -3626,  -5592,   2019,   901 },
    {  -3547,  -5064,   1176,  1738 },
    {  -5075,  -3899,   2087,   266 },
    {  -4086,  -6311,   1479,   360 },
    {  -6210,  -5220,   -199, -1477 },
    {  -3910,  -5063,   1356,   -15 },
    {  -7616,  -4977,    461,  2401 },
    {  -6118,  -6131,   1258,  -563 },
    {  -6127,  -4968,   1286,   -27 },
    {  -4121,  -5852,   1113,  1476 },
    {  -5157,  -4881,   1162,  -662 },
    {  -4637,  -5031,   1179,   709 },
    {  -5509,  -5452,   -397,  1224 },
    {  -4597,  -6861,    646,   467 },
    {  -6247,  -4043,    468,   278 },
    {  -5336,  -6465,    874, -1472 },
    {  -6998,  -6346,     78, -1798 },
    {  -4915,  -4530,   2756,  -203 },
    {  -6048,  -4373,   1468,  1052 },
    {  -4273,  -7100,    942,  -323 },
    {  -6552,  -4287,   2351,    69 },
    {  -6954,  -4613,    722,  1521 },
    {  -4201,  -5361,    763, -1562 },
    {  -6881,  -5596,   -748,   669 },
    {  -6695,  -3547,    -34,  1299 },
    {  -3981,  -5728,     84,   111 },
    {  -4663,  -4809,   2173, -1031 },
    {  -6599,  -6077,   1303,   256 },
    {  -7596,  -4265,  -5791, -4140 },
    {  -6610,  -2758,  -5288, -3936 },
    {  -5880,  -3865,  -6563, -3088 },
    {  -7228,  -5510,  -7677, -3912 },
    {  -8854,  -6553,  -8318, -5361 },
    {  -9362,  -5249,  -6413, -4319 },
    {  -4418,  -3110,  -6368, -4358 },
    {  -5544,  -4203,  -6863, -5013 },
    {  -3056,  -4316,  -5567, -3181 },
    {  -3078,  -5999,  -5051, -2657 },
    {  -5884,  -6292,  -5756, -4013 },
    {  -4825,  -4549,  -5535, -4053 },
    {  -4443,  -6126,  -5316, -1368 },
    {  -3972,  -6341,  -6098, -2686 },
    {  -5751,  -2781,  -5398, -6230 },
    {  -4466,  -6135,  -5570, -3679 },
    {  -4291,  -5992,  -3564, -5189 },
    {  -7189,  -4429,  -7279, -6082 },
    {  -5076,  -4433,  -2748, -5366 },
    {  -6225,  -2825,  -6833, -5663 },
    {  -2989,  -4792,  -3960, -4492 },
    {  -7836,  -7773,  -7722, -5741 },
    {  -6559,  -5703,  -5844, -5589 },
    {  -7612,  -5438,  -4136, -3774 },
    {  -4218,  -4176,  -6591, -2333 },
    {  -4837,  -5063,  -6581,   322 },
    {  -6590,  -5990,  -2980, -3847 },
    {  -5558,  -2971,  -5489, -1932 },
    {  -7001,  -5323,  -4975, -1697 },
    {  -4694,  -2688,  -6904, -3044 },
    {  -8511,  -5379,  -5767, -2549 },
    {  -7548,  -5412,  -6522, -2572 },
    {  -6597,  -4973,  -6423, -1274 },
    {  -6415,  -4022,  -5168, -1072 },
    {  -5528,  -5530,  -7218, -2345 },
    {  -4845,  -4805,  -5943, -1227 },
    {  -6049,  -7150,  -6744, -2161 },
    {  -9061,  -7299,  -8542, -4375 },
    {  -5010,  -5546,  -5416,   -82 },
    {  -4135,  -4205,  -5109, -3373 },
    {  -3311,  -5869,  -4007, -5061 },
    {  -5993,  -6472,  -3962, -4718 },
    {  -2966,  -5832,  -2821, -6305 },
    {  -4851,  -5152,  -2067, -3930 },
    {  -3620,  -4441,  -3362, -5836 },
    {  -4469,  -5221,  -4534, -5592 },
    {  -4022,  -6335,  -4321, -6107 },
    {  -4899,  -4503,  -3084, -3725 },
    {  -4490,  -8276,  -4620, -6236 },
    {  -6591,  -4342,  -7365, -4063 },
    {  -6498,  -5057,  -5553,   485 },
    {  -6060,  -2714,  -7093, -4144 },
    {  -6199,  -7774,  -7094, -4057 },
    {  -7536,  -6424,  -6415, -4265 },
    {  -7439,  -2454,  -6348, -4827 },
    {  -5333,  -7565,  -4417, -4639 },
    {  -4353,  -7103,  -4197, -2689 },
    {  -5229,  -6549,  -5129, -6804 },
    {  -6129,  -7701,  -5236, -4836 },
    {  -6797,  -3983,  -3884, -4406 },
    {  -6624,  -4467,  -4745, -5052 },
    {  -3324,  -7596,  -2720, -6553 },
    {  -5473,  -6284,  -1704, -4511 },
    {  -4131,  -7263,  -3180, -5196 },
    {  -7116,  -5565,  -3469,   685 },
    {  -6002,  -6021,  -3858,   576 },
    {  -3144,  -8203,  -1291,  -434 },
    {  -6096,  -7027,  -4004,  1353 },
    {  -3943,  -7709,  -2344,   -36 },
    {  -4510,  -6767,  -2642,   631 },
    {  -3657, -11541,  -2570, -3984 },
    {  -5959,  -8854,  -1333,  -867 },
    {  -6699,  -8866,  -1606,  -344 },
    {  -3836,  -7961,  -2334, -2028 },
    {  -3430,  -8045,  -3037,  -672 },
    {  -3868,  -9184,  -3635, -1819 },
    {  -4258,  -9060,  -2621, -1008 },
    {  -3595,  -8693,  -2022,  -752 },
    {  -4573,  -8048,  -3166, -2622 },
    {  -4852,  -7903,  -1405,   256 },
    {  -4591,  -7057,  -1560,   965 },
    {  -6963,  -7655,   -980,   808 },
    {  -5179,  -6641,  -3356,  1196 },
    {  -7102,  -6941,  -2798,  2123 },
    {  -6867,  -5834,  -3320,  -770 },
    {  -5977,  -7369,  -2500,  -778 },
    {  -6160,  -6400,   -934, -2543 },
    {  -6741,  -7608,   -355, -1289 },
    {  -6856,  -6466,  -1433, -1643 },
    {  -4786,  -6292,  -4970,   376 },
    {  -5407,  -8866,  -2255,  -400 },
    {  -3814,  -6506,  -1387, -3620 },
    {  -4998,  -6137,  -1200, -4092 },
    {  -5123,  -9557,  -2849, -1306 },
    {  -4259,  -6444,  -4395,  -338 },
    {  -5221,  -6810,   -883,  1225 },
    {  -6137,  -6215,  -2165,   554 },
    {  -3895,  -6557,  -3176, -1829 },
    {  -3886,  -8188,    -87,  -954 },
    {  -7243,  -6707,  -2216,  -316 },
    {  -5592,  -7606,     85,  -432 },
    {  -3957,  -7945,   -504,  -144 },
    {  -4617,  -7624,    218,  -312 },
    {  -4797,  -8737,   -844, -1051 },
    {  -4478,  -8516,  -1401,  -454 },
    {  -4557,  -7058,   -302, -2332 },
    {  -6623,  -7736,   -271,   -50 },
    {  -3157,  -7532,  -1111, -2207 },
    {  -3590,  -7300,  -1271,   517 },
    {  -4442,  -7306,   -507,   590 },
    {  -6458,  -7524,  -2807,   666 },
    {  -4991,  -8466,  -3363,  -785 },
    {  -7474,  -7541,  -1056, -1839 },
    {  -7501,  -8316,   -938,  -180 },
    {  -5329,  -7739,   -579, -2341 },
    {  -4549,  -7063,   -176, -3539 },
    {  -5191,  -8612,  -1504, -4250 },
    {  -3083,  -7058,  -2251,    32 },
    {  -4003,  -7043,  -1093,  -791 },
    {  -5523,  -8093,   -678,  -114 },
    {  -3022, -10265,  -2070, -3109 },
    {  -3905,  -6274,   -182, -3652 },
    {  -3269,  -9217,   -551, -2650 },
    {  -3138,  -9314,  -1726, -1704 },
    {  -4420, -10339,  -1744, -3459 },
    {  -4163,  -8609,  -2298, -4113 },
    {  -5566,  -6505,  -1241,  -463 },
    {  -3130,  -9746,  -2352, -4884 },
    {  -7825,  -3439,   1451, -1468 },
    {  -8451,  -3318,   2360,  -435 },
    {  -8462,  -4130,   1438, -1024 },
    {  -9425,  -4564,   1328,  -689 },
    { -11014,  -3202,   2278,  2080 },
    {  -8269,  -2761,   -146,  -440 },
    {  -7497,  -2618,   -166,   413 },
    {  -8250,  -3060,    522, -2133 },
    {  -8365,  -5366,   1347,  -451 },
    {  -8589,  -3979,   2943,   714 },
    {  -8111,  -2572,   1272, -1748 },
    {  -7830,  -5193,    605, -1484 },
    {  -8119,  -4736,   2141,   256 },
    {  -7724,  -4769,   1463,  -812 },
    {  -7363,  -3911,   2540,     4 },
    {  -7974,  -3397,   2363,  1366 },
    {  -7359,  -4204,   1752,  -958 },
    {  -7622,  -3505,    660,   916 },
    {  -9934,  -3665,   3165,   828 },
    {  -8721,  -4162,     62,  1718 },
    {  -9433,  -4768,   2722,  1234 },
    {  -7960,  -4496,    138,  1528 },
    {  -8198,  -3454,   -443,   631 },
    {  -7756,  -2246,    655,  1137 },
    {  -8841,  -3145,   1113,   829 },
    {  -7817,  -3298,   1251,   230 },
    {  -9413,  -2733,    323, -1862 },
    {  -9408,  -4168,   1270,  1549 },
    {  -9037,  -3892,   -942,   283 },
    {  -8255,  -3849,   1301,  1762 },
    {  -9057,  -3987,    -41,  -682 },
    {  -9441,  -4187,   2019,  -111 },
    {  -9740,  -3178,   1602,  -871 },
    {  -8344,  -2474,   1461,  1506 },
    {  -9752,  -2925,   1996,  1243 },
    {  -9199,  -3796,    180,   537 },
    {  -9060,  -2405,   1140, -1562 },
    {  -9348,  -2376,    309,  -162 },
    { -10786,  -3182,     -5, -1500 },
    {  -8142,  -4540,   -434,  -826 },
    {  -7528,  -2341,   1104,   -73 },
    {  -9360,  -2658,   3062,    56 },
    {  -8267,  -2335,   2000, -1193 },
    { -12169,  -3154,   1287,  -640 },
    { -11398,  -2120,    946, -1163 },
    {  -8940,  -4559,    328, -1696 },
    { -11025,  -4213,   2813,   840 },
    {  -9224,  -3581,   2224,  2039 },
    {  -8943,  -3337,   1248, -1298 },
    {  -7900,  -4042,    485, -2080 },
    {  -9221,  -1947,   2191,  -880 },
    { -10762,  -1800,   2516,  -324 },
    { -10095,  -2238,    981, -1335 },
    { -11908,  -2808,   3255,   645 },
    { -10640,  -4105,   1283,  -595 },
    {  -7663,  -2863,   2467,  -797 },
    { -10712,  -3854,   3710,  1538 },
    { -10823,  -2893,   1408,  -801 },
    {  -9874,  -3832,    256, -1638 },
    { -10394,  -3391,   2315,   -94 },
    { -11525,  -4079,   4153,  2122 },
    {  -9546,  -2088,   1541,   481 },
    {  -8731,  -2433,   1042,  2160 },
    {  -7852,  -3977,  -1370,  1677 },
    {   7072,  -3420,   1398, -1741 },
    {   6180,  -1976,   1280, -3557 },
    {   7692,  -1793,   2844, -1700 },
    {   8363,  -1773,   3104, -2679 },
    {   9213,  -3266,   3756, -3542 },
    {   9650,  -2644,   1426, -1318 },
    {   7712,  -2796,   3686, -1975 },
    {   7316,  -3517,   2821,  -622 },
    {   7434,  -2594,   2305, -2264 },
    {   7237,  -1797,    255, -3114 },
    {   8663,  -1983,   1338, -3056 },
    {   6616,   -952,   4059, -2652 },
    {   8823,  -1327,   1362, -1356 },
    {   9938,  -1722,   1287, -2362 },
    {   7207,  -1057,   1913, -1315 },
    {   7508,  -1585,    870, -1982 },
    {   8217,  -3680,   1417, -3170 },
    {   8329,  -2541,   1684,  -585 },
    {   8062,  -2335,    252, -2800 },
    {   8204,  -4108,   3097, -2569 },
    {   7701,  -3367,    576, -3008 },
    {   7350,   -786,   2414, -2129 },
    {   6948,  -2568,   1607,  -225 },
    {   7684,  -2387,   1308, -3449 },
    {   8306,  -3458,   2394, -1454 },
    {   8438,  -2781,   1043, -1362 },
    {   9175,  -2076,   2144, -1987 },
    {   8347,  -2709,   3489, -4301 },
    {   5696,  -2377,   2870,   851 },
    {   8825,  -1243,   2219, -2603 },
    {   8801,  -1614,    584, -2513 },
    {   8413,   -384,   1421, -2244 },
    {   9228,  -3050,   3279, -2164 },
    {   6342,  -2698,   3547,  -107 },
    {  10053,  -2476,   2837, -3168 },
    {   7439,   -604,   3177, -3991 },
    {   7749,  -1064,   4329, -4855 },
    {   8655,  -2177,   2252, -3519 },
    {   8490,   -228,   1958, -3233 },
    {  10513,  -2968,   1911, -2340 },
    {   8146,   -862,   1884, -1723 },
    {   7788,   -666,   3004, -2891 },
    {   7785,  -1620,   4133, -3417 },
    {  10262,  -3731,   3455, -2971 },
    {   8570,   -905,   4519, -4649 },
    {   9129,  -2562,    463, -2465 },
    {   9451,  -3587,   1904, -3056 },
    {   6549,  -2236,   3010, -4523 },
    {   7175,  -2684,   2967, -3458 },
    {   9872,  -3278,   1054, -2472 },
    {   9153,   -931,   1217, -2565 },
    {   8789,  -3469,    753, -2568 },
    {   6683,  -3791,   1797, -3968 },
    {   6801,  -1977,   2311,  -452 },
    {   6336,  -1572,   2612, -3264 },
    {   7996,  -1008,    730, -2964 },
    {   7521,  -1059,   1573, -3694 },
    {   8148,  -3973,   2600, -3572 },
    {   7765,  -1532,   2528, -3856 },
    {   7404,  -3918,   4472,  -143 },
    {   8894,  -1398,   3299, -3685 },
    {   5768,  -2041,   1487,  -637 },
    {   5131,  -2865,   2463,  -811 },
    {   6439,  -1568,   3500, -1550 },
    {  -8878,  -6798,  -5319, -1452 },
    {  -6332,  -9713,  -3112,  -990 },
    {  -8444,  -6316,  -3694,  -687 },
    {  -6123, -10840,  -3637, -4358 },
    {  -4784,  -9580,  -4577, -2581 },
    {  -6108, -10515,  -4859, -2524 },
    {  -7605,  -7518,  -2327, -2797 },
    {  -9662,  -8775,  -2467, -2010 },
    {  -6494,  -7523,  -4715,  -118 },
    {  -8290,  -8982,  -1672,  -317 },
    {  -8798, -11051,  -3888, -1426 },
    {  -6273,  -6623,  -6791,  -142 },
    {  -8313,  -7668,  -2141, -1275 },
    {  -6453,  -8412,  -3589, -4102 },
    {  -6747,  -7750,  -5690, -2498 },
    {  -7814,  -6693,  -3174, -2446 },
    { -10383, -10130,  -3931, -2364 },
    { -10606,  -8467,  -5539, -2772 },
    {  -9475,  -6671,  -3305, -2271 },
    {  -8982,  -9457,  -5635, -4005 },
    { -10111,  -7965,  -6515, -4180 },
    {  -7301,  -6479,  -5364,   720 },
    {  -9543,  -8999,  -7921,  -912 },
    {  -9534,  -8562,  -3469,  -384 },
    {  -7601, -10344,  -3205, -1127 },
    {  -8088,  -8620,  -4954, -2888 },
    {  -8202,  -8406,  -7038, -3775 },
    {  -7312,  -8324,  -3334, -1775 },
    {  -8566,  -9262,  -8071, -4174 },
    {  -7068, -11300,  -5573, -2907 },
    {  -8295,  -8952,  -4366, -1544 },
    { -11104, -10210,  -2285,  -384 },
    {  -5213,  -7520,  -5008, -1339 },
    {  -5889,  -7940,  -5987, -1385 },
    { -10816,  -8201,  -4153, -1485 },
    { -10277,  -8919,  -6315, -1652 },
    {  -5888, -10320,  -3821, -1733 },
    { -10497,  -7181,  -6083, -3032 },
    {  -7721,  -9724,  -6591, -5336 },
    {  -5688,  -7894,  -3486, -2552 },
    { -10014, -10500,  -3247,  -820 },
    {  -6301,  -8765,  -4506, -2923 },
    {  -8261,  -7847,  -6213, -1552 },
    { -10212,  -7481,  -8113, -3954 },
    {  -6938, -10874,  -6074, -4703 },
    {  -7183, -10968,  -4446, -1773 },
    {  -7120,  -9193,  -1966, -2509 },
    {  -6234,  -9263,  -2313, -4284 },
    {  -8503,  -9857,  -2429,  -608 },
    {  -9372,  -7844,  -8391, -2120 },
    {  -7951,  -7157,  -6535,   -11 },
    {  -7256,  -9473,  -2172,  -660 },
    { -10063,  -9612,  -2515,   -15 },
    {  -6684,  -9134,  -6109, -4206 },
    {  -8204, -11932,  -5220, -2306 },
    {  -9710,  -6706,  -4115, -3275 },
    {  -6855,  -7078,  -2409, -4447 },
    {  -7344,  -7673,  -4479, -4116 },
    {  -8851,  -6842,  -4927, -2948 },
    {  -8927, -10452,  -5633, -2194 },
    {  -8627,  -9002,  -7176, -1575 },
    {  -8209,  -9722,  -7021, -3324 },
    {  -3770, -10249,  -3623, -4816 },
    {  -8183,  -7465,  -4090,   646 },
    {  -8163,  -7149,    200,   498 },
    {  -8289,  -6266,    686,  -206 },
    { -10030,  -6241,  -1032, -1864 },
    {  -8793,  -8327,   -773,  -169 },
    {  -9149,  -6215,    969,   -15 },
    {  -8303,  -5859,     -7,  2006 },
    {  -9682,  -7283,    255,  1322 },
    {  -9293,  -7227,     71,  -231 },
    {  -8525,  -6215,    287,  -837 },
    { -10477,  -5379,   1159,  1449 },
    { -10726,  -7856,   -130,   102 },
    {  -8694,  -7461,  -1210,   690 },
    {  -9367,  -5324,   1103,  3170 },
    { -10686,  -8055,   -831,  1633 },
    {  -9201,  -6873,  -2704,  2258 },
    {  -8421,  -5358,  -1405,   226 },
    {  -9066,  -5830,   -307, -1571 },
    { -11150,  -7381,  -2746,  -900 },
    {  -9978,  -5925,  -2006,  -437 },
    {  -9464,  -4741,   -273,  1061 },
    { -10543,  -6684,  -1113,  1660 },
    { -10073,  -5576,   1083,  -269 },
    {  -8826,  -5763,   1600,  1486 },
    { -10445,  -9071,  -1253,   -64 },
    { -12085,  -5799,      2,   769 },
    { -12939,  -6663,   1650,  1437 },
    { -10932,  -6434,  -1252,  -649 },
    { -11650,  -7826,  -2053,   710 },
    { -12122,  -6733,  -1889,  -731 },
    {  -9093,  -6095,  -2463,  -842 },
    { -10977,  -4364,    469,   420 },
    { -11488,  -6908,   -521,   893 },
    {  -9669,  -5478,   -842,   337 },
    { -10606,  -5203,   -632, -1361 },
    { -10198,  -6284,   1662,  1277 },
    { -10135,  -5292,   2435,  3493 },
    { -11027,  -6561,    655,    56 },
    { -10977,  -5030,   1127,  -358 },
    { -12766,  -3986,   1348,  -335 },
    { -14244,  -7731,    264,   317 },
    { -15124, -10309,   -508,  1447 },
    { -12821,  -8638,   -608,   137 },
    { -13076,  -8693,  -2852,  -431 },
    { -11156,  -5546,  -2252, -1600 },
    {  -8692,  -7366,   -819, -1223 },
    { -12507,  -9816,  -1714,  -121 },
    { -10712,  -6666,    544,  3349 },
    { -12462,  -5890,  -2491, -2318 },
    { -12468,  -7226,    437,   232 },
    { -11300,  -5226,   2068,   687 },
    { -11994,  -8320,   -626,  2728 },
    { -12222,  -5476,   1142,    18 },
    { -10277,  -8122,  -2418,  2003 },
    { -13418,  -6115,  -3563, -2802 },
    { -14759,  -9834,  -1243,    21 },
    { -13699,  -5665,   1525,   507 },
    { -16269,  -9476,   -701,   163 },
    { -12677,  -5437,   -247, -1019 },
    { -11827,  -4295,   -181, -1243 },
    { -12847,  -4496,   2984,  1123 },
    { -13860,  -7915,  -1166,  -547 },
    { -12276,  -8145,  -2290, -1527 },
    { -11417,  -4830,   2983,  1854 },
    { -11793,  -6002,   1163,  1940 },
    {  11443,  -4920,  -3235,  3151 },
    {  11300,  -6616,  -1506,  1175 },
    {   9198,  -4628,  -2060,  2390 },
    {  10532,  -4027,   -643,   912 },
    {   9902,  -3573,  -1606,  1327 },
    {   9653,  -3536,  -2240,  1869 },
    {   9948,  -5171,   -423,  2662 },
    {  12316,  -4004,  -1989,   281 },
    {  12125,  -4800,  -1265,  -163 },
    {  10650,  -2617,  -2337,  1462 },
    {   9909,  -4968,  -2376,   916 },
    {  12944,  -4647,  -1958,   460 },
    {  12988,  -5283,  -1141,    41 },
    {  12321,  -2915,  -3621,  1025 },
    {  11449,  -2894,  -2728,   351 },
    {  12087,  -3041,  -2002,   -32 },
    {  11558,  -4031,  -1343,  -399 },
    {  12983,  -3740,  -3516,  1245 },
    {  12099,  -2515,  -2752,   225 },
    {  12515,  -3465,  -2701,   550 },
    {  14683,  -5022,  -5272,  2996 },
    {  12260,  -3383,  -1215,  -528 },
    {  13810,  -5422,  -2443,  1166 },
    {  13421,  -5378,  -1886,   721 },
    {  12961,  -4259,  -2594,   796 },
    {  12266,  -2104,  -4768,  1591 },
    {  13523,  -4710,  -3045,  1342 },
    {  12437,  -2099,  -5610,  2117 },
    {  11850,  -2183,  -3497,   661 },
    {  12275,  -3936,   -597,  -697 },
    {  12459,  -5253,   -517,  -544 },
    {  12835,  -4094,  -1322,  -168 },
    {  14360,  -5677,  -3305,  1859 },
    {  13905,  -4552,  -4309,  2117 },
    {  11559,  -3412,  -1847,   -81 },
    {  13379,  -3167,  -5764,  2746 },
    {  11910,  -1634,  -4342,  1052 },
    {  12662,  -4742,     71,  -974 },
    {  13057,  -3254,  -4424,  1705 },
    {  15046,  -5706,  -4851,  3019 },
    {  14162,  -4142,  -5514,  2843 },
    {  12764,  -1845,  -6684,  2888 },
    {  13714,  -2374,  -7838,  3857 },
    {  13295,  -1663,  -8293,  4073 },
    {  10032,  -4152,  -3403,  1421 },
    {  10942,  -5386,  -2222,   950 },
    {  10532,  -6385,  -1750,  1925 },
    {  10273,  -5972,  -1534,   643 },
    {  10605,  -4782,  -1695,    27 },
    {  10988,  -5153,  -1123,  -341 },
    {  11629,  -5884,  -1060,    48 },
    {  10441,  -4045,  -2431,   311 },
    {  10788,  -3595,  -4171,  1807 },
    {  12110,  -5686,  -2127,   976 },
    {  11746,  -4773,  -2639,   891 },
    {  11541,  -5299,  -3031,  1732 },
    {  11416,  -2559,  -5359,  2198 },
    {  11583,  -5376,   -704,   677 },
    {  10416,  -3214,  -3516,   872 },
    {   9651,  -5435,  -1618,  3255 },
    {   9973,  -5133,   -996,  3923 },
    {  11707,  -4643,   -430,  -796 },
    {  10994,  -2709,  -3587,  2302 },
    {  10716,  -5118,   -645,   270 },
    {  14100, -10314,   1095,  1531 },
    {  12944,  -8049,   1105,  -741 },
    {  13276,  -7035,   -511,   274 },
    {  14008,  -7254,   -283,   139 },
    {  11594,  -6536,    -91,  1671 },
    {  11732,  -8645,    746,    15 },
    {  14613,  -7085,  -1578,  1183 },
    {  13083,  -6224,   -750,    -4 },
    {  13988,  -6256,  -1592,   820 },
    {  14678,  -8683,    441,   126 },
    {  15571,  -8872,   -521,  1139 },
    {  15642,  -9533,    341,   697 },
    {  15960,  -9586,   -168,  1121 },
    {  15464, -10239,   1433,    -1 },
    {  14934,  -7887,  -1046,  1080 },
    {  15252,  -7630,  -1899,  1628 },
    {  15485,  -8384,  -1234,  1484 },
    {  15962,  -8638,  -1815,  1931 },
    {  16501, -10664,    398,  1167 },
    {  16146, -10145,    411,   918 },
    {  14573,  -7475,   -697,   601 },
    {  14302,  -7996,     28,   257 },
    {  14769,  -6792,  -2286,  1574 },
    {  14144,  -6137,  -2169,  1257 },
    {  14770,  -6271,  -3111,  1933 },
    {  14110,  -8312,   1083,  -531 },
    {  15235,  -6991,  -2993,  2174 },
    {  13222,  -5805,    547,  -891 },
    {  14796,  -8762,   1254,  -246 },
    {  16040,  -9181,  -1005,  1551 },
    {  16487, -10086,   -373,  1420 },
    {  15077,  -9479,    966,    51 },
    {  13026,  -6468,    932, -1080 },
    {  12703,  -6152,    -33,  -573 },
    {  15641,  -6810,  -4128,  2874 },
    {  13282,  -7673,   1583, -1283 },
    {  12373,  -7150,   1512,  -917 },
    {  12992,  -7751,   -678,   783 },
    {  10907,  -6858,   -313,  2597 },
    {  13026,  -8963,    125,  2152 },
    {  12770,  -9946,   1957,  -505 },
    {  12482,  -6849,  -1268,   833 },
    {  13790,  -6181,   -138,  -279 },
    {  12709,  -8382,   2044,   227 },
    {  12244,  -6630,    203,  -457 },
    {  14209,  -6816,  -1032,   632 },
    {  15134,  -8267,   -288,   640 },
    {  13619,  -6157,  -1090,   356 },
    {  14044,  -7413,    725,  -484 },
    {  12958,  -7753,   2585, -1980 },
    {  13188,  -8396,   2306, -1558 },
    {  14379,  -9980,   2132,  -688 },
    {  14275,  -9857,   1162,   179 },
    {  13690,  -8648,   1621,  -889 },
    {  11770,  -6829,   -746,   278 },
    {  12732,  -8202,    286,    90 },
    {  13630, -10146,   1867,  -207 },
    {  12072,  -8740,   1299,  -645 },
    {  12852,  -9492,   1226,    62 },
    {  11792,  -7382,    -54,  -116 },
    {  13779,  -9014,    487,   351 },
    {  11951,  -7729,    121,   834 },
    {  11970,  -9781,   2276,    -4 },
    {  12680,  -7984,   2787,  -787 },
    {  13300, -14488,   6408, -1927 },
    {  13635, -15355,   9153, -3073 },
    {  12804, -13566,   5517, -1625 },
    {  16624, -10854,   1690,    28 },
    {  20387, -18532,   6162,  -261 },
    {  16515, -12642,   3392,  -519 },
    {  15800, -11095,   2151,  -202 },
    {  16824, -11790,   1651,   599 },
    {  17604, -13213,   2563,   538 },
    {  17892, -14177,   3562,   147 },
    {  16987, -11399,    869,  1052 },
    {  17003, -12456,   2442,   265 },
    {  21657, -21806,   9198, -1250 },
    {  16825, -13341,   3980,  -686 },
    {  17525, -12714,   1887,   805 },
    {  16419, -11034,   1216,   617 },
    {  20931, -19939,   7469,  -684 },
    {  18452, -15390,   4573,  -191 },
    {  14778, -10077,   2841, -1209 },
    {  17402, -13319,   3042,   160 },
    {  19365, -17922,   7087, -1061 },
    {  16298, -11941,   2810,  -351 },
    {  19087, -16176,   4775,   -84 },
    {  17666, -12289,    938,  1224 },
    {  18581, -15894,   5132,  -430 },
    {  19823, -16717,   4142,   545 },
    {  19960, -19423,   8400, -1492 },
    {  18973, -16817,   5906,  -594 },
    {  19079, -15431,   3528,   503 },
    {  16667, -12485,   4467, -1302 },
    {  19791, -17797,   6196,  -529 },
    {  20005, -17606,   5354,   -20 },
    {  20123, -18599,   6886,  -728 },
    {  19068, -14805,   2394,  1105 },
    {  14443, -13723,   5631, -2029 },
    {  14730, -14231,   5631, -1450 },
    {  16089, -15959,   7271, -2029 },
    {  13473, -11200,   3236,  -924 },
    {  14413, -10902,   2347,  -267 },
    {  17666, -18662,  11381, -3496 },
    {  14749, -11042,   3305,  -275 },
    {  15304, -10486,   1869,  -240 },
    {  14809, -12126,   3369,  -616 },
    {  16896, -16561,   7307, -1845 },
    {  15782, -14336,   5380, -1264 },
    {  16395, -15520,   6415, -1588 },
    {  13681, -11114,   2584,  -320 },
    {  14244, -12326,   4480, -1632 },
    {  15247, -13119,   4265,  -898 },
    {  13987, -12091,   3469,  -597 },
    {  13941, -12770,   4240,  -839 },
    {  13771, -13627,   5252, -1384 },
    {  15010, -16074,   7592, -2249 },
    {  15852, -17226,   8619, -2655 },
    {  18921, -16916,   6875, -1501 },
    {  14909, -11678,   2768,  -295 },
    {  18988, -18353,   8424, -2070 },
    {  15457, -15080,   6218, -1513 },
    {  14916, -15512,   6949, -1883 },
    {  18108, -14702,   4681,  -701 },
    {  17600, -15733,   5616,  -775 },
    {  14070, -13683,   6472, -2626 },
    {  13832, -11914,   5201, -2232 },
    {  18846, -19009,   9192, -1961 },
    { -11981, -10994,  -6324, -2264 },
    { -10976,  -9047,  -6546, -3828 },
    { -11288, -10532,  -7014, -4191 },
    { -10139, -10189,  -7799, -2688 },
    { -10555,  -9988,  -9181, -2040 },
    { -11596, -11339, -10022, -2707 },
    { -13400, -13395, -11306, -4206 },
    {  -9774, -12281,  -7466, -4133 },
    { -10842, -13125,  -8777, -4956 },
    { -11964, -15082,  -9779, -5095 },
    {  -9382, -10188,  -9053, -4927 },
    { -11562, -11296,  -3651,  -985 },
    {  -9287, -10083,  -7918, -4069 },
    { -12821, -16556, -11410, -6195 },
    { -12628,  -8959,  -4521, -1113 },
    { -13845, -11581,  -3649,  -681 },
    { -12685, -10269,  -5483, -1275 },
    { -14988, -12874,  -5107, -1189 },
    { -13761, -11367,  -6202, -1804 },
    { -13225, -11249,  -7820, -3354 },
    { -14809, -11992,  -3202,  -312 },
    { -15620, -15519, -10210, -3433 },
    { -12954, -10200,  -3139,  -611 },
    { -11536,  -9981,  -5284,  -923 },
    { -13034, -12417,  -4612, -1098 },
    { -16911, -15505,  -6123, -1352 },
    { -17396, -17685,  -8330, -2171 },
    { -14120, -10764,  -2265,   -99 },
    { -12598,  -7367,  -5406, -3530 },
    { -14143, -12793, -10909, -5226 },
    { -14692, -16871, -11626, -5554 },
    { -12581, -11197,  -9194, -3837 },
    { -16752, -16726,  -9746, -2808 },
    { -10600, -10358,  -6560, -1227 },
    { -14573, -13312,  -8957, -3393 },
    { -10172,  -8463,  -8579, -3387 },
    { -11418, -12421,  -5522, -1842 },
    { -11855, -14204,  -6669, -2625 },
    { -13308,  -8191,  -3941, -2194 },
    { -10007, -12266,  -5022, -1811 },
    { -13532, -15771,  -9497, -3175 },
    { -11760, -11148, -10339, -5529 },
    { -12149, -12763, -11198, -3697 },
    { -12029, -12119,  -8555, -1792 },
    { -16995, -19957, -11447, -3471 },
    { -13144, -14504,  -9988, -3191 },
    {  -9938, -11064,  -6139, -3162 },
    {  -8873, -11550,  -8294, -6550 },
    {  -9303, -13010,  -6150, -2711 },
    { -15463, -10469,  -1766,  -170 },
    { -15985, -11693,  -3007,  -650 },
    { -17142, -10671,  -1434,    47 },
    { -16063, -13858,  -4817, -1058 },
    { -19446, -19599,  -9594, -2464 },
    { -20076, -18744,  -8313, -1889 },
    { -15047, -16085,  -7590, -2250 },
    { -13481, -16195,  -8552, -2998 },
    { -13829, -14869,  -6704, -1932 },
    { -16357, -18484,  -9802, -2959 },
    { -10551,  -8393,  -9303, -5070 },
    { -11345,  -9156,  -5641, -3107 },
    { -13217, -13449,  -9270, -4541 },
    { -11988, -13732,  -9995, -6374 },
    { -11007,  -9519,  -5168, -4107 },
    {   9930,  -7858,   8061, -4375 },
    {   8274,  -7867,   5992, -2096 },
    {   9692,  -9675,   7621, -3670 },
    {   9589,  -8110,   6509, -3010 },
    {  12617, -11976,  10122, -5360 },
    {  11867,  -8895,   7948, -5323 },
    {  10388, -10482,   9234, -4324 },
    {   8188,  -8220,   7810, -2737 },
    {  10407,  -8787,   4806, -1930 },
    {  10348,  -8845,   9233, -6614 },
    {   9422,  -7091,   4820, -2878 },
    {   9758,  -9796,   5584, -2256 },
    {  10188,  -7994,   5347, -3343 },
    {  11133,  -7455,   4015, -2306 },
    {  10676, -10744,   6093, -2629 },
    {  11522, -12184,   7848, -3375 },
    {   8805,  -9883,   5317, -3071 },
    {   9498,  -9654,   6555, -3592 },
    {  10488,  -8008,   4066, -1252 },
    {  11261,  -8930,   6068, -2738 },
    {  12180, -10397,   5027, -1531 },
    {   9138,  -8531,   3601, -1959 },
    {   8107,  -8380,   4970, -2061 },
    {   9737, -13248,   6438, -2617 },
    {  11178, -10423,   2622,  -522 },
    {   9572, -12372,   5199, -2019 },
    {  12057, -12144,   4147, -1099 },
    {   9047,  -9925,   2516,  -665 },
    {  10790,  -8030,   5882, -4386 },
    {   7199,  -8426,   6337, -2841 },
    {   7778,  -8285,   3529, -3442 },
    {   7559, -10569,   3484, -1332 },
    {   9404,  -8115,   7484, -5541 },
    {   7792, -11976,   5546, -2573 },
    {   9313, -10264,   7661, -5195 },
    {   6701, -10725,   4370, -1784 },
    {   4918, -11361,   4507, -4527 },
    {   5147, -12305,   3978, -5556 },
    {   6525,  -9899,   4481, -3129 },
    {   7538, -12855,   6060, -4826 },
    {   8659, -12111,   7159, -4430 },
    {   8440, -11304,   4547, -1747 },
    {   9216, -10918,   3507, -1195 },
    {   6165,  -9254,   4771, -4677 },
    {   9163, -11019,   5637, -4935 },
    {  13441, -11509,   6676, -2434 },
    {   7912,  -9398,   6663, -4048 },
    {  11723, -13745,   8131, -4148 },
    {   6065, -10257,   5005, -6327 },
    {  11618, -12417,   5336, -1894 },
    {   8891, -13924,   8407, -6131 },
    {   9622, -12563,   7908, -5109 },
    {  11479, -10315,   8349, -3991 },
    {  11676, -14103,   6611, -2330 },
    {  11951,  -8953,   3829, -1550 },
    {  10486,  -8044,  10493, -5920 },
    {  11801, -10769,   9763, -5305 },
    {   6109,  -8676,   5827, -1346 },
    {   7030,  -9611,   5624, -5761 },
    {  12808, -12886,   8683, -4148 },
    {  13213, -10464,   6381, -3189 },
    {  11796, -13681,  10703, -6075 },
    {   9639,  -7949,   9625, -3944 },
    {   8538,  -6997,   5309,   453 }
};

/* quantization tables */

const uint32_t ff_dca_scale_factor_quant6[64] = {
          1,       2,       2,       3,       3,       4,       6,       7,
         10,      12,      16,      20,      26,      34,      44,      56,
         72,      93,     120,     155,     200,     257,     331,     427,
        550,     708,     912,    1175,    1514,    1950,    2512,    3236,
       4169,    5370,    6918,    8913,   11482,   14791,   19055,   24547,
      31623,   40738,   52481,   67608,   87096,  112202,  144544,  186209,
     239883,  309030,  398107,  512861,  660693,  851138, 1096478, 1412538,
    1819701, 2344229, 3019952, 3890451, 5011872, 6456542, 8317638,       0
};

const uint32_t ff_dca_scale_factor_quant7[128] = {
          1,       1,       2,       2,       2,       2,       3,       3,
          3,       4,       4,       5,       6,       7,       7,       8,
         10,      11,      12,      14,      16,      18,      20,      23,
         26,      30,      34,      38,      44,      50,      56,      64,
         72,      82,      93,     106,     120,     136,     155,     176,
        200,     226,     257,     292,     331,     376,     427,     484,
        550,     624,     708,     804,     912,    1035,    1175,    1334,
       1514,    1718,    1950,    2213,    2512,    2851,    3236,    3673,
       4169,    4732,    5370,    6095,    6918,    7852,    8913,   10116,
      11482,   13032,   14791,   16788,   19055,   21627,   24547,   27861,
      31623,   35892,   40738,   46238,   52481,   59566,   67608,   76736,
      87096,   98855,  112202,  127350,  144544,  164059,  186209,  211349,
     239883,  272270,  309030,  350752,  398107,  451856,  512861,  582103,
     660693,  749894,  851138,  966051, 1096478, 1244515, 1412538, 1603245,
    1819701, 2065380, 2344229, 2660725, 3019952, 3427678, 3890451, 4415704,
    5011872, 5688529, 6456542, 7328245, 8317638,       0,       0,       0
};

const uint32_t ff_dca_joint_scale_factors[129] = {
       3288,    3490,    3691,    3909,    4144,    4387,    4647,    4924,
       5218,    5528,    5855,    6199,    6568,    6963,    7374,    7810,
       8271,    8758,    9278,    9831,   10410,   11031,   11685,   12373,
      13103,   13883,   14705,   15578,   16500,   17482,   18514,   19613,
      20770,   22003,   23312,   24688,   26156,   27699,   29343,   31080,
      32925,   34871,   36943,   39133,   41448,   43906,   46506,   49258,
      52177,   55273,   58544,   62017,   65691,   69584,   73711,   78073,
      82703,   87602,   92795,   98289,  104111,  110285,  116820,  123740,
     131072,  138840,  147069,  155776,  165012,  174785,  185145,  196117,
     207735,  220042,  233086,  246894,  261523,  277017,  293434,  310823,
     329236,  348748,  369409,  391303,  414490,  439043,  465064,  492621,
     521805,  552725,  585475,  620170,  656920,  695843,  737073,  780745,
     827008,  876014,  927923,  982902, 1041144, 1102834, 1168181, 1237404,
    1310720, 1388382, 1470649, 1557790, 1650098, 1747876, 1851441, 1961147,
    2077355, 2200441, 2330825, 2468935, 2615232, 2770195, 2934335, 3108206,
    3292378, 3487463, 3694108, 3913000, 4144862, 4390455, 4650611, 4926176,
    5218066
};

const uint32_t ff_dca_scale_factor_adj[4] = {
    4194304, 4718592, 5242880, 6029312
};

const uint32_t ff_dca_quant_levels[32] = {
          1,       3,       5,     7,      9,     13,     17,      25,
         32,      64,     128,   256,    512,   1024,   2048,    4096,
       8192,   16384,   32768, 65536, 131072, 262144, 524288, 1048576,
    2097152, 4194304, 8388608,     0,      0,      0,      0,       0
};

/* 20 bits unsigned fractional binary codes */
const uint32_t ff_dca_lossy_quant[32] = {
         0, 6710886, 4194304, 3355443, 2474639, 2097152, 1761608, 1426063,
    796918,  461373,  251658,  146801,   79692,   46137,   27263,   16777,
     10486,    5872,    3355,    1887,    1258,     713,     336,     168,
        84,      42,      21,       0,       0,       0,       0,       0
};

/* 20 bits unsigned fractional binary codes */
const uint32_t ff_dca_lossless_quant[32] = {
         0, 4194304, 2097152, 1384120, 1048576, 696254, 524288, 348127,
    262144,  131072,   65431,   33026,   16450,   8208,   4100,   2049,
      1024,     512,     256,     128,      64,     32,     16,      8,
         4,       2,       1,       0,       0,      0,      0,      0
};

/* Vector quantization tables */

DECLARE_ALIGNED(8, const int8_t, ff_dca_high_freq_vq)[1024][32] = {
    {  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
       0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0 },
    { -4, -2,  2,  1,-16,-10,  1,  3,  1,  0,  6,  1, -3,  7,  1,-22,
       2, -4, -3, 11, 14,  6, -1,  1,-13, 29,-28, 10, 10, -8,  0, -9 },
    { -8,  8, -7, 10, -3,-12, -5, -8,  1, -2,  9, -2, -5,-18,  1,  9,
      -8, -8,  3, 41,  7, -9, -9, 22,-42,-29, 14,-18,-14,-32,  1,-15 },
    {-16,  8, 15, 16,-16,  5,  2,  7, -6,-16, -7,  1,  1, -3, -2,  0,
       8, 20,-26,-11,  2,-17,  0, -3,-34,-37, 10, 44, -2, 22,  2, -4 },
    {  7, 14,  5,  6, 15, -1,  3, -3, -9,-23, -5,-14,  8, -1,-14, -6,
      -5, -8, 54, 31, -6, 18,  2,-19, -2,-11,-30, -6,-19,  2, -2,-14 },
    {  1,  2, -2, -1, -3, -3,  1, -5,  1, -3, -4, -8,  5, -4,  0,  1,
       3,  7, -5, -4, -3,-12,  3, -2, -3, 12,-53,-51,  6, -1,  6,  8 },
    {  0, -1,  5,  1, -6, -8,  7,  5,-18, -4, -1,  1,  0, -3, -3,-14,
      -1, -6,  0,-14, -1, -1,  5, -3,-11,  1,-20, 10,  2, 19, -2, -2 },
    {  2,  4,  3,  0,  5,  0,  3,  1, -2,  0, -6, -3, -4, -5, -3, -3,
      -7,  0,-34,  4,-43, 17,  0,-53,-13, -7, 24, 14,  5,-18,  9,-20 },
    {  1,  0, -3,  2,  3, -5, -2,  7,-21,  5,-25, 23, 11,-28,  2,  1,
     -11,  9, 13, -6,-12,  5,  7,  2,  4,-11, -6, -1,  8,  0,  1, -2 },
    {  2, -4, -6, -4,  0, -5,-29, 13, -6,-22, -3,-43, 12,-41,  5, 24,
      18, -9,-36, -6,  4, -7, -4, 13,  4,-15, -1, -5,  1,  2, -5,  4 },
    {  0, -1, 13, -6, -5,  1,  0, -3,  1, -5, 19,-22, 31,-27,  4,-15,
      -6, 15,  9,-13,  1, -9, 10,-17,  4, -1, -1,  4,  2,  0, -3, -5 },
    { -7,  3, -8, 13, 19,-12,  8,-19, -3, -2,-24, 31, 14,  0,  7,-13,
     -18,  0,  3,  6, 13, -2,  1,-12,-21,  9, -2, 30, 21,-14,  2,-14 },
    { -3, -7,  8, -1, -2, -9,  6,  1, -7,  7, 13,  3, -1,-10, 30,  4,
     -10, 12,  5,  6,-13, -7, -4, -2, -2,  7, -3, -6,  3,  4,  1,  2 },
    { -8,  9,  2, -3, -5,  2,  0,  9,  3,  7, -4,-16,-13,  3, 23,-27,
      18, 46,-38,  6,  4, 43, -1,  0,  8, -7, -4, -1, 11, -7,  6, -3 },
    {  1,  1, 18, -8, -6,  0,  3,  4, 22, -3, -4, -2, -4,-11, 40, -7,
      -3,-13,-14, -7,-10, 14,  7,  5,-14, 11, -5,  7, 21, -2,  9, -3 },
    {  0,  0, -2,  4, -2,  0,  2,  0, -1,  2, -1,  0,  0,  2,  2,  2,
      -1,  1, -3, -1,-15, -2,-63,-27,-21,-47,-14,  1,-14, 10,  0,  2 },
    {  1,  0, -4,  0, -3, -9,  4,  2,  6, -6,  0, -5, 11, -7,-15,  6,
      -7, -6,  3,  7,-15, -5, 23,-13, -6, 12, -8,  9,  2, -3,  3,  4 },
    {  6,  0,  3,  0, -2, -4,  2,  1,  1, -1,  1, -2, -1, -4,-22,-15,
     -46,-66, 10, 20,  2,-17, 12, -6,  1, -2, -2,  0,  1, -5,  1,  2 },
    { -1,  0,  0,  1,  0, -4,  0,  1,-10, -3, -8,  5,  7,-11,  2,-11,
      29,-25, 11, 10,  0, -1,  5, -7, -2, -5, -2,  4,  4, -3,  5, -2 },
    {  1, -1, -1, -3, -2,  1, -8, -3,  2, -2,  4, -5, -1, -7, -2,  1,
     -14, -7,  3,-30,-15,-14,  3, -4, -1,  3,-13, -1, -3,  1,  2,  3 },
    { -1, -2, -3,  2,  2, -3,  3,  1, -3,  2,  0, -4,  6,  5, -5, 10,
     -57,  3, 22,-50,  1, -2, -5, -6, -1,  5,  1,  2,  2,  1, -2,  2 },
    {  2,  0, -1, -7,  2,  1,  3,  2,  0,  4,  3, -2,  3, -3,  4, -4,
      24,-35, -3, 38, -6, -5, 15, 20,  3, 16, -7, -5,  0, -4, -5,  0 },
    {  0,  1,  0,  0,  0, -1, -1,  1,  1, -1,  1, -2,  0,  0,  0,  0,
       0, -1, -2, -1, -5, -2,-43, -3, 46,-52,-10,  7, -8, 11, -2, -1 },
    {  0,  0, -1,  0, -1,  2,-41, 33,-44,-48,-15,-26, -9,  6,  3,  3,
      -3,  2,  2,  2,  2, -1, -1, -2,  1,  3,  0,  0,  5,  2,  3,  1 },
    { -4,  1,  6,  1, -6, -1, -2,  1,-14, -4,  0, -5, -2,  2, -2,  0,
      -6,  1,  0,  8,-21, 32, -3,-36, -6, -2, -1, -7,  3,  0,  1, -6 },
    { -3, -2,  3,  0,  2,  2,  8, -4, -4,  6,  2,  1,  3, -6,  4,  3,
      13,  0,-12, -1, 25,-20, -2,-23,-15,  7, -3,-11, -3,  6, -1,  0 },
    {  0,  0, -3, -1,  0,  0, -2, -1, -2, -2,  1, -1,  0,  0, 10,  3,
      -2,  3,  3, -7, -6, -5,  0, -4,-60,-16, -6, 38,  5,  6, -5,  0 },
    {  0,  1,  0,  0,  0,  0,  0,  0,  1, -1, -1,  0,  1,  0,  0,  1,
       0,  0, -1,  0, -8,  2, -9, 10, 40, 31,-56,-21,  4, 20, -4,  7 },
    { -2, -2,  0,  4, -3, -1,  7,  3,  1,  3, -8,  0,  3,  1,  2,  5,
       1, -2, 14,  5,  4,  5,  5,  5, -5,  9,-66,  0,-20, -2, -8,  4 },
    { -2, -1,  4, -1, -8, -2, -4, -1, -3, -3,  2, -7, -3,  5,  7, -2,
      45, 31,-17,-16, -2, -2, -1,-22,  1, -1, -3,  3,  5, -3,  5, -1 },
    { -4,  0,  7,  5,  8,  7,  2,  9, -9, -9, -7,-11, -3, -8, 17, -4,
      34, 32, 18, 22,  1,  2,  1, -7, -5,  6, -1,  6,  4, 10, -2, -7 },
    {  6,  0, 14,  9,  6, -1, -2, -3,  4, -6, -8,  4,  7, -1, 28, 38,
      15, -1, 16,-11,  5,  8,  4,-10,  3,-10,-17,  5,  3,  3,  3,  1 },
    {  1,  1,  2, -1,  2,  1,  0,  0, -1,  0,  0, -2,  1, -3,  0,  1,
       2, -2, -4, -2,  0, -1,  1, -3,  1,  1,  1, -1,  8,  8, 66, 33 },
    { -5,  2, -3, -7,  2, -8, -4, 10, 17,-18, -7,  4, -4, -7, -6, -6,
      -5,  5,-12,  2,  0,  6,  8, -2,  1,  4,-11,  2,  1,  8, 31, 19 },
    {  6,  9, 16, -6, -6, -1, -2, -3,-11, -2,  7,  7, 17,  3,  4, 10,
       2,  5,-13,  8,  7,  1,  4,  5,  7,  6,  7, -8,  9, -8, 33,  6 },
    {  3, -1,  1,  0, -7, -5,  0, 14, -7,  1, -7,  1,  2, -4,  7, 10,
     -16, 12,  1, -6,  3,  8, -1, 10,-13, -6,-12,-23, 12, -3, 30, 14 },
    { -2,-15,  0,  8,  3,-19,  5, -3,  2,  3, 13,  7, 14, -3,-10,  0,
       8,  5, -6,-16, -8, -8, 14,  2, -1,  1, -9,-11, 11, -5, 27,  9 },
    { -8,  6, -4,  4, -4, -1,  5,  4,  1, -7, -5, -4,-15,  1,  9,  0,
       8,  4,  1,-17, 11, -2,-19, -1, -6, -8,  3,-12,  3,-17, 33,-10 },
    { -3, -1,  2,  7,  7, -2,  9,  8,-18, -1,-13,-10, -3, -3, 11,  8,
      -2,-12, -8,  1,  4,  9, 14, 10, -3,  0,  2,  1, -2,  3, 31, 10 },
    { -3,-10,  8, -1, -5,-11,  7, -5,  3,  6,  1,  4,-16, 10,  5, -4,
      -2,-10, -1, 13,  6, -5, -7, 12,  7, -3,-17,  1, 12, -4, 29,  8 },
    {  1,  2,  5,  2, -6, -7,  0, -1,  6, -1, 10,  6, -4,  5,  2,  2,
      -2, -8, -6,-11, 14,-13, 27,  3, -2,-12,  5,-16,  2,-26, 20, 15 },
    { -1, -3, -5, -3, -3,  6, -1,  3, -5,  1,  7,  2,  1,  0, -1, -1,
       0, -1,  9,  7, -6, -3,  4, -5, -4,  8, -8,-25, -8, -4, 34, 23 },
    { -1, -2,  1,  1, -1, -2, -1,  1, -1,  0,  0,  0,  0, -2, -1,  1,
       0,  2,  1, -1,  4,  0,  0,  1, -1,  0,  5,  3, 12, -9, 68,-16 },
    { 10,  0, -8, 14, -6,  1,-12,  0,  0, -3, -5,-11, -6, 12,  9,-10,
      -3,  5,  0,  7, 11,  2,  4, -3, -8, -3,  7,  4,  3, -3, 34,  4 },
    {-12, 13, -5,  7,-11, -2, -1,  1, -4,-14,-21,  3, -3, -3, -4, -7,
      -9, -4,  3,-17, -2,-13, 10, -2, 12, -4,  0, -9,  1, -5, 31, 10 },
    {-10,  6,  5,  6,  4, -7, 10,  0,-28, -3,  0,-11, -1, -5, 16,-10,
     -16,  7, 20,  2, -4,  2, -5,  0, 15,  6,  5,-10,  7, -9, 20,  4 },
    {  1, -7, -2, -7,  4, -3, -2, -7, -1,-14,  6,-16,  4, -5, -4, -6,
      -5,  0, -2,  2, -6,  9, -5,  4,-18,  8,-10,  8, 15,  0, 32,  1 },
    { -5,  7, -3,  7, 15, -4,  0,-16,  9,  5, -5,  5,  4, -3,-12, -9,
     -18, 10,  2,  2, -3,  7,  3, -1,  6, -9,-10,  3, 15, -4, 35, -7 },
    { -1,-10,  2,  2, -4, -2, 10,  2, -1,  2, -2,  1, -1,-14,-11,  3,
      -8,  5, -8, -2,  6, -1, -7,  1,  7,  5,  7,  8, 30, -4, 30, 14 },
    {  2, -2,  1,  2,  3, -8,  3,  0, -2,  0, -9,  2,  1,  4, -6, -1,
      -2,  5,  0,  1, -2, 12,  6, -3,  9, -3,  4,-12, 21,-39, 24, -2 },
    {  3,  5,  1, -2, -2, -2, -3,  6, -8, -2,-11, -8, -1,  4,  2,  2,
      -4,-10, 12, -5,-11,  1,-15,-34,-11, -7,-11, -1,  7,-14, 38, -1 },
    { -4,  4,  8,  9,  8,  1, -5, -9,  4, -2, 15, -4, 11,-15, 20, -1,
      -1, -3,  4, -9, -2, -2, -2,  8,  6, 12, -5,  0, 11,-12, 27, -4 },
    {  0,  8, -4,  3,-11,  6,-11,  2,  3,  0,  5, -8, -7, -6, -9,-21,
       4,-11, -1,-16, -7, 16, -3,  7, -7,  4, -5,  0, 11, -7, 31,  3 },
    {  1,  3,  4, 11,-11, -2, -3, -6,  6,  5,  0,  3, -9, -6,  4, -4,
       0,  4, -8, 13, -6,-13, -1, -5, -1,  4,  0,  0,  9,-22, 24, 18 },
    { -7,  3, 10,-13, -6,  6, -6,  6, 22,  1,  0,-14,  2,  3,  7, -1,
       8, 20, -1,  5, -4, 13,  9, -9, -9,  6,  0, -4,  0, -8, 31, -4 },
    { -3, -4,  0,  1,  7,  3, -7,  0,  5, -2,  1,  3,  3,  1, -5, -2,
       5,  2,-11,  4,  0, -1, 12,  0, -3,-13, 15,  8, -6,-27, 34,  0 },
    { -3, -3, 10, -4,  2, -1, -3,  0, -1, -1, -4,  2,  6, -2, 12,  1,
       3, -6, -7, -6, -5,  4,-19, -6, -8,-34, -4, -8, 10, -7, 23, 10 },
    { -7,  0, -1, -6,  8,  4, -4,  2, -5, -8, -7, -9, -8,  5,  9,  7,
      -6,  1,-12,-12, -1,-16,  5,  0, 16,  3, -7, -8, 27, -4, 23, 15 },
    { -8,  4,  8,  5,  6, 11, -3,  5,  3, -1,-11,  6, -5,  0,  2, -6,
      -3, -6,  4, -1,  5, -5,-12, -6,  7, -5,  9,  3,  6, -7, 29,  1 },
    {  1,  3, -2, -2, -6, -2,  1,  6, -6, -3,  1,  2,  3,  4,  1,  5,
      -1,  0,  4,  2, 11,  6,  2, -3, 13, -9,-19, 18,-15,-10, 36, 21 },
    { -3, -3,  2, -1, -7,  6, -4,  1, -3, -1, -2,  2,  3, -7, -3,  0,
      -2,  0, -2,  6,-19,  3, -8,  2, -6,  7, -1,  0, 29, -6, 28,-10 },
    { -5,  1, -3, -7,-12, -4,  1,  1, -1, 13,-10, -1, -9, -5,-13,  6,
      13,  3, -4,  2,  3, 11,  2,  6,-25,-16, -6,  0, 14, -1, 27, 16 },
    { -6, -1, -7, -5, -2, -5, -5, -1,  9,  1,  0,  3, -8,-12, -6,  5,
      -6,  5,  3, -9,  1,  4, -7,-10, -9, -7,-17, -5,-15,-23, 25,  3 },
    { -8, -2,  9, -3, -4,  3, -1,  8, -7, -7, -5, -4, -2,  9,  4, -1,
      -7, -4, -5,-16,  3, -6, 18,-13, -9, 16,-15,  8, 15,-10, 24,  5 },
    {  1,-38,  2, 34,  9, 10, 11,  2,  2, -6,  3,  2, -2,  5,  4, -7,
      -1,  1,  4,  0,  3,  1, -8, -1, -6,  5,  4,  2, -4,  5,  2, -1 },
    {  1,-22, 15, 18, -2, 10,-16, -9, -8,-11,  8,  4,  0,  7,-14, -5,
      -1, -7, 12, 17,  9,  5, -7, -4,-12, -6,  7,  0,  7,  2, -2,  1 },
    {-11,-29,  7, 10, 19, -1, -8, -9,  7,  1,  9,  6,  8, -7,-14,  8,
      -3,-11,-13,  0, -7,-23, -2, -8, 12,  9,  2, 14, 19,  1, -1,  5 },
    {-24,-27,-11, 36,  2,  6, -3,  4, -6,  8,  0, 12, -1, -4, -6,  3,
       4, -1,  2, -3, -2,  3,  2, -1, -2, -4,  0, -1, -2,  7,  2,  3 },
    { -9,-24, 11, 13,-10,-12, 12, -2,  7,  4,  8, 13, -3, -3,  2,  9,
      -3, -4,  4, 13,  5, 13, -6, -3,  1, 15,  7, -3,  0, 19, -2, -9 },
    { -8,-15,  7, 14, -4, -5,  2,-18,-19, -2,  2, 17, 16,  6,-10, 10,
      -9, 14, -1, -5, -1, -6, -7,  2,  9, 11, 13,  6, -5,-12,  3,  2 },
    {-10,-37, 13,  1,  3,-14,  0,-20,  4, -3,  8,  2, -2, -3, -9, -5,
      -3,-17, -1, 13,-11,  2, -6,  4,  4,  0,  3,  1, -9, -4, -5, -4 },
    { -2,-22, -5, 46, -8,  5,  9,-11,  8,  7,  7, -1, -1, -2, -7,  2,
      -3,  3, -1, -2,  7,  0,  2, -1,  1, -2, -2, -3,  6,  0, -4, -6 },
    {-16,-27, 15, 16, -4, 14, -7,-26,  2, -2,  6,  5, -3, 11,  0,  2,
       3,  9, -7, -1,  2, -4, -4, -1,  6, 10,  1,  1, -3, -2,  3,  0 },
    { -3,-22, 10, 26,  1,  2, -3,  3, 17, -3, -7,  9,  1,-21, -4,  5,
       3,  0, -7, -6,  3,  3, -8, -7, -9,  3,  7,  1, -8, 12,  6, -7 },
    { -9,-25,  3, 18,  9, -6,-11,  0, -5,-12,  9, -8, -7, -6, -6, 22,
       2, -6, -3, 15,  3,  2, -2,  9, 14,-10, -7, 15, 13,  6, -2, 11 },
    {  5,-20, -5, 28, 11, 10, -4, -4,  0, -7,  3,  5,  2, -5, -8,  2,
       6, 10,  9, -9,-18,  3, 14,  1,  3, -3, -1, -6,  7,  7,  2, -1 },
    { -8,-30,  7, 12, 10,  8,  7,-13,-16,  0,  1, -1, -6,-11,-15,  4,
       1, -2, 10,-15,  1, 11, -2,  8,  9, -7, -7,  9, -5,  2,  7,-18 },
    {-10,-32, 10, 11,  3, -1,  3, -5,  5,  2, 14, -6,  3,  1,  5,-15,
     -11,  6, 20,  4,  0,-12, -7,  3,  1, -1, 10,  6, -1, -9, -4, -1 },
    {  1,-25,-14, 12,-11,  9,  9,-16,-24,-17, 22, -9, 11,-30, -3, -4,
       6, -7,  9,  2, -1, -5, -6,  2, -1, -1, 10,  1, -3,  3,  4,  8 },
    {-14,-26, -6,  9,  8, 17,-11,-24, -7, -4, -8, -2, 10,  2,  2, -1,
       2, 13, 12, -7,  4, -6,-10,  6,  6,-13,-11, -7,-16,  0, -2,  5 },
    { -4,-30,-13, 12, 16, -6, 12,-16,-13,  5, 15, -2, -2,-10, -7,  7,
      11, -1, -4, -2, -4,  7,  4, -8,  1,  3,  0, 11,  3, -2, -5,  4 },
    { -4,-21, 20, 22,  2, 20, -8,  1,-12, -5, -9,  4,-10,-17, -3, -8,
      -3,  3,-12,  1, -3,  0,  7,  4,  7,  7, -3,  7,  5,  3,  1, -5 },
    {-12,-20,  2, 29, 11, -6,  9, -7, -6, -4,  0,  6, 17,-13, -2,-10,
     -17, -1,-18,  2,  0, 14, -6,  1,  0,  3,  2,-10,  1, -5, -2,  5 },
    { 16,-37, -1, 26, -2,-14,  1, -5,-14,  2,  2,  3,  6,  1,  1,  4,
       0, -1,  0, -2, -2,  4,  9, -6,  0, -2, 10, -7, -2,  4,  1,  0 },
    { -9,-24,-12,  5,  5,  3,-17,-14,  4,  3,  2, -4, 10,-22, -8, -3,
       6,  1, 12, -8,  4,  1,  9, -1, 18, -3,  6,  5,  3, -5,  9, -5 },
    {-14,-33, -2, 20,-13,-10,  2, -7, -1, 11, -9, -8, 18, -3,  1,  8,
       0, -2, 10,  7, -2,-13,  9, -3, -4,  5, -2, -2, -1, -5,  1, -7 },
    {-10,-23,  8, 14,  1,  7,  1, -3, -7,  4,  1,  1,  8, -7, 15,-14,
      13, 14,  2,  5,-13, -5, -8, -1,  6,  3,  6,  9,  6, 15, 14,  5 },
    {-13,-25,-10, 13,-17,-24, -7,-13, -6,-10, -8,  2,  0,-13,-10, -4,
      -8,  4, -9,  9, -4,  4, -3, -3,  3,  3, -5, -9,  1, -2, 11,  2 },
    {-12,-23,  1, 18,-11, -2,  5,  9, -5,  5, 14, -9, -3, -2, -6,  2,
      -2, 11,-13,  1, -3, 11, -9, -4, -2, -6,  8, 10,  1,  4,  2,  1 },
    { -5,-18, 16, 22,  2,  0,  8, -6, -9, -7, 10,-16, 23, 10,-11, -1,
       7,  2,  7,  2,  1, -5,  6,  1,  0, -4,  9,  2, -3,  1,  0, -4 },
    { -3,-26, 14, 11,  2, -9, 17, -2, -1, -5,-16, -9, -5, 10,-13,  1,
       6, 12, 10, 11,  0,  0, -3,-14,  6, -2,  0,  4, -5, -1, -7, -1 },
    {-10,-33,  1,  8, 11, -5,  1, -6,  7,  4,  5,  6,  1, -2,-10, -5,
      -6, 12,-11,  5,-10,  4, 12, -1, -1, -3,  4, -1,  9,  0, 16,-17 },
    {-14,-37,  7,  7, -2,  5, -8,-11,  2,-13,  4,-19,  1,  8,  8,  4,
      -9,  2, -4,  3, 12,  2,  4, -4, -8,  8,  1,  4,  8, -1,  6, -2 },
    { -6,-30, 18, 17,  1,-22, -3,  4, -7,-10,  7,  0, -8,  8, -1,  4,
       2,  8,  6, -2,  2,  7,  4,  4,  3, -6,  2,  1, -3,  1, -1, -5 },
    {-17,-18, -3, 22, -8,  1,  9, -2,-17, 20, -5, -5,-12, -5,  4, -5,
      -9,  8, -2, 16, -3,  0, 19, -8,  8,  1,  2, -4,  0, 11,  0, -3 },
    { -9,-23,  3, 10,  4,  4, -3, -2, -2, -2,  1,-22, 11,  0, -2,  5,
      -2, 14, -9,-11, -4,  7,  5, 32,  1, -3, -7,  0, 21, -9,  7, -6 },
    {  0,  0,  0,  2, -1,  1,  0,  1,  3,  0,  0,  1,  0,  1,  0,  1,
      -3,  0, -1, -2,  0, -1, -1, -3, -1,  1, -4,  1, -1, -5,-69,-19 },
    { -3, -5, -8,-12,  4, -3,-19,-11, -5,  0,-14,  7, 18, -6,  7, 22,
       8, 14, 15, 10,  3, -1, -3,  5, -1,  7, -7,  1, -6,  3,-26,-11 },
    { -1, -6,  4, -4, -5,-16,  0, -6, -3, 11,  1,  0,  9,  5, 16,  3,
      -4,-33, -4,  4, -7,  0,  1,  6,-11, -2,-13, -2,-18, 20,-25,-16 },
    {  4,  0, -1,  0, -5,  1,  0,  2,  0, 11,-10,  4,-10,  7, 16,  2,
      16, 15,  2, -1,  2,  9,  2,  8, -3, -5, -2,  0, -3,  0,-33, -2 },
    { -3,-15, 10, 10, -9, -1,  7,  3,  5, -5, -8, -8, -3, 15, -9,  4,
      12, 13,-13,-14, 10, -6,  9, 22,-27, 23, -1,  5,-24,  2,-30,  5 },
    {  0, -2,  7, -5, -5,  3,  5,  3, -3, -5,  2,  1, -4,  3, -3, -1,
       1, -2, 10, 22, -3, -4, -2, -2, -7,  3,  8,  1, 14,  4,-37,  9 },
    { -3, -4, -1,  1, -4,  0,  6,  2,  6, -7,-10,-10, -1, -4, 11, -3,
       7, -6,  4,-12, -1,  5,  1, -7, 10, -6, 17, -4,  8,  3,-40, 13 },
    {  2, 12,  4, -7, 14, -3, 16, -2, 18,  2, 13,  5,  5,  1, 11, -1,
       0,  9,  2, -6, -1,  2, -6,  2, -5,  3,  5,  1, -1,  1,-32, -7 },
    {-16, 11,  7, -4,  2, -5, -9,  9, 11, 11, 15,-13,-11, 11,  9,  4,
       3, -8,-10, 12, 12,  0,  0,-16, -9, 13,  2,  9,  4,-13,-33,  3 },
    {  6,  4,  5,  4,  3, -1,  5,  6,  4,  2,-11, -1,-15,-11, -1,  1,
      11, -3, -2, 24, -4, -6,-25,-10,-15, -8,  0,  0, -5,  4,-30,  2 },
    { 10, -3, -6,  1, -9, -5,  6,  9,-10, -3,  8, -1,  4, -1, 11,-11,
       3,  9, 11, -3,  6,-17,  5, -8,-33,  9,-13, 19, -2,  9,-25,  2 },
    {  0,  0, -1, -3,  0, -2,  1,  0,  0,  2,  1,  0, -2,  0, -1,  2,
       0, -1,  4, -1,  2, -3,  4, -2,  3,  3,  1,  0,-15, 12,-63, 27 },
    { -2, 14,  9, -1,  3,  0,  1,  1,-19, 15,  3,  4,  0,-10,  1, -5,
       3,  0, -5,-10,  2,-16, -4,  8,-12, -6,  7, -5,-10, -1,-33, -4 },
    {  0,  3,  1,  3,  1,  2,  4,  4,  9, -6, -8, -5,  1,-12,  3,  8,
     -10,  6, -1,  1, 13, -5, -5,  2, -4, 13,-18,-10, -7, -9,-33, 10 },
    { -6, -3,-12,  5, -1, 11, -6,  0, -2,  1,  2, -7,  3,  1,  3, -2,
       1,  8,-10,  7, -1, -3,  3,  0, 13,  1,  6,  7,-16, -7,-39,  8 },
    { -6, -1, 11,  6, -3,  8,  3, -5,  3,  0, -5, -2, -6, -3, -4,  2,
      -3, 13,-11,  1,  7,  5, 19, -5, -3,-15, -1,  7, -1,  6,-33,  8 },
    { -7,  3, -4, -3, -4,  1,  6, -5, -5,  6, -8, -1, -7,  4, -1, -6,
      -2,  1,  7,  0,  1,  1, -5,  2, -2,  0,-13, -2,-31,-14,-39,-12 },
    {-10,  9,  0, -3,  1, -1, -1,  0,  1, -5, -1, -4, -2,  5,  2, -7,
      18, -8, -2,-19, -7, -7,-12,-14,-11, -1, -9,-13, -7,-12,-31, -9 },
    { -3,-16, 10,  9,  1,-10,-12,  2, -2,  2,  7, -3, -3,  1, -4, -5,
      -9,  5,  7,  3, -1,  4,-11, -8,  4, 13,-10, 13, 10, -4,-36,  1 },
    { -7,-12,  4,-20, -7, -7,  2, 11, -1, -2,  3,-12,  1,  0, -6, -7,
       6,  4, 13,  3, -3,  4,  3, -6,-12,  5, -5,-22,-13, -8,-37, -6 },
    { -7,  5,  3,  5,  7,  9,-14, -3, 10, 17, -1,  1,-12,  5, -6,  0,
      -4, -9,  0,-11,-14,  3, 13,  6,-25, -8,-12,  4,-10, 18,-30, -1 },
    {-10,  6,-10,  6,  6,  1,-10,  0, -7,  5, -2, 17,-18, -4,  0, -3,
     -16, -6, -3, -8,  5,  1, -4,  6, -7, 16,  6, 10, -1,  0,-32,-11 },
    { -1,  9,  9, -5,  4,  9,  6,  9, -4, -2,  7, 11,  4,  2, -5, -4,
      -6,  0,  2, -3, -1,  5, 10,  0, 12,-10,-18, -3, -1, 14,-33,  2 },
    {  4, -8,-18, -4, -5,-11,  4,-10, -4,  9, 13,-12,  1, -6,  1,  2,
       4, -9,  8,  3, -6, 21, 13, -1, -2,  1, -2,  6, -7,  0,-30,  1 },
    {  6, -1,  2, -3, -1, -4,  6, -4,  0,  4,  2,  2, -9,  2,  6,  3,
      -2,  4, -1,  9, -6,  0,  7, -8,  5, 19, -2,  9, -5,  2,-33, -8 },
    {  2,  1, 12, -5, -8,  8,  3, -2, -4,  1, -2,  5, -4, -9, -8, -8,
       7,-11, -4,  6,-10,  7, -1, -1, -2, -1, 16, 32, -7, 20,-33, -6 },
    {-18,  2,  6, 13,  9,  9, -1,  3,-17, 24, -2, -6, 28,  8, -2,  6,
       3,-10,-34,-16,-13, -4,-15,-11,-12, -3,-10,  4, -8,  4,-31, -4 },
    {-11,  0, 18,  2,-16, -9,-13, -2, -2,-12, -3,-22, 30,  0,  8,  3,
       9, -4,-16,  1,  0,-11, 15, -2, -4,  6, -5,  6,  1,  2,-25,-12 },
    { 14, -1,  5,  7,  3,-15, -8,  1,  5, -2, 12, 13, 11,-25,  3,  1,
       0, -2, -4,-16,-23,  0, -5,-17,  7,  5, -9,  6, -5,  2,-32, -7 },
    {  3, -1,  6, 14,  2,-12, -9, -9,  4,  7,  4,  6,  5, -8,  4,  2,
       4,  5, -2,  8,  8, -6,  0, 10,-20, -1,  3, -1,  8, 23,-33, -5 },
    { -3, 11, -6,  3, -4,  5,  7,  3,  4,  5, -2,  3, -1, 30,  6,  1,
       8, -6,  0,  0, -9,  6, -9,  4,  2,  9, -6,  1,-12,  0,-34, 18 },
    {-17, 13,  0,  1,  9, -4,-11,  0,  7,  0,-10, -4, -1,  6, -6,  4,
       1,  6, -9,  3, -5, -6,-11,  2, -4, 14, 23, -3,  2,  5,-30, 12 },
    {-14,  5,-27,  2,  0,  7,  1,  4, 30,  8,  7,  5,  1, -1,  0,  5,
       8,-10, 48,-11, 12, 33,  6,  8,-15, 20, -2, -5, 32,  5,-19, 10 },
    {-16, -4,-12, -7, -2,  0,  8, -6,-20,-18, 16, -3,  0, 31, -2, 11,
       2, -9, 49,-19,-12,-23, 10, 26, 16, -2,  4,-21,-14, 13,-11, -9 },
    { -5, -9, -1,  3, -5,-21,  2, 10,  0,  0, 10,-21, -7,  7,-26, -9,
      22, 32, 58, 11, -3, 11, -5, -8,-13,  6, -5, -9,  1, 10, 14, -8 },
    {  7,  7, 10,  3, -2, -1,-11,-11, -6,-43, -3, 14,-19,-18, 19, 18,
     -32, 10, 45, -6,  6, 21,-20,-12,  2,  4,  6,  6, -4,  3,  3,  1 },
    { 21, 22, -3, -2,-11, -6, -1, -2,  8,  8, 32,-21,  7, 28, -4, -6,
      -3, -2, 50,  2,  2, 27, -5, -8, 12,  7, -5, -1, -4,-17, 27,  6 },
    { 13,  7,  2, -6,-12,  2,-10, -5,-17, 11,  4, 17,-12, -2,  5,-17,
      37,-16, 48,-14,-18, 29,  8, 24, 11, -5, -9, 11, -1,  1,-13, -3 },
    {  1,  1, -1,  2,  0,  0,  0, -1,  1, -1,  7,  2, -3,  3,  0,  6,
       2, 10, 54,-25,  7, 54, -5, -6, -1,-15,  9, 13,-24,-15,-12,  3 },
    { 21,  5,  8,  3, -3, -4, -2, -4,  3,-11, -5, -8,  9, 16,  8, -9,
     -10, -3, 46,-46,  2,  1,-10, 10, 17, 11,-20,-36, 10, 14,  0, -5 },
    {  7,-13, -6, -9,-24, 45,  2,  8,  8,  0, 17, 20, 12,-24,  1, -7,
     -15, -3, 46,-13, -2, 20,  1,-13,-11,-13,  2, 15,  1, 10, -1,  3 },
    {  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1,  0, -2, -1,
     -16, -9, 31,-69,-34, 26,  7, 17, -1, -6, -1,  0,  0,  0,  0,  0 },
    {  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -4,
      -5,-20, 18,-82, 22,  3, -7,  9,  4,  6,  2, -4, -1,  0, -2,  2 },
    {  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  0,  4, -1,
      15, -5, 62,-36,  4, 52, -7,  5,  0,  6,  1,  2,  1,  1, -1,  0 },
    {  3,-19, 19,-20, 13, -4,-11,  8,  8,-16, 10,  1,-14, 30,  1,-33,
      10,-11, 45,-30,  3, -4, -3,-13,  7, 12,  3,-22,  3, -2, -4, -2 },
    { -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  2,  1,
      11,  8, 70, 48,-10, 21,  4,  9, -9, -9, -4, -6,  0, -1,  0,  0 },
    {  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  2,  0,
       2, -1, 80,  2,-15,-36,-10, -5, -2,  8, -2,  2,  0,  0,  0,  0 },
    { 10,  8, -8, -8,-24, 12, -1,  0, 20,  9, -1, -2,  2, -2, 12,-10,
      -2,-13, 35,-43, 44, 15,-10,-25,  4, 10, -3, -5, -5,  7, -1,  3 },
    {  1,  1,  1,  1,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0, -2, -1,
     -18,  9, 49,-72,  7, -8,  7, -5,  2,  3,  2, -2,  1, -2, -3,  1 },
    { -1,  4, -3, 10, 19,  4,  3, 20,  6,-24,  6,  9,  8, 15, 18, 18,
     -36, 19, 57,-11,  4, -3,  8,  7,  2, -3, -2, -9,-15, -2, 12, -4 },
    { 20,  3, 11, -9, -4, 22, 42,-25,  1,  5,-10,-19,  0,  9,-16,  5,
       2, 10, 44,-29, 17, -3, -9, -2, -1,  8, 14, -7, -1, 16, -5,  1 },
    { -7, 16,-11, 12,  6, 33,-15, 14,-23,  2,-26,  8,  2, 10,  0, -5,
       8, -8, 38,-38, -4,  5,  5,  5,  1, 22,-15,  7,  6,  0,  4, 28 },
    { -1,-12,  2, 10, -2,  0,  7, 17, 12, 22, -4, 10, 25, 29,  5, 18,
       4,  1, 27,-39, 31, 17,  2,  2, 22,-23, 13, 16,  1, -7, -4, -5 },
    {  0,  1,  0,  0,  0,  0,  0,  0,  0,  0, -1,  0, -2,  0,-14,  0,
      -7,-11, 49,-22, -4, 19, 17,-39,  4,-29, 10,  2, 36, -4, 23, -1 },
    { -2, -2, -2, -2,  1, 15, -5, -7,-16, -8,-19, 16, -3,-20, 36, -9,
      -3, 20, 39,-20,  0,  2, 27,-16, 10, 10,-14,-22,-16, -3, 13, -8 },
    {  5, -9,  6,-25,  7, 37, 13,-10, -5,  3, -5,  7, 18,-22, -7,  9,
      -5, -4, 50,-11, -4, -5, -5,  8, -4, -2, -4,-27, 14, 20,  7, -9 },
    {  0,-14,-10,-27,-14,-17, -6, 26, 10,  2, 14,-12, -5,  0,  8,  9,
       0,-28, 55, -7,-12, -7,  4,-10, 10,  7,-12, 11,  3,  5,  9, -8 },
    {  2, 23,  4, -2, -1,-20, -2, 14, 10, -9, -9,-24, 10,  0, 11,-12,
      12, 11, 49,-25, -2, 29,  7,-13, 21,-10, 11,-17,  3,  1, -8,  5 },
    {  3,  0,-14, -6, 18, -2, 17, -9,-19,  9, -5,  9, 14,  6, 19, -3,
      27,  1, 41,-21, 20,-15, 33,  0, 26, 14,  7, 10,  3, 20, -3,-12 },
    { -1, 16, 15, -8,  3, -8, -8, 21, -5,-16,-29,  4,  1, -6, -4,-28,
       2, 31, 37,-26, -2, 13, 24,  8, -9, -6,-29, 10,  7,  2,  7,  8 },
    {-10,-10, 11, 13,-32,  2, 16,  9, 14, 23,-15,-13, 24, 13,  4,-27,
      14, 12, 31,-18, 17, 23, -2, -7,-14,  9,-17, -6,-10, 20,  9,  6 },
    {  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  2,  0,
       5,  1, 89,  8, 10, -6,  2, -1,  0,  0,  0,  0,  0,  0,  0,  0 },
    {  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  2, -1,
       4, -7, 64,-50,  7, 37,  2,  5,  0,  0,  0,  0,  0,  0,  0,  0 },
    { -2,  5,  3, -4, -4, -3,  2, -3,  3, -3,  5,  4,  1, -6, -1,  1,
       6, -2, 50,-35, -7, 43,  7, -7, -5,-26, 24, 21,  3,-15,  5,  6 },
    { -8, 21,-19, 33, -8, 22,-11, 17,  3,  0,  0, -2,  1, -3,  6, -1,
      10, -8,  4,-11, -4, -5,  0,  8, -4,  3,  1, -4,  4,  2,  8,  4 },
    { -7,  5,-20,  9,-22,  3,-14,  1,  6, 13, 23, -2, -4, -7,  2,  0,
      11,  4,  6,  3, -7,-11, -7,  4,  5,  5,-12,  8,  2,  4,  7, -3 },
    { -7,  6, -4, 20,-20, 16, -2,  7,  6, 16, 11, 12, -7, -7,  5,  3,
      -9, -4,  1,  2,  5,  2,  1, -9, -2,-17, -4,  6,-10,  7, -7, -6 },
    { -9, 18,-17, 12,-24,  1, -1,  4, 14,  9,  4,  3,  2,  8,-12,-14,
       4, -8, -4,  7,  7,  6, -1, 13, -9, -4, -1,  1,  0, -4, 15,  8 },
    {-25,  2,-11,  6, -5, 24,-28, -5,  8, 12, -2,  6,  8, -3,  8, -9,
      -1, -5, -1, -5,  6, -1, -1, -1, -4,  8,-12, -2,-13,  7,  2,  1 },
    {-14, 14,-18, 20,-10, 12, -2,  9,  1,  0, 12, -2, 15,-10, 26,-17,
      16,-11, 10,-10,  9, -2,  4, -8,  2, -3,  4,  4,  2, -3, -5,  1 },
    {-18, 12,-18, 21, -6, 12, -6, 13,-25, 18,  1, 11, -9, -5,  0, 10,
      -5,  3, -3,  8, -9,  7,  4,  2, -9,  0,  5,  0,  2, -3,  9, -8 },
    { -4, 16,  1, 18,-30,  9,  1,  6, -8, 13, 13,-12, -6, -1, 13,  7,
       6,  2,-15, -3,  5,  5,  1, -6,  1, -5,  0,  2,-16,  0,  3, -4 },
    {-21,  1, -2,  6,-43, 18, -1,  5, -1,  4,  6, -2, -1, -3, -1, -3,
       0,  1,  2, -9,  0, -1,  0, -2,  0, -1, -1, -2,  6,  0,  1, -2 },
    {-23, 10,  4,  7,-32,-11,-18,  2, -2, -7, -6, -3, -3,-12, 19,  3,
      -5, -6, 16, -6, 16,  2, 16, 16,  8, -2, 13,  8,-15,-11,  2, 10 },
    { -8,  2,-13,  2,-29, 24,-20, 19,  1, 10, -4, 10,  1,  2, -9, 11,
      -1, -2,  9, -5, 19, -7, 16, -9, -2,-18, 11,  1,  1,  0,  7, -3 },
    { -6,  3,  4, 13,-26, 10,-10, 28, -7, 28,  1,  7,  0,-14,  5,  7,
       4, -4,  3, -2,  3,  3,-11,  7,  6,  4,  0, -1,  2, -1, -3,  2 },
    { -6, 16,-31, 13,-10, 17, -6,  4,-14,  4,  4, -1,-10, 12, -5,  1,
     -14, 15,  0, -8,  1, -5,  3,  3,  9, -5,  7,-20,  7,  4, 11, -5 },
    {-19,  3,-17, 14,-12, 16,-22, 18, 14,  8, -2,  4, 10, 12,-14,  4,
      -3,  2,  3,  7, -7,  7, -6,  2, -2, -4, -5,  0, -5, -2,  2,  1 },
    { -9, -7,-11, 24,-36, -9,-11,  5,  7,-12,-13, 18, -2, 20,  1, -4,
      -1,-10, 15, -6, 14,  1,  0,  2,  1,  2, -9,-16,-11,  7, 13,  0 },
    {-24, 24,-18, 18,-22, 14,-11, 13,-12, 11,-10, 11, -7, 11, -5, -4,
      -1,  1,  5,  2,  3, -1,  1, -5,  7, -4,  5, -6,  8, -7,  8, -6 },
    { -6, 18,-22, 22,  5, 11, -1,  6, 19, 22,  8,  4, -8, 20, -2, 15,
      -6,-18,  0,-33, -9,-12, -1,  6,  5,  2,  5,  5, -5,-17, -3, -3 },
    {  1, 11,-16,  9,-18, 11, -4, 18, 20, 26,-10,  8,  1,-11,  8, -4,
       0,  7,  3,  5,  2,  2, 10, -2, -4,  4, -4, -2,  1, -4, -5, -1 },
    {-10,  6, -1, 18,-17, 27, -3, 10, -2, 12, -7, -9,  1,  1, -1,  7,
     -12, -1, -7, -6, -1,  8,  3,-15,  8,  9,  3, -7,  4, -1,  1, -1 },
    {-14,  6,-16, 22,  2,  5,  0,  5,-18, 11,  6, -3, 22,-20, -9, -3,
       6, -6, -7,-15,  1, 15, -8, 11,  8, -3, -8,  1, -8,  2,  6, -2 },
    {-21,  5,-19, 19, -7,  4, -7,  0, -8,  6, 12,  5, -3,-22,-13, -6,
      -1, -3, -2,-14,  6, -3,  1, -8, -7, -5, -6, 11, -3,-10, -5,  2 },
    { -1,  9,-12, 15, -6,  6,-19, 14, -9, 11,  3, 12,-17, -3,  8, -4,
      -3, -4,  1, -5,  4,  5, -7,-15, -7, 15, -6, -5,  1, -5, -3,  1 },
    {-12, 20,-15, 20,-14,  3,-14,  9, -6, 33,-13,  6, -2,  8, -6,  7,
      -5, -6, -3, -3,  0,  8, -3, -3,  1, -2,  2,  2,  6, -5, -5, -2 },
    { -7, 12,-18, 12,-18, 10, -4,  8,  2,  4,  8,  9,  0,  3, -8,  3,
       6,-12, -4,  1, 25, -5, -9,  6, -7,  0, -9, -7,  3, -5, -4, -4 },
    {-18, 12,-10, 11,-22,  0,-15,  5, -2,  2, -3,  6, -4, -4, -3,-15,
      -2, -3, 21,  6,-12,-11, 19,  3,  3,-14,  7,  0,-11,-22,-10,  0 },
    {-15,  2,-30, 15,-17, 13,-16,  8, -7, 10, -8,  2, 11,  3, 10, -7,
       7,-22, 12,-10,  3,-12,  6,-10, 12,-10,  7, -8,  5,  2,  9,  1 },
    { -9, 11,-14,  6,-10, 21,  5, 12, -5,  5,  7, 21,  6,  2, -2, -1,
      -1,  4,  2,-20,-18, -1,-14,  3, -1,  4, -7, 10,  1, 11,  4, -4 },
    {-22,  8,-30, 13,-21, -4,  4, -1, 12,  9, -2, -3,  2, -6,  4,-13,
      -2,  8,  8,  1, -7,  3, -4, -5, -1, -7, -2,  8,  8,  7,  8,  0 },
    { -6, -4,-35, 16,-13, 15,-11, 14, -7,  9, -1, 11,  7,  0, 13, 10,
      -1,  8,  1,  1, -2,  8, -1,  2,  2,  3,-10, -1,  7,-13, -3, -7 },
    {-15,  7,-16, 14,-18, 17, -6, 14,  3,  4,  7, -3, 10,-22,  5,-15,
       4, -4,-11, 15,-15, 11,-11, 20,  1,  0,  2,  1, 11, -3, 11, -7 },
    {-12,  3,  5, 16,-37, -1, 15, 15,-15, 10,  3,-10,  1, 15,  7,-15,
     -13,  8,  9, -3,  2, 12, -8,  2, -5,  0, -3,  4,  5, -9, -4,  5 },
    {-16, 26, -4, 14,-22, 26,  6, -3, -8,  4, 21,  6, 16, -4,-11,  7,
     -10,  3,  3,  7, -4,  2, -9,  8, -2,  2,  5, -2, -4, -2,  7, -1 },
    { -7,-10,  4,  3,  2, -4,-12,-10, -4, -5, 16, 19,-16,  1,  2, -9,
     -10,  0,  9,  7, -8,  3, 12,  8, -6,-11,-13, -1, -3,-20,  6, -5 },
    {-14,-17,  3, -5, 14,-12,-12,  8, -6,-25, 21, 21, 10, -8,-12,  4,
      10, -4,  3, -9, 11,  9,  0,  4,  2,-15,  1,-14,  4,  1,  0, -4 },
    { -4, -9, -3, -1,  6,  3, -6,  6,-10, -4, 14,  8,  2, -3,-12,-19,
       0, 11,-20,  1,  6, -2,-27, -6, 10,-17,-14,-17, -9,  8, -8,  3 },
    {-12,-13, 16, -4, -2, 12, -7,-11,  2,-13,  3,  7,-16,-18, -1,-12,
      -2,  1,-12, -9, -2, -6,  2,  9,-22, -3, -4,-14, -7,  7, -1,  2 },
    { -7, -8, -8, 15, 15, 18, 15, 16, -4,-37, 11, 15,-12, -1, -3,  3,
       6,  6,  0, -5, -3, -5,  9,  1,  1,-11, -1, -8, -6,  2,  3,  0 },
    { -6,  7, -5,-12, 13, 10,-18, -4, -3,-21,  6, 16,-15, -7,-12, -9,
       1,-12, -1, 10, -2, -1, -3,  4, -4,  1,-16, -1, 12, -9,  5,  9 },
    {-14, -5,  9,  3,  4, 26,-28,  3, -6,-24,  4,  5,  3, 13,  5, -1,
       3, -1,  3,  1,  1, -5,  3,  0, -7, -8, -7, -3,  3, -5,  4,  0 },
    { -4,  2,-10, -6, 25, 26, -6, 10, -6, -8, 15, 11, -6, -3,  2, -7,
       5, 14,  9, -1,  0,-12,  4, -4,-10,  1, -3,  3, -2, -2, -6, -1 },
    {-10,  8,-15,-10, 19, 17, -8,  0, -3, -7,  7,  5,-13, -1,  7, -7,
       1, 13,-12,-13, 17,-12,  1, 26,-18, -3, -5, -6,  4,  5,  8,  1 },
    {  2, -5,  3,  0,  0,  0,  2, -3, -2, -5,  7, 13, -4,  9,  0, -5,
       4, -1,-11, -8, -4,  0,-13,  2,-47,-23, -8,-11, -4,  4, -2, -3 },
    {-18, -4,  4,  5, -1, 17,-12, -8,  1,-12,  7, 20,-12,  3, -2,-11,
      16, 12, -6,  1,-13,-16, -6, -3, -3, -5,  4,-12, -5, -9, 10,  1 },
    {-11,  0,  4,  7,  7,  8,  3, -1,  3,-19, 32,  8,-19, -8,  2,  4,
     -12, 15,-16,  3,  1,  9, -2,  1, -2,  8,  5,  6, -4, -1, 11, -8 },
    {  3, -1,  4, -2, 14, 32, -9,-23,-10,-12, 22, 15, -1, -2, 10,  0,
       4,  6, -8,  4,-15, -2, -1, -4,  0, -8,  4,  1, -8,  3,  4,  1 },
    {-17,-12,  6, -8, 16, 13,-20, -8, -1,-16, 10, 21,-19, 11, -9, -5,
       7, 18, -6,  7, -7,-18, 13,  2, -2,  8,-12, -9,  2,  4, -5, 16 },
    {  4,  0, 17,-11, 12,  7,-12,  5, -1,-25, 30, -8, -7, -6, -4, -7,
       9,  8,  7,  3,  3,-16,  8,  0, -2, -2,-18, -3, -4, -5,  1,  4 },
    { -3, -6,  6,-16, 17,  6, -3,  2, -9,-17, 12, 11, 11,  2,-20,  8,
       1,  1,  0,  2, -2, -6,-21,-13, -9,-15, -1, -8, -6, -8,  0, -2 },
    {-11, -7,  6, -9,  3,  6,  8, 16,  4, -5, 23, 26,-10, -3,  4,  0,
       2,  2, -4,  4, -2,-12, 12, 10,-11,  0,-10,-16,  3,  0,  0,-10 },
    { -5,-16, 10, -6, 27, 13, -3,  4, -2,-13, 15,  5,  2,  5,  3, -4,
      13, 12,-11, -7,  0,  1, 11, 12,  2, 13,-15, -8,  9, -2,  3,  8 },
    { -5, -8,  4,  3,  9,  3,-11, 10, 14,-25, 14,  8, -2,  5,-12,-21,
       2, 10, -7,  2, -3,  2,  0,  2, -1, -3, -5, -6, -1,-16,  2,  8 },
    { -1,  5,  1,-11,  5,  9, -7,  8,-13,-12,  4, 12, -4,  1, -1, -1,
      27, 29, 10, 15,  2, -6, -3,  4,-21, 10, -9,-11, -6, -1, -9, -3 },
    { -6, -3, -1, -6, 11, -5,  0, -2, -5,-31, 11,  3, -1,  5, -3,  4,
       5,  7,-10,  5,-10,-13,  4, 12,-15, -2,  2, -7,  1, -9, -3,-10 },
    { -3, -7, 17, -8, -5, 36,  8, -7, -8,-20, 12,  8,  1, -1,  3,  0,
       1,  4,-10,  3,  1,  4, -2, -3, -2, -3,-10,  4, -1, -7,  3,  2 },
    {-13, -3, -5,  9, 22,  6,-23,  3,-10, -7, 17, 17, 18,-14, -8, -8,
       2,  4, -8,  2, -3, -8,  6,  4, -1,  7,  0,  0, -3,  0,-12, -3 },
    { -3,-10,-15, -3,  9,  3,-23, -9,-13,-18, 12, 13, -2,  0,  1,  8,
      -1,  2, -7,-12, -5, 14,  2,  1,-22,  6,-10, -8, -9, 28, -7,-14 },
    { -3,  1,  2, -1, 13,  7, -2, -7,  1, -3,  6,  9, -3, -2,  4, -2,
       2,  1,-10, -2, -2,-22, -2, -7,-10, -5,-11,-27,-12,-16,  4, -7 },
    {  2, -6, -3,  1,  8,  0, -2, 12, -3, -4, 58, 15,-10, -4, -2,  2,
      -2,  0, -2, -6,  2,  4, -1,  1, -4,  1, -1, -5, -4, -3,  3,  1 },
    { 10, -1,  0,  5, 21,  7,-14,  6, -3,-16, 15, 17,-16, 13,  3, -6,
      -4,  6,-12, -5,  1, -4, -7, -8,  2,  3, -6,  6, -1, -8,  5,  4 },
    { -6, -2, -8,-11, 15, 10,  0,  8, -6,-15, 33,  8, -2, 18,-15,-11,
       5, -1,  0, 15,-15, -4, -4, -1, 10,  7,-13,  4, -4,  0,  8,  3 },
    { -7, -2,  0, -2,  0, -2, -4, -5,-14,-16, 12, 38,  7, 12,  6, -4,
       0, -1,  0,  3, -2, -6,  0,  2, -9,  1,  0, -1,  0, -2,  4,  1 },
    { -8, -4, 18,  1, 14,  5,-12, -3, 20,-17,  5, 19,-11, -8, 11, -3,
       3,  9, -7, -8,  9,-17,  2, 15,-10,-11,  5, -5,  7, 15, -6, -2 },
    { -7,  2, 38,  5, 19, 16, -5,  4,-13,-20,  0,  4, -4,  6,  4,  2,
      -7,  6, -8, -2, -5, -7,  6,  3, -4, -3, -2, -3,  7, -6, -4,  0 },
    {-11,-12,  8,-15, -3, 14, -7,-22,-11,  2, 22, 14,-19,  2,-19, -6,
       1,  3,-18, 14,  2, -6, -2, -8, -3, -6,  5, -7, -8, -4,  1,  1 },
    {  8,  7, 25,-21, 12, -6, -5, -4,-10,  6,  0, 10,  1,-12, 18, -5,
     -15,  4,  1, 14, -1,  5,  8, -7,  1, -7, -3,  9, 10,  1, -1,  0 },
    {  9, 10, 32,-15,  8,  2, 11, -7,-18, -8,  2, -6, -9,-16, -3,  3,
      -1,  3,  1, -5,  4, -2,  1, -8,  0, -6, -3,-11,  1,  5,  0,  0 },
    { 14,  0, 23,-25, 22,  3,  7, 10,  0, -2,  7,  8,  0, 10,  0,  0,
       3,  2,  3,-10,  0, 10,  0, -7,  0, 10, -1, -5, -7,  1, -1,  2 },
    { 12,  0, 25,-18, -5, -4, 13,-10,  3, -6,  7, 21,  0,-16,  3,-10,
      -6,  5, -7, -3,  2,  5,  3, -6,  4,  9, -8, 12, -2,  3,  2,  4 },
    { 31, 15, 27,-20, 10, -7, 15,-10,  9, -8,  4, -5,  3, -3,  5,  6,
      11, -2,-12, -2,  6, -2,  1,  2, -1, -1,  1,  1,  3,  1,  1,  2 },
    { 12, -4, 13,-23, 12, -6,  2,  4, -3, 13,  6, -7,  5,-19, -7, 18,
       1, -7,  7,  1, 16, -7,  3,  0,  3,  0,-12,  8,-11,  9,  4,  7 },
    { 29,  1,  3,-22, -5,  6,  0, 12,-14, 11,  1,  6, -3,  4,  6, -2,
       4,-13, 12,  1,  1,  3,-11,  9,-10, -1, -7, 16,-11, -1,  3,  9 },
    {  4,  4, 36,-23, -5, -8,-15,  1, -6,  3, 13, -1, -5, -7,  4,  9,
       2,-11, -3,  5,  1,  3, -6, -1, -4, -4, -2,  2,  3, -1, -5, -2 },
    { 19, 10,  6,-17,  2, -4, -2, -4, -3, 13,  2,  2,-13, -7, -3,-11,
       9, -6,  1, -9, -5,  4, -5, -9,-18, -7,-11,  9,  4,-11,  8,  4 },
    { 16, -3,  9,-16, 18, -2,-12,-16,-11, 11,-18, 16,-13,  6,  2,  8,
       3,  8, -4,-16, 10,-11, -1, -3, -8,  5, -9, -4,  9, -4,  0, -3 },
    { 14, 15,  3,-23, -5,  7, -8, -6,  2, 17,  2, 12, -8,-12, 13, -1,
      -9,  3,  1,  1, 19, 15,  4, -1,  1,  2, -3,  2, -3,  1,  5,  3 },
    { 32,  5,-10,-47, -5, -1,  4, 11, -7,  0,  2, -2,  1, -7,  6, -4,
       6,  2, -4, -2,  2, -2,  0, -4,  1, -6, -5,  2, -2, -1, -3, -4 },
    { 20,  8, 10,-21, -7, -9,-16, 12,  1,  4,  6, -5,  9,-11, -7,  4,
     -11, 28, -3,  2,  4, -6, 10, -8, -5, -5, -9,  9, -2, -1,  6, -5 },
    { 38,  3, 23,-25, -6,-18,  3,-10, -8,  6,-10,  1,-10,  2,  2,  0,
      -7,  2, -4,  5, -1,  8, -3,  0,  3,  3, -1,  1,  0, -4, -4,  0 },
    { 20,  5, 16,-22, 24,-18,  2,-12,-14, -7, -3, 10,  2,  7,-10,  2,
      -8,  1,  8, -1,  4,  1,  4, -2,  5, -9,-18, -8,-13,  5,-11, 10 },
    { 14,  8,-12,-16,  9,-11, -3, -6,-25, -7,  6,  5, -7,-16, 10,  2,
      -7, -1, -9, -3, 16,  4,  3,  3, -3, -3,-15, 13, -3,  4, 13, -7 },
    { 16, -9, 19,-23,  7,-19, -3, -5,-15, 11,-21, 21,-16, 18, -1,  6,
      10,-10, 18,-14, 16,-15,  6, -5, -9,  5,-17, 13,-10, 13,  0, 10 },
    {  8, -4,  4,-24,  8,-21,-18,  9,-11,  4, -6, 17,  5, -9, -2, -2,
       2, 15, -2, -3, -2,  1,  7,-13, 15,-10, -8,-11,  3,  3, -1, -1 },
    { 14, 17,  6,-32,  5,-17, -2,  0, 15, -1, -5, 16,  1, -5, -2,  9,
      -3,  8,  4, -2, -2, -4, -3,  1,  0,  7, -3,  4, -5,  0, -7,  2 },
    { 24,  6, 22,-12,  8,  3,-14,  4, -7,  8,  6,  5,  6,  1,  6,-12,
      15, 10,  4, 11,  9,  6, -7, -4, 10, -9,  2, -1, -5, 11, 15,  3 },
    { 17, 12,  3,-23,  5, -1, -2,  1, -9, -1, -3,  1,  8,  1, -5, 17,
      11,  0, -2,-11,  7,  4,  0,-27, -7,  1,  2, -8,  9,  7,  5,  3 },
    { 12, 10, 12,-10, -4,  5, -1,  2,-24,  5, -8,  2,  6,-17, 19,  5,
      12, -2, 16, -7, -6,-14,  4,  1, -3, 13,-16,  5, -1,  4,  1,  1 },
    { 31,  9, 11,-17, 10, -3, -7,  7,  1,  2,  2,  4, -3, -1, 11,  4,
      -5, -8,  1,  4, 15, -6,-28,  1,  8,  3, -6,  5, 17, -2,  2, -4 },
    { 11, 19, 16,-26,  0, -7, -7,  2,-13,-15,-12,  9, -3, 27,  8,  4,
      -6,  1,  4, -6, 11, -1, -6, -7, -3,  0, -6,  4, -6, -7, -3, -1 },
    { 10, 18, 16,-32, 19, -9, -4, -3, -7,  8,  8, -3,-11, -2, -6,-16,
      13, 13, -6, -1, 10, -2, -2, -9,  0, -3,  9,  4, 11, -2, -6,  6 },
    {  9,  4, 19,-33,  4,  7,-12, 36, -3, -1,  8, -2,  2, -8, -9, -4,
      -8,  0,  1, -1,  0, -4, -4,  3,  0,  3,  6,  0, -6,  2,  0, -2 },
    { 25,  7, 15,-12,  2,-24, -1, 24, -4,  4,  9,  0, -2, -9,  4,  6,
       3, 13, -3,  1,  5, -1, -3, -5, -1,  7, -2,  3,  4,  4,  1,  0 },
    { 19,  6,  8,-20,  9, -9,  5, -4,-13,  7, 11, -3,  5,-13, -9,  6,
     -11, -1,  0,  4, 11, 26,  3,  6, -7, 12,  6, -3,  1, -9,  7,  1 },
    { 15,  6, 19,-23, -3, -9,  3, 16, -6, -4,  6, -5,-10,  1, 16,-14,
       2,  0,  2,-13, -3,  8, -6,  3,  1,  1,  2, -5, 12, -4, -8, -3 },
    { 14,  4, 16,-20,  1, 12,  0,  6, -3,  9,  4, 16, 10,-16,  5,  7,
       5, -4, -4,-18, -3,-11, -4,  4, -7,  3, 13,  7,  3,  3,  2, -7 },
    { 22,  3, -1,-30, 18, -3, -9,  9, -2, 11,-16, -2,-14, 12,  0,  4,
      -5,  4, -1,  3,-20, 12,  4,-10, -2, -2,-12,-12, 10,  6, 11, -3 },
    { 15,  7,  2,-21,  5,  4,  9, -9,-33,  7,  7,  3, -6,-14, -8, 10,
      12,  0,  2, -1,  5,  4, -2,  0, -7,  0,  2,  4,  0,  1, -3,  8 },
    { -7,  0, 12,  3,  0, -6,  8, -4,  0,  2, 14,-15,  2, -7,-31, -3,
      14,  0, 14,-15, -1, -4,-15, 10,  1, -3,  1,  2,  5,  2, -8,  1 },
    { -2,  5,  1,  0, -3,  3,  3, -6, -1,  2, -4,  1,-19,  0,-11, 18,
      11, 10, 21,  5,  6,  2, 10,  3, -6,  0, -2, 13,  5, -1, -2,  9 },
    { -9,  1, -5,  0,  0,-15,  8,  4,  8,  3,  8, 12,-13, -2,-39, -2,
       4, -4,  5, -3, -4,  3, -3,  3, 10,  5,  3,  2, -3,  5, -2,  8 },
    { -9,  6,  6, -8, 12,-12, 23,-18,  4,-15, -5,  2,-20, 13, -7,  7,
       7,-12, 14,-12,  6,  1,  1, -3, -8,  9,  0,  1, -7,  3,  7, -6 },
    {-18, 13,  4,  3,-10,-30,-10, -6,-14,  1, -7, -4,-35,  5,-25, 11,
       9,  8, 19, -4, -7, -3,-18, -8,  1,  5, 10, -4,-14, -9,  3, -4 },
    { -6, -1,  4, -9, -9,  4, 20,  0,  0,  3, 11,  7,-16,-17,-20, 11,
      -6,-14,  1,  4, 19,  2, -8,  6,-15,  3,  6, -5,-14,  3,  7,  2 },
    {  1,  6, -2, -8, -5, -3,  3, -8, 21,  1,  3, 16,-14, -2, -9, -4,
      13, -2, 18, 14, 14, 19,-13,  5,-10,  2, -3,  3,  5,  5,  1, -1 },
    { -1, -5, -6, -2,-11, -7,  5, -4,  5, -1,  0,  3, -3,  2,-19, 18,
      16,  4, 14,-22, -2,-11,-22,  1, -1, 11,  1,  2, 11,-10,  7,-12 },
    {  1,  4,  5, -1, -9, -5,  1, 12,  5,  6, 12,  9,-24, 23,  1, 20,
      14,-11, 13,  5, -2, -2,  5,  6,  2,  1, -9,  6, 10,  5, -4, 11 },
    { -1, -1,  1,  7, -3, -4,  8,-16, 15, -1, -7,  9,-22,-11,-11, 10,
      16,  9, -2,  4, 13, 10,  6, 16,  4,  7,  1, -8, -7,-14, -7,  4 },
    {  1,  3, -6,  0, 15, -9, -4,  0,  4,  6, 12,  9, -6, -5,-22, 17,
       7,-11, 15, -5,  1,  3,-19,  0,-15, -3, 16,  5,  5, -7,-11, 12 },
    { -2, -1, 13,  2,  4,-24, 37, -5, -2, -6, 12,  7, -2,-23, -4,  9,
       2, -3,  3,  2,  3,  3,-14, 11,  0, -4, -2, -2,  3, 10,-10,  4 },
    {  2,  9,  8, -6,-28, 14, 28,-11, 18,-11,  0,  2, -2,  4,-12,  3,
       6,  0,  7, -7, -6,  2,  5, -1, -1, -1,  5,  2,  3,  0, -3,  9 },
    { -7, 14,  5,-10, -3,  7,  4, -5,  7, -8, -7,  4,-12, 14,-16, 25,
       3,  0,  1, -5, 12,-10,  0,-10,  0, 12, 12, 17, 12, 10, -1,  0 },
    { -4, -2,  5, -2,-17, -3,  5, -5,  7,-17,  1,  5, -4,  4,-20,  0,
      11,-15, 13, -8, 10,  1,  1,  5,-12,  9, -8,  0,  6, -1,-11,  4 },
    { -3, 12, 13,-15, -7, -7,  0,  5, 33,  3,  3, -6,-13, -7,-15, 10,
       3,  3,  3, -5,  2,  7, -1,  0,-12,  2, 11, -6, -9,  0,  5, 11 },
    { -8,  5, 10, -7,-14, -4, 13,  0, 18, -3, -6,  7,  1, -6,  0, 21,
       8, -7, 10, -8, -3, 17, -9,  0, -5,  1,  4,  8, -3, 11, -5,  0 },
    { -8,  8, -3, -8,  8,-11, 16,-16, 17,  0,  8, 16,-17, 10,-16, 10,
      -8,  6, 11,  0, 10,  7,  4,  5,  7, -5, -5, -6, -7, -5, -1, 16 },
    { -6,  0,  6,  1, -8, -8,  8, -7, -5,-10,-11,  8,-19,  6, -7, 13,
       5, -3,  4, -8,  7, -1,-18,  9,  0, -5,  6, 26,  3,  8,  2,  4 },
    { -2, -2, 23, -2,-20,  2,  7, -7, -6,-15,  3,  9,-19, -2,-10,  7,
      -2,  7,  9, 11,  0,  4, -4,  6,  9, -2,  4, -3,  4,  3,  2,  8 },
    { -6, 12, 10,-10, -7,  4, 17, 11, -6,  1, 12, 11,-18,  8,-12,  4,
       1, 13,  6,-13, 23,  9, -5,  8, -2, -5,  1,  3,  0, -2, -4,  4 },
    {  7,  1,  7,-17, -8,  8, -1, -7,  5, -6,  4, -3,-16,  9,-24, 18,
      -3, 10, 13,-11, -6,-11, -4, 10,  0, 11,  8,  2,  6, -5,-11,  4 },
    { -4,  1, -5,-10,  0, -3,  9, -2,  4, -1,  1,  5,-41,-10, -7,  4,
      -3,  3,  1,  0,-12,  4, -3,  0,  2, -1, -2, -5,  3,  2, -7,  5 },
    { -2,  1,  4,  4, -3, -6,  1,  0, 12, -5, 11,  0,-17, -3, -1, 11,
       4,  1, 27,-12,  0,-14,  2,-15, -3, -9,  0, -7, -3, 15, -8,  6 },
    { -6,  4,  9,  2,  4,  3,  7,-10, 28,  1, -2, 48,  7,  0,-10, 10,
       1, -9,  2, -1,  0,  3, -5,  5, -4, -2,  7,  7,  1,  3,  2,  5 },
    { -3,  3, -1,  3, -9,  0, -1,  3,  2, -6, 39,-14,-12,  5,-19, 21,
       7, -6,  4, -1, -4,  0, -4,  1,  0, -9,  1, 10,  0, -2,  0,  7 },
    {  4,  2,-29, 12,  5, -3, 16, -6, 15,-13, -4, -1,-13, 22,-16, 17,
      16,  4,  9, -4,  4, -6, -4, 11, -8,  7,  8,  4,  3, -3, -7,-13 },
    {  0,  3,  3, -6, -4,  0,  9,  0,  5,  0, 10, 10,  4,-13,-12, 16,
      23, -4,-12, -6, -4, 20,  2,  0, -4, 23,  1,  8, 11, -4, -5, 15 },
    { -6,  4,-15, -9, -1,-19, 12,-30,-17, -4,  1,-13,-13,  4, -3, 26,
       5,-25, 11,-14, -6,-13,  0, -7,  9,  2,  8, -1, -8,  1, -8, 13 },
    {  1,  6,  1, -4, -4,  1,  2,  0, -3,  2, 10,  6, -6, -2,-11,  4,
      32, 15, 15,-47, -8,  3,-12,  4, -5,  4, -1,  0, -5,  5,  1, -7 },
    {  2, -1,  0,  0, -1, -6,  0, -6,  4, -4,  5,  9, -5,  1, -3, 51,
       4, -5,  4,-14, -1, -4, -3,  1, -4, -1,  0,  2, -8,  0,  1,  2 },
    {  0,  4, -2, -7, -2, -9,  6, -8, 11, -3, -6,  3,-11, -8,-12,  8,
      11,  5, 19,  3,-24, 19,-14, 11, -5,-18, -8,-12, -5, -4, -1,  4 },
    { 16,  9, 10, 14,-18, -2,-18,-27, 10, -5, 12, 14,  4,  0, -2, -6,
     -12, -7, -1,  3,  4,  7, 11, 10,  5, -5, -7,-16, -3, -6,  6,  9 },
    {  7, 15, -9, 10,-19,  4, -5,-37, -2, -4,  8,  2,  4, -1,  1,  9,
      -5, -5,-12,  1, -1, -8,  3, -3,  4,  6,  9,  3,  3, -1,  2,  4 },
    { 13, 17,  3,  9, -7, -7,-15,-17, -8,-13, -4, -8, 19,  2, 16, 25,
       7, 15,  2, 16, -5, -6,-10, -9, -7, -6, -2, -7,  7,  2,  4,  5 },
    { 24,  7,  9,  8,-13, -2,  0, -4,  1,-13,  3,  6,  7, 10, -4, 15,
       5,  7, -4,  5, -5,  3, 13, -7,  5, 15,-11, -2,  7,  5,  8,  6 },
    { 17,  6,-15, 23, -2, -1, -6, -2,  0, -4, 11, -3, 12, 15,  6, -8,
     -15, 10, -9,  7, -1,-11,  2, -8, -4,  3,  4,-10,  4,  4, 11,  1 },
    { 21, 12, -3,  6, -8,  8,-11, -8, -5, -5,  3,  7, -1, -5, 12, 15,
     -10,-11,  3, 15,  8,  4,  2,-15,  0, 14,  1, -8, -1,  3, 10, -7 },
    { 16, 12,  5, 13, -6, 15,-23,  0,-17, -9,  0,  4, -9, 13,  6, 18,
       0,  0, -4, -1,  0, 14,  5, -1,  8, -4, -8, -6,  5, -2, -2,  0 },
    { 14, 16, -1, 12,-15, -9, -6,-20,  4,  6,  8,  9,  3,  1, -9, -4,
      -1,-11,  9, 11,-12,  1,-14, -7,  2, -8, 11,  9, -4, 10,  4,-16 },
    { 13, 10,  3,  7,  0, -8,-33, -6,  4, -4, 19, -2, 14,  6,  5,  7,
       6, -3, -1,-10,-10, -9,  4, -3,  5,  9,  2,  2, 10,  9, -2, -3 },
    { 11, 10, 25, 18, -1, -6,-21,-21,-11,-16,  6,  5, 14,  4,  8,  7,
       0,-10, -7, -9, -5, -4,  3, -1,  1,  6, -1,  6, -2,  2, -3, -9 },
    { 15,  9,  5, 22,-17, 15, -9,  7,  7, -9, 13,  9, 10, -1,  8, -3,
      -2,  6,  1, 17,  8,-14,  7, -3, 12,  9,  1,  0,  1, -5, 17,-18 },
    { 25, 19,-17, 12, -4,-10,  1,-13,-19, -7, -3,  9,  6, -2,  3,  1,
       4, -2,-11,-14, -1, -7, -5, -9,  7, -1, -3,  4, -5,  1,  0, -1 },
    { 20,  8, -3,-10,-24,  3, -6, -2,  0,-12, 14,  6,  7, 11,  4,  7,
     -12, -5, -8,-10,  5, -1, -4,  4, 16,  7,-14,  6, -1, -2, -7,-11 },
    { 16, 18, 17,  1,-15, -6, -5, -3, -1,-19,  8, -2,  2,  8, 12,-19,
     -12,  8,  0, -3, -1, -1,  4,-14,  9, -1,-12, -1, -7, 10, -3,  5 },
    { 18, 12, -7,  7,  0, -3,-13,  0, -1, -4,  9, -2,  6, -1,  0,  1,
      15,-21,  1, -8, 25,-19, 13, -9,  2, 12,  5, -7, -3, -1, -3,  1 },
    { 13, 16, -4,  9, -2,  2, -1,-19, -7, -4, 18, -6, 14, 18, -5,  4,
      -6, -3,-19,-14, -1,-12, 10,  6,  7, 17,-12,-13,-10, -4,  5,  4 },
    { 27, 17,  4, 14, -9, -2, -4, -8,  0, -6, 14,-11, -7,  2, -3, -3,
      -2, -3,-13, 12, 16,  1, -5, -9,-10,-11, -2,  3, -7,  5, 11, -7 },
    {  7, 17,-16, -2,-14,-28, -7, -8, 15,-10,  7, 15,  8, 17, 13, -1,
       4, -7,-12,-11,  0,  0,  2,  3, -3,  7, -6,  6,  1,-16,  1, -2 },
    { 23, 11, -9, 15,-23, -4, -6, -4,  2, -9, -7,  9, -8,  3,-13, -4,
       8, 18, -6, -2,  1, -5,  6,-14, -5, -2, -6, -5, -3, -2,  4, -5 },
    { 12, 13, 18, 18,-35,  2,  7,-17,  3,-11,  6,  9, -3, -2, 10, -4,
       3,  3, -2, -7,  0,  2, -4,  0, -4,  0, -6,  5, 10,  4, -3, -1 },
    { 19, 11,  1, 20,-14,  4, -9,-13, -2, 11,  0, 17, -1, -1, -1, -1,
      -5, -8,  0,  5, -1, -8,  5, -1,  3,  2,-12, 21, -2,-24,  5,  7 },
    { 15, 15,-15, 17,-14,-22,  3, -4,-11, -3, -7,  1, 18, 10,  1, 10,
      -6, -3,  8,  2, -7,  0, -2,  1,  1,  2, -9, -2,  1,  2, -3,  4 },
    { 45, 13,  8, 17, -5,  2,-16,  2,  8, -2,  8,-15,  4,  5, -1,  7,
      -6, -2, -6,  2, -3,  0,  0, -9, -1,  7,  2,  3, -3, -3, -1,  5 },
    {  1, 18, -8, 18,-12,-10,  3,  4,-22,-12, 20,  8, -3,  9,  2, 10,
     -10, -3,  9,  3,  6, -3, 10, -1, -3,  2, -2,  4,  2,  3, -3,-18 },
    {  9, 10, -5,  9,-35,-21,-18,-16, -1,-12, -6, -7,-15,-19, 12,  4,
       4,  9, -7,  2, 14,  1,  4,  0, -1,  6, -7,  2,  1,  1, -4,  4 },
    { 31,  8,-17, 35, -8,  1, -5, -6, -7, -6, 10, -2, -3,  6,  9,  3,
      -6, -2,  3,  3,  5, -3,  0,  6,  0,  1, -5, -3, -2, -4, -1,  0 },
    { 18,  4, -8,  7, -8,-15, -1,-16, 12, 18,  3, 19,  2,  4,  8,  8,
       0, -5, -8,-12, 10, -5,  0,  1,  0,  4, -3, 16, 11, 11, -2, -6 },
    { 27, 15,-17,-10,-23,-22, -1,-14, -4, -7, 20, -2, -7,  6, 15, -5,
      32,  4,  9,-11, -3, -8, 11, -4, -1, -4, -8, -6, -4, -5, -2, -7 },
    { 22,  4, -7,  2,-15,-11,-17,-10,  2,  0, 15, 11,  7, 12, -8,  6,
     -10,-18, -6,-12,  7,  3, 22,  3, -7, 14, -5, -2,-13, -7, -1, -7 },
    { 18, 13,  9, 24, -4,-19, -9,-11, 13,  8,  2,  4, -1,  8, 14, 10,
     -12,  0,  0,  5, 10,  5,  4, -1,  5,  1, -1, 11,  2, -4,  0, -9 },
    { 15, 19, -5,  1, -4,-10, -8,-27,  6,  8,  5, 10,  4, 11,  5, -5,
     -11,  0,-11,-14, -4, -9, -8, -8,  6, -9,  4, -5, -1,  1,  5, -4 },
    { 18,  1,-13, 14,-14,  9,-15, -7, 12,  1, 13, -4,-20, 12, 10, 12,
     -12,  7,  1,-13, 10, -6,  5, -3,  4,  8, 10,-13, -3, -6,  9, -3 },
    { 19,-14,  5, -8, -6,  2, -5,  5, -3, -1,-28, 11, 18, -6, -4, -2,
      11, 14,-43,-42,  9,  2, 20,-23,  6, 32,  0,  5,  0,  6,  9,  5 },
    {  8, 11,-14, -1,  7, 12, -7,  2,-16,  2, 10, -3, -1, -7, -7, -1,
       1,-10,-60,-23,-18, 42,-13,  9, 18,-11,  0,  1,  0,  2, -5,  1 },
    { -5, -1,  2,  0,  3, -3,  3, -2, -6,  0, -3, -3,  7,  2,  0, -2,
      -2,  3,-34,-15, 37, 47, 10, 20,  9,  1,  3,-21,-25,-33,-14,  8 },
    {  5,  6,  2, -2, -2, -2,  6,  5, -5,  7, -3,  1, -5,-13,  9,  3,
     -17,-19, -2,-79,-12, -7, -8, -6, -2, -2, -1, -1, -7,-13,  6, -1 },
    {  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1, -1,
       0,  3,  4,-87,  6,-11, 16, -9, -1,  8,  0,  5,  0,  1,  2,  1 },
    { -5,  6,  2,-24,  5, -9, -7,  0,  7,  3, -3, 16,-14,-16,  0, 18,
      15, -9,-14,-28,-17, 53, 14, -6,-28, -1, -3,-10, -7,-14, 19,-15 },
    {  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -3,  0,
     -13,  0,-53,  3,-22, 63, 19, 16,  1,-11,  0, -3,  0, -3,  0,  1 },
    {  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, -1,
      -1, -6,-43,-43, -2, 65,-13, -4,  9,  1,  1,  2,  1,  0,  0,  1 },
    {  0,  1,  0,  0, -1,  0,  1,  1,  0,  0,  1,  2, -1, -1, -3, -1,
     -23,  1,-61,-55,  3,-28, -6, -4, -4,  8,  2,  1,  1, -1,  0,  0 },
    {  0,  1, -1,  1, -1,  0, -1,  0,  1, -1,  0,  1, -1,  0, -9, -4,
     -48,-19,-52,-46, 11,-12,  5,-14,  0,-10,  0,  0, -1, -2, -1,  0 },
    {  0, -3, -1, -4,  2, -1, -7,  3,  1,  3, -1,  1, -3,  0, -7,  0,
       3, -7,-61,-51, -4,-21,-16,-21,-11, 14, -7,  8,  3, -5,  1,  2 },
    {  0,  0,  0,  1,  0,  0, -1,  0,  0,  0,  0,  0,  1, -1,  9, -3,
      56,-11, -6,-67, -1, 13,  0,  7,  1, -9, -1, -1,  0,  0,  1,  0 },
    { 14,  9, -2, 14,-10,-10,  9, -5,  1, -8,-23, 30,  8, -7, 23,  8,
       2, 10, -1,-27,-17, 57, 22,  4, -5,  2,-12, -6,  2, -7, -4, -9 },
    {  1,  5, 12, -2, -2, -3,  2, -3,  6,  0,  4, -2, -8, -6,  0, 16,
     -15, 29,-55,-29,-24, 29,  3, 10,  6, 13, 10, -5, 21, 11,-14,  5 },
    {  4,  2, 26, -6, 10, 11,-23,-10,-27,-20,  3,-24,-11,-10,-13, 25,
     -10,  5, -9,-36, -7, 43,  3,-13,  6, 13, -2,  0,  1,  3, -3, -4 },
    { -1,  0, -1,  0,  0,  0,  0, -1,  1,  0, -1,  0,  0,  0, -1,  1,
     -12, 12,-26,-64,-15, 29, 37, -7, -3,-12, -5, 14,  8, -8,-10, -2 },
    { 19, -4,-11,-16,  8, 14,  5, 19,  3, 22,-11,-21, -1, -6,-11, 11,
      10,-24,-23,-40, -8, 20, 17,  5, 13, -6,  3, 14,-20, -8,  3, 28 },
    {  2,-12, 10,-14,-18, 26,-22,  4, -2,  5,-21,  8,  3,  1, 19,  0,
     -12, 24,-14,-40, 15, 29,-15,  6, 15,  1,-19,  2,  4,  7,-12, -3 },
    {  0, 17, 13,  7, -5,-11,  2,-19,  3, 38,-21, -3, -6, -4,  7,  1,
       1, -5,-40,-10, -2, 35,  8,  8,-10, -8, -9, 33,  4,  4,  0, -2 },
    { -2,-12,  7, 29,-24,  2, 16, -1, -7, 16, 10, -2, -2, -2, 13, -2,
     -37, 15,-22,-40,-11, 33, 10, -1,  8, 10,  6,  8,  9,  0,-12,  2 },
    { 15, -8, -9, -2,  7,-17,  7, 19, 14,  4, 12, 27, 11, 10,  4, 11,
     -15, 14,-13,-48,  5, 18,  0, -9,-36,-11,  2,  4,  5,  5,-15,-12 },
    {-12,  0,  3,  4,  7, -5,  5,-14,-24,-18, -6,-15, -8,-20,  1, -7,
     -33,-28,-40,-38,-18,-10, -5, 17,-12,  4,  3, -5,  5,-13,  4, -7 },
    {  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, -1,
      -3, -9,-49,-60, -5, 45, -1,  6,  0,  0,  0,  0,  0,  0,  0,  0 },
    {  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, -1,
      -3, -9,-49,-60, -5, 45, -1,  6,  0,  0,  0,  0,  0,  0,  0,  0 },
    {  0,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0, -1,  0,  0,  0,
       3, -2,  9,-29,-11, 55,  8, 32,-36,-13, -7, 37,  4, 11,  0,  3 },
    {  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
       4, -1,-39, -4,-30, 63, 28,-17, -6, 10,  7,-14, -9, 11,  9,  7 },
    {  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1,
      13, -2,-50,-32, 22, 51,  4,  7,  6, 11,-20,-13,  9, -5, 21, -4 },
    {  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, -1,
      -3, -9,-49,-60, -5, 45, -1,  6,  0,  0,  0,  0,  0,  0,  0,  0 },
    {  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, -1,
      -3, -9,-49,-60, -5, 45, -1,  6,  0,  0,  0,  0,  0,  0,  0,  0 },
    {  0,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0, -1,  0,  0,  0,
       3, -2,  9,-29,-11, 55,  8, 32,-36,-13, -7, 37,  4, 11,  0,  3 },
    {  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
       4, -1,-39, -4,-30, 63, 28,-17, -6, 10,  7,-14, -9, 11,  9,  7 },
    {  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1,
      13, -2,-50,-32, 22, 51,  4,  7,  6, 11,-20,-13,  9, -5, 21, -4 },
    { -8,  2,  1, 22,-31, -6,-25, -3, -3,  1,-15,-11, -2, -3,  4,-13,
      -9, 15,-18, 37, -7,-37, 12,-13,-11,-25,-10,-11,-22,  7, 16,  7 },
    { 14, 10,  4,-10, -1, -5, -7, -3, 16, 13, -5,-15,  5, 11, -1,  8,
     -27,  7,-12, 49, 17,-22,  9, -2, -9, -1,  2,-15, -1, 41,-18,-17 },
    { -4, -9,-15, -3,  3,  4,  4,  2,  7, -3, -7, -8, -5, 17,-19, -7,
      36, -9,-38, 17,  1,-48, 11,-18,-13, -2, -8,  4,-10, -5, 21, 11 },
    { 15,-13,  4,  2,  1, -5, -2,  1,-10,  7, -1,  3, -6,  0, 11,-11,
       8, 20,-17, 51,-17,-41,  2, 15,  4,  8, -2, 16,-32, -1, 17,  6 },
    { -8,  8,-18, -5,  4,  6, -3,  8,  0, -4,  2,  0, -1, -4,  5,  8,
      30, 30, -8, 70,  2,  8,  2,  0,  7,  1, 13, -1, -6, -7,-11,  2 },
    { -8, -7,  9,-10,-13,  6,-11,-14, 13, 25,-26,  5,  2, -5, -5,  5,
      -8,  4,  0, 33, 12,-38, -4,  6, 13,  6, 25, 34, -1, 25,-19, -5 },
    { 18,  3,-17,  4, -8,  7, 20,  1, -1,  5, -5, -2, -8,  8,-35, 15,
      24, 43, -5, 51,  5,-12, -3,  1, -2,  3, -3, -3, -9,  8, -9,  2 },
    {  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,
       2, 10, 24, 76, -2,-22, 11, -1,  4, 33,  4,  1, -1,  1,  2,  0 },
    {  0, -1,  0,  1,  0,  1,  0,  1,  0,  0,  0,  0,  0,  1,  2,  0,
      24, 13, 32, 70, 26,  5,-21, -9, -6,-15,  2, -2,  2,  4,  1,  1 },
    {  5, -4,-11,  4, -4, 22, 10, -2, 13,-11, -4,-21,-17,  0, -7,  4,
      10,-34, 11, 52,  2,-46, -5,  0,  0, -1,  2,  4, -9,  1,  1, -7 },
    {  0,  1,  1,  0, -1,  0,  1,  0,  1,  1,  0,  1,  0,  0, -3,  1,
      -8,  9, -1, 64,-13,-61, -3,  3, -5, 10,  1,  3, -1, -1, -1, -1 },
    {  0,  1,  0, -1,  0, -1,  0,  0,  1,  0,  0,  0,  1,  1,  2,  1,
      10, -2,-31, 79,-10, 27,  0, -1,  3,  8,  1,  1,  0, -1,  0, -1 },
    {  3, 12, 10, 26,-19, 10, -9,  6, -4,-15, 10,  3,-16,  6, 11,-19,
       3, 10, 18, 44,  5,-30,  5, -9, 21,  4, 20, 10, 14,-25,  8,-17 },
    {  0,  0,  0,  1, -1,  0, -1,  0,  1,  0,  1,  1,  0,  0, -6, -2,
       8, -8, 13, 69, 26,-19,-25,-17, 16,  6,-12, 22,  2, -6,  9,  5 },
    {  0, -1,  0,  1,  0, -1, -1,  0,  0,  1, -2,  1,  0,  0, -4, -1,
     -34,-15,-33, 56,  9,-42,  9, 10,  6,  9, -8,-11,  0, -6, 15,  5 },
    { 10,  2,-14, -3,-15,-35, -1,  7,-18, 14,  8, -1,-15,-26,  6,-15,
     -18, 22,  9, 33,  0,-32, -9,  3,-11,  7,  4, -1,  5, 30,  9,  1 },
    {  4, 15,  0,  6, -5,-11,  9,  6,  6,  6, 14,  2, -1, 10,-24,-25,
      -2, -4, -1, 37,  2,-29, 14, -9, 22, 17, -2, 33, 10,-25, 11,-11 },
    {  0,  5,  2, 18,-12, 21, 22, 33, -7, 21, -9, -7,  7,-15, -7, 16,
       7,  0,-14, 44, 10,-25,  5, -4, 15, -8, 10, -4,  5,  9, -1, 16 },
    {  3, 13, 12, 12,  8, 25,-23,  8,-22, -3,-18, -8, 15, 12,  9, 19,
       0,  0, -9, 49,-27,-15, -9,-15, 12, -8,-16, -7, 13,  5, 13,  2 },
    { 12, -6,  7, -2, 20, -9,-14, 12, 13, -5,-17, 22, -8, -4,  2,  7,
     -13, -2,-15, 43, -5,-30, 27,  4, 10,-27,  5, 27,-10,-10,-18,  0 },
    {  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,
      -1, 10,-18, 70, -2,-52, -1, -7,  0,  0,  0,  0,  0,  0,  0,  0 },
    {  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,
      -1, 10,-18, 70, -2,-52, -1, -7,  0,  0,  0,  0,  0,  0,  0,  0 },
    { 15,-13,-20, 16,  2, 13,  5,-11, -8, -5, -3,  2, 24,-23, 30, -7,
      11, 30,-15, 43,  5,-15, 15, -3,-14,  1,-23,  8,  3,  9,  4,-11 },
    {  0, -1,  0,  1,  0, -1, -1,  0,  0,  1, -2,  1,  0,  0, -4, -1,
     -34,-15,-33, 56,  9,-42,  9, 10,  6,  9, -8,-11,  0, -6, 15,  5 },
    { 10,  2,-14, -3,-15,-35, -1,  7,-18, 14,  8, -1,-15,-26,  6,-15,
     -18, 22,  9, 33,  0,-32, -9,  3,-11,  7,  4, -1,  5, 30,  9,  1 },
    {  4, 15,  0,  6, -5,-11,  9,  6,  6,  6, 14,  2, -1, 10,-24,-25,
      -2, -4, -1, 37,  2,-29, 14, -9, 22, 17, -2, 33, 10,-25, 11,-11 },
    {  0,  5,  2, 18,-12, 21, 22, 33, -7, 21, -9, -7,  7,-15, -7, 16,
       7,  0,-14, 44, 10,-25,  5, -4, 15, -8, 10, -4,  5,  9, -1, 16 },
    {  3, 13, 12, 12,  8, 25,-23,  8,-22, -3,-18, -8, 15, 12,  9, 19,
       0,  0, -9, 49,-27,-15, -9,-15, 12, -8,-16, -7, 13,  5, 13,  2 },
    { 12, -6,  7, -2, 20, -9,-14, 12, 13, -5,-17, 22, -8, -4,  2,  7,
     -13, -2,-15, 43, -5,-30, 27,  4, 10,-27,  5, 27,-10,-10,-18,  0 },
    {  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,
      -1, 10,-18, 70, -2,-52, -1, -7,  0,  0,  0,  0,  0,  0,  0,  0 },
    {  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,
      -1, 10,-18, 70, -2,-52, -1, -7,  0,  0,  0,  0,  0,  0,  0,  0 },
    { 15,-13,-20, 16,  2, 13,  5,-11, -8, -5, -3,  2, 24,-23, 30, -7,
      11, 30,-15, 43,  5,-15, 15, -3,-14,  1,-23,  8,  3,  9,  4,-11 },
    { 16,-18,  7, -4, 31,-15, -9,-13, 20,-12, -6,  0, 12, -6, -2,  4,
       3, -3, -1,  0,  1,  3,  3, -2,  1,  6,  4,  0, -3,  2, -5,  1 },
    { 38, -5,-13, -4,  8,-15, 11,  1,  2, -4, -1,  9, 13,  4,-12, -7,
       0, -2,  7,  2, -6, -2, -3, -2,  3, -4,  6, 15,  1,  1,-11, -2 },
    { 47,-22,  9,-26,  3, -5,  2, -7,  4, -2,  2, -2,  3,  0,  3, -4,
       3, -3,  2, -3,  7, -3, -1,  1,  1, -5,  5,  0,  2, -5, -3, -2 },
    { 14,-16,  2, -6,  7, -2, -7, -4, -4, -7, 14, -3,  7,-19,-14,-17,
     -29,  6, 26, 16, -5, 13, -4, -1, 21, 14,  1,  3, -6,  0, -7, -1 },
    { 29,-11,  5, -3,  4, 11,  4,-10,  1,-22, -3,-10,  5,  4,  2,  8,
      -2, -7,-12,-12, -8, -3,-18, -2, -9, -5, -1, -3,  2,-14,-14,  7 },
    { 28,-12,  5,  3,  9, -7,  0, -2,  2,  1,  4,  0, -7, -3, -2,  4,
       4, 14,  8, -1, -4, 14, -7, 17, -2, -2, -9,  2, 19, -7,  9, -8 },
    { 31,-18,-22,  8, 15, -5,-10,-15,  1, 10,  6,  7,  6, -8,  2, -1,
      12, -3,  3, -1,  1,  5, -6, -4,  0,  1,  7,-10, -2,  4, -3, -4 },
    { 53,-30, -4, 12,  2,  3, -3, -3,  0,  1,  6,  5, -5, -4, -7,  1,
       0,  2,  1,  3,  1,  5,  0,  2,  2, -1,  0,  4,  2,  0, -2,  0 },
    { 27,-18, -3, -2,  4, -8,  3, -2,-11,  2, 10, -8, -8, -4,  0, -2,
       8,  0,  9,  0,-16, 11,  1, -6, 13, -3,-10,-13,-15, 25,  1,  0 },
    { 35, -5, -1, -8, 23, 11,-14, -3,  2, -2,  8, -6, 17, -2,  7,  0,
      -2, 10,-17, 13, -2, -2, 11, 11,-14,  2, -2, -3, -8, -1,-12, -5 },
    { 29, -9,  7,  3,  2,-10,  0,  3,  9,  0, -3,  5,  1,-10, 10, -5,
       3,  6,-20, -9, -6, -4,  1,  0, 12, 17, -8,  9,  3, -1, -9,  0 },
    { 15,-16, 18,-19, 16,-15, 17,-18, 13,-16, 17,-14, 15, -9, 13,-17,
       9, -7,  4, -5,  3, -4, -3,  0, -6,  7, -9,  7, -2,  7, -9,  9 },
    { 21,-10,  7, -2, 12, -7, 13,-17, 11, -2, 20,  3,  5,-11, -6, -6,
     -15,  0, -9,  5,-11,  7, -1,  7,  8,-10, -9,  3, -5,  9, -8, -2 },
    { 23,-22, 15, -5, 16, -4, -3,-12,  9,  3, -1, -2, -8,  2, -2,-16,
       3,  4, -2, -6, -7, 12, -8,  2,-14,  2, -7, 11, -2,  6, -4, -1 },
    { 34,-17, -4,  8,  4, -6,  1,  8,  4, 16,  3,  6, 12, -1, -1,-15,
       6,  4, -7, -6,  6,  0,  2,  1, -2,  2,  3,  3, -3, -2,  8, -6 },
    { 18,-18,  2, -2, 10,  1, 18,-23, -3,-10,  0,  4, 20,-19, -3, -4,
       2,  8,  6,  1, -3,  1,  1,  3,  5, -1,-11,  3, -7,  5, -1,  1 },
    { 15,-14,  2,  3, 10, -8, 12,-13, 13,-15,  6, -8, -4,-10, 14, -9,
      24,  2, -7,-18, 13,-11,  8, 14, -6, -2,  3, -1, -4,  7, -7, -4 },
    { 20,-12, 13,  5, -1,-10, 15, -6,  8, -1, -3,-10, 17,  0, -6,-19,
       2, -1,  8, -3,-16,  0, -3,  2, -2,  0,  8, -9,  0,  1,-10, -9 },
    { 32,  0, -9, -5, -1,  5, 13,-11,  8,  3, 11,-11,  0, -8, -2,-14,
       7, 10,  6, -5,  1, 10,  2, 12,-10,  4,  4,  6,  4,  0, -7,-10 },
    { 16,-14, 10, -7, 11,-11, 11,-11, 18,-13,  8,-15, 16,-11, 13, -9,
       8, -7, 12,-11,  7, -6,  3, -5,  9, -5,  4, -1,  7, -4,  8, -3 },
    { 24,-27, -1,  5,  8, -5, 12,  7,  4, -3,  3, -1, -9,-11,-13, -5,
      10,  0,-13,  7,  1, -5,  4, -9,  7, -3, 13,  2, -5, -3,-17, -2 },
    { 23,-19, 15,  1,-10,-18,-12, -6,  8, -3, 12,  0,-12,-10, -4, -4,
       8,-10,  4,  2, -2, -8, 13, -3, -2, -6,  2, -3,  5, -2,  2, 11 },
    { 25,-12,  4,  2, 24, -3,  3, -6, 14, 11,  0,-21, -3, -3,  1, -8,
       7,  0,  0,  3,  3, -6, -7,  6,  2,  1, -4,  5, -1, 10, -2,  9 },
    { 24, -8, -6,  7, 16,-12, 13, -1, 11,-21,  2, -6,  3,-12,  0,  9,
       4, 11, -7,  1,  4,  1, -8,  3,  3, -6,  3,  3,  0, -8,  8,  4 },
    { 25,-21, 13, 14, 13,-18,  4, -3,  0, -5, -4,  5, -3,  0,  4, 12,
       7,  3,  5, -5,  2, -2,  3,-10,  2, -9,-15,  6,  1,  7, -5,  1 },
    { 23,-16, -2, 10,  4, -1,  3,  1, 32,  3, -5, -2,  9, 10, -1, -4,
      -6,  2,  9, -1, 14, 12, -6, -1,-17, -2, -4, -9, -7, -6, -8,  3 },
    { 50, -8,  5,  2,-11, 10,  0,  0,  6, -3,  7,  0, -3, -2, -3,  0,
       6, -4,  2, -5, -9,  0,  3, 10,  1, -7, -2, -3, -6, -9,  1, -2 },
    { 28,-17,  0, -2,  2, -9,  1,  5, -4, -1,  0,  0, 19,-27,  5,-12,
       7,-14, -3, -6, 10, -2, -4, -2,  4, -5, -2, -7,  1,  7, -9,  4 },
    { 22,-19, -6, -6,  3,-22,  3,  5, 20, -8,-14, -5,  1,  1, 20,  2,
      16,  6,  3, 14,  4,  3,  5,  1,  5, -7,-10, -6,  3, -6,  1,-14 },
    { 29,-14, -8, 13,  8,-10, -6,  4,  4, -6,  5, -7,  1, 12, 14, 11,
      -7,  1,  2, -9,-11, -9,  0,  4, -1,  7, 10,  4,  4, 20, -1,-11 },
    { 18, -9,  4,  1,  7,-29, 12,  1, -1, -9, -2, -1, -2,  2,  9, -8,
     -13,  5,  4,-13, -4,  2, -5, -7, -6, 14,-10,-34, -3,  1, -3,-13 },
    { 38, -9, 24,  8, 11,  4, -6,-11, -2,-12,  1,  1,-11, -8, -5, -2,
     -15, -8,  8,  0,  1, -7,  5,  4, -1,  8, -2, 11, -3, -1, -5, -5 },
    {-20, 11, -4, 24,-11,  1, 15,  4,  0,-28,-10, -1, 10, 10, -6,  5,
      -6,  2,  7, -2,  1, -2, -6, -3, -7,  1,  2, 12, -1,  7,  0, -2 },
    { -9, 10,-23, 27, -4,-17, 20, -6, 14,-17,  5, -1,  5, -9, -7,  5,
      -6,  4, -2,  9,  0,  8,  0,  1, -3, -3, -5, -8,  5, -2, -2, 12 },
    {-10, 19,  4,  9,  1,-16, 17, -2,  9,-29,-16,-11, -4,  7, -5,  4,
      -1, -3,  3,  2,  3, -4,  5,-12, -2,  6,  5, -4,  4,  1,  4, 10 },
    {-20, 10,-24, 14, -5, 11,  9,  0, 16,-20, 10, -5, -6, -6, -1,  2,
      -4,  5,-16,  8, -2,  5,  5,-11,  9,-11,  4,-11, -1, -1,  4,  3 },
    { -9, 11,  3, 19, 24,  4,  5,-14, 30,-17, -4, -2,-17,  7,  2,  3,
       1,  3, -7, -4,  2, -3,  1,  4, -1, -1,  3,-12, -2,  3, -3, 10 },
    {-19, 18, 11, 19, 19, 19, 10,  4, 13,  6,  5,  4,  8,  3, -2, 12,
      -6, -2,  7, -6, 15, 12, 16, 16, 18, -3, -4,-20,  0, 10, -9, -3 },
    {-21,  9, 20, 12,  0, -3,  5, -9, 15,-13,  5, -5, -6, 24,  2,  9,
      -5,  2, -7,  2,  5,  7, -5,  2, 15,  3,  1, -1, -4, -2,  7,  0 },
    {-18, 16, 13, 15,  2,-10, 14,-11,  4,-11,  5, 12, 12, 20,  8, 30,
       2, 11, -9,  7,  0, -3,-16, -5, -6,  5, -4,-21,  0,  5,  6,  1 },
    {-26,  8,-13,  9,  6,-10,  2,-11,  7, -4,  6,-19,-11, -6,-12, 16,
       0,  5, -7,  8,  5,  6, 17, -9, 10,-10,  5, -3,-11,  2,  4, 10 },
    {-11, 17, -3, 22, -5, 18,  3,  1,  4, -5, 14,-27,  5, -7, -4, -5,
     -10, 11,  1, 15,  1,  1, -6, -5, 10,-22, -7, -7,-15, 13, -4,  5 },
    {-17, 14, -7, 13,  3,  0, 13, -6,  9,-14,-22, -1,  1, 19, 14, -3,
       4,-13,-13,  2, -4,  8, -2, -2, 13,-12, 13,-12, -7, -5, -3,  6 },
    {-17, 17, -1, 33,  6,  3,  9,-16,  3,-14, -8,  6,-17,  8,  3, 13,
       8, -6,  3,  1, -2,  0, -2,  8,  4,  9, 13,-10,  4,-17,  0, -6 },
    {-20,  7,  7, 21,  1, -3,  7, -3, -2,-12,  9, -7,  2, -3, 14,  1,
      -1, -7, 12,-10,  5,-20, 11, -2,  0,-24,-17,  6,  6, -4,  3, -1 },
    { -8, 10,  6,  7, -1, -6, 28, -6, 10,-33,  1,-20,  0,-12, 10,  1,
      -6,  8, -3, -1,-10,  8,  5,  0, 10, -2,  8, 16, -5, -3, -7,  4 },
    {-17, 13,  3, 15,  1, -5, 27, -5,  6, -6, 12,  2, -4,  8, -1, -3,
      -2, 12,-15,  3,  4,  1,  2, -9,  0,-16,-21,  2, -4, 16, -7,  4 },
    {-15, 20,  8, 17,  5,-14, 15,-11, 21,-11, 13,-13,  2,-15,-13,  1,
      -5,  5,  2, 10, -9,  4, -1,  3,  2, -4, 13, -5,  1, -4,  5, -3 },
    {-21,  8,  2, 16, -1,  2, 15,-16, 13,-12,-12, -7, -8,  2, -7, 11,
      -8,  5,  2, -7, 16, -4,  1, -7,  3,-15,  6, -5, -8,  2, -8,  5 },
    {-15, 17, -6,  3, -3,  3,  9, -7, 14,-23, 11,  1, -1,  4,  7,  6,
      -1,-14,  7,  6, -8,  5,  1,-15, 10, -9,  2, -3, -1,  4,-10, -4 },
    {-10, 18,  3, 11,  1,  4, 14,-14,  7, -4, 15,-10, 10,-11, 10, -4,
       5,-14, 10,  4, 15,-12, 15,-13, 20,-15, 14,-15,  8,-11,  4, -6 },
    { -7, 23,  2, 20,  7,  8, 19, -5,  9,-16, -8,-17, -5,  1,  5, -6,
      -8,  1, -6, -4, 10,  6,  6,  2,-11, -4,  0,  2,  4,  7,  9, -4 },
    {-15, 20, -5, 22, 11, -8,  9, -5, 10,-13, -8,  8,  2, -2, -3,  7,
       6, 10,  1,  2, -5, -9,  1, 10, 16,-22, -7,  0,  7,  7,  6,  1 },
    {-26, 19, -5,  3,  5, 25, 18, -5,  9,-14, -8, -6, -2, -6,  2,  3,
      -8, -2, -7,  7, -3,  7,  3,  4, -8,  0,  1, -8, -4, -2, -2,  1 },
    {-20, 14,-10,  6, -3,  7,  8,-32, -2, -7, -2,-10, 16,-12, -9, 15,
      -2, -5, -6,  2, -7,  5,  9,  1,  6, -7, -1,  0, -2, -4, -7,  3 },
    {-14, 16,  4, 11, -8,  1, 23, -4, 17,-13,-10,  1, 12,  9, 12, -4,
       7, -1, -1,  5, -8, -6,  3,  3, -6, -3,-18,  0, 18, 20,  4, -2 },
    {-33, 19,-10, 30, 15,  2, -3, -1, -4,-14,  7, -7, -1,  7, -8,  9,
      -1, -3, -5,  2,  2,  4,  0,  5,  0,  0,  2,  3,  3, -3, -3,  4 },
    { -6, 20,  0,  5, 17,-10, 18,-17,  9,-16,  4,-13, -6,  2,-14, 14,
     -28,  9,-12, 25, -4,  7,  7, -8,  6, -6, -2,-10,  2,-11, -1,  2 },
    {-12, 14, 12, 52, -3,  5, -5,  4,  8,-13,  2, -5, -4,  2, -2, -1,
      -2,  3,  3,  5,  2,  3,  0,  1, -5,  2, -4, -3,  1, -5, -2,  0 },
    {-13,  6,  9, 24,  0,  8, 14,-15, 18, -9,-11, -8,  3, 15, -2, -4,
      -9,  4, -3, 12, 14,-13, 11, -4,  2, -4,  0, -6, -6, -6,-14, -1 },
    {-10, 28,  3, 12,  9,  3, 11,-28,  6,-11, -7,  4,  0,  7,  8, -9,
       0, -6,  0,-16,  4,  7,  4,  4,  7,  3,  4, -7,  0, -3,-10,  6 },
    {-11, 14, -2, 19, -1, -1,  7,  9, -2,-27, 10,-14, 15, -4, 12, -4,
       2, -2, -6, 12, -6,  0, -5, -4, -5,  1,  3,-11,  5, -9,  3, -8 },
    {-18,  7, 13, 16, -4,  3,  9,-10, 10,-10, -3,-22, -4,-12,  3,-16,
       0, -3,-16,  8,-11,  1, 10, -7, 15,  3,  0, -1,-13,  8,  1,  6 },
    {-20, 10,-10, 10,  8, -1,  6,  0, 16,-12,  9,-10, -1, -5, -4,-13,
      13, 16, -8, 12, -2, 14, 18, 13,  0,-16,  2, -5, -5, -5, -4,  3 },
    {-14,  5, -7,-17,  5,-13, 23, 20, -4, -1,  1, -6, 13,  5, -1,  4,
     -14, -2, -7,  8,  3,  2,  2, -7,  2, -1,  4,  7,  3, -9, -1, -5 },
    {-19,  3,-24,-28, -9, -7, 19,  3,  2, 19,  7,  5,-13,  8,-15,-17,
       3,-11,  4, 13,  3,  2, -1, -3, -4, -4,  2,  0, -5, -6,  6,  2 },
    {-17, 18,-30,-20, -2, -3,  1, 15, -1,-11,  6, -4, 11, 11, -4, -5,
     -10,  0,  0,  1,  3, -7,  8,  2,  5,  1,  5, -5,  1,  6,  4,  1 },
    { -6,  1,-30,-25, -1, -8, -2, -9,-17, 16,  3, -1, -2, -9, -6, -7,
      -3, 12,  6, -4,-10,  0, 10, -8, -6, -5, -3,-11, -4,  0, -1, -3 },
    { -1, -1,-34,-28,  1,-10,  2,  9,  4, 16,  2,  6, 14, 17,  0,  7,
      -4,  4,  4,  4,  0,  1, -1, -5,  8,  1, -4,  1, -9, -2,  5,  6 },
    {-11, 14,  1,-31, -7,-24,  9,  7,  6,  5,-13,  1, -1,  3,  4, -1,
      -2, -8, -6,  3,  5, -4, -6,  7, -2,  5,  3,  3,  0,  0, -5,  2 },
    {-25,  8,-11,-18,  1, -4,  8, -3, -4, 15,  6, -5,  8,  2,  3,  4,
      -4,  5,  6,  8, -7,  6,  1,-11,-15,-13,  9, -4,-14, 10, 12,  7 },
    {-20, 11,-15,-25,  3,  4, 18, 13, -4, -5, -9, -1, -5, -2, -2, -7,
      16,  5, -4, -5, -7, -2, -3, -9, 11, -2,  0, -7,-17, -6,-11,  6 },
    {-11, 18, -5,-20,-15, -3,  9, 11,-20, 12,  5,  5, 11, -3,  7,  1,
      10, -6, -3, -3,  3,  3, 14, -7, 10,-17,  9,-11, -2, -6,  7,-12 },
    {-20,  8,-14,-17, -9,-13, -3,  0,-27,-14, -3,-14,  4,  3,  6, -6,
       7,  4, 23,  9, 11,  9,  3, -4,  9,  2,  4, -1, -6,  1, -8,-11 },
    { -9, 14,  2,-37, -7, 13,  6,-11, -6,  9, 18,-11, -6,  2, 12,  4,
      -1,  3,  1, -2, -2,  1, -9, -4, -2, -3,  3,  5, -6,  0, -2, -8 },
    {-29,  8, -1,-13, -2,  8, 23,  2,-10,  7, 13, -6, -5, 11, 13,  0,
     -10,-13, 11,-12,-10,  6,  4,  6,  4,  3,  6, -5, -9, -2, -1,  3 },
    {-18,  6,-10,-55, -4,-11, -2,  0,  1, -3, -9, -6,  3, -2, -1,  6,
       3, -1,  3,  1, -4, -7, -2,  6,  3, -2, -1, -3, -2,  0,  4,  1 },
    {-14,  5,  3,-21, -8,-16, -4, -2,-11, 27, 15,-20,  3,  0,  1,  1,
       2, -5, -5,  4,  1, -9,  5, -3,  3,  0, -4, -2,-11, -4, -3,  7 },
    {-17, -1, -9,-17, -8,-18, 12,-13, -9, 13, -3,  3,  3, -3,  1, -2,
       0, 16, -9,  6, 12,  9,  5, 11,  2,-15,  1, -4,-16,  7, -4,-12 },
    {-18,  8, -6,-11, -8, -7, 13,  7,  1,  6,  8, -1, 21, -4, 14, 15,
      18, -4, -3, 15,  0,  9,  4,  7,  3, -1,  9, -2,  0,  7, -8,  2 },
    {-10,  7,-18,-29,  3, 12, 12,  9, 11,  4, -1,-15,  1, -1,  8, -2,
      -2, 10,-15, -1,  0,  6, 12, -6, -1, 10, -6, -3,-11, -4,  9, -6 },
    {-14, 14, -9,-21,-12, -2, -1, -7, -5,-10,  5, -8,  0,  6,  9,-11,
      11, -3, -5,  3,  8, 15, -2, -4,-22,  4, -6, 12,  2, 13,  6, -7 },
    {-12, 11, -5,-29,-25,  4, 12,-13,-11, -7,  4,  2,  2, -5,  5,  8,
       7, -5, -5,  6,  3,-10,  1, -6,  6, -6, -5, -1, -2, -4,  7,  6 },
    {-15, 11, -5,-16,  0,-13, 26,-23, -6, -3,  5, -2, -2, 21, -6, -3,
      -5, -1,  6, -1,  0,-13,  2, -3, -9, -1, -4, -3,  5, -4, 12,-16 },
    { -9,  9, -1,-17, -3, -6, 12,  6,-18, -2, 11,-14, -6,  3, 14,-12,
     -11, -5, 14,  2,  5, -8, -4,-11,  2, -5, 16,  6, -7, -4,  8, 13 },
    {-13,  5,  3,-28,-14,  0,  6, 23,  5,  4, -1,-17,  1, -3,  0,  0,
       5,  4,  0,-18, 14, 10,  4,  2,  5, -2,  4, -3,  2,  0,  2,  0 },
    {-15,  4,-13,-16, -3,-12, -2,  2,  7, 10,  9,  3, 11,  4, 23, 14,
       9, 16,  4,  1,-12, -3,  4, -7,-15, -7,-10,-14, -6, -8, -1, -6 },
    { -7, 10, -5,-10, -3,-13, 16, -1,-12,  7, -3,-12,  2, 13, 13,  2,
      17, 15,-13,  1, -5, -2,  3, -1,  1, -3,  6, -3,-12,-16,  7, -7 },
    {-11, -5,-12,-30, -6,-22,  1,  4, -6, -3, 12,  6,  7,  0, 16,  6,
      -2,  0,-22, -2, -9,  2,-13,  8,  6, -8,  4, -7, -1, -6,  4,  6 },
    {-14,  5,  1,-27, -4,  2,  1, 14,-11, -7, -8, -4,  1,  8,  0, -6,
     -13, 11,-12, -7, -5,  1, 10,  7,  3, -2,  0,  6, -8,  2, 10, -1 },
    {-10, 10,-25,-13,-20, -4, 19,  3, 13,  5,  5,  7, -8,  2,  4,  2,
       3, -1, -1, -9, 14, 10,  9, 14,  3,  3, -6,  0, -5,  4,  1, -1 },
    { -9, 15,-18,-17,  4,-11,  6,  7,-12,  8, -1,-11,  2,  3,  7, 16,
      -3, -9,  7,-12, 23,  0,  6,  7,-14, -9,  8,  1, -2,  6, -2, -1 },
    { -6,  9,-16,-26,-14,-11,  9, -6,  5, -2, 13, 17, 21,  7, 18,-19,
       6,-23, -2,-15, -2,  2,-10, -8,  2,  1, -2,  4, -3, -4, -5, -4 },
    {  0,  6, -5,-28,-17,-32,  2,-10, 11,  3, -5,  9, 10,  3, 11, 11,
      -3, 12, -2,  2,  4, -6,  9, -4, -4, -4, -4, -9,  2,  0,  2,  4 },
    {  0, -8,-18,-34, -9, -7, -4,-11, 10, 15, 11, -1, -8, 15,  6,-13,
       9,  2, -4,-12,  0, -1, 19, 12,  6,  5,  0, -3,-10,-12,  3, -5 },
    {-10,  6, -9,-17,-12,-11,  9, -6, 11, 11, 18, -7,  0, 16,  4,  2,
      -6,  3,-12, -1,  0,  1, -5,-22, -2,-12,  0,  6, 17,  5,  5,  6 },
    { 12, -5,  7,  1, -5, -2, -1,  2,  2, -4, -3, -3, -3, -2,-29, 11,
       5,-13,-73, 24, 12,  4,-14,-10,  5,  1,  0,-11, -7, -7,  7,  3 },
    { 10, -3, -1, -3,  4,-11, -5, -2, -8,  7,  9,  2, -8, -6,  6,  7,
      21, 17,-54, 47,-14,-10, 14, 19, 13, 21, -4,  3,  1,  2, -4,  2 },
    {-12,  4,-16,-12,  5, -9, -4, 19, -7,-22,-22,-17,  3,  0, -6,  8,
      23, -4,-55,-28,  2,-26,  2,  1,  4,  0,-13,  6,  0, 10, -7,-11 },
    {  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  3, -1,
      35, -1,-67,-35,-24,-24, -6,  2,  2, -2,  1,  3,  2,  0, -1,  1 },
    {  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  0,  5,  0,
      41, -4,-73,-15, 18,  4, 17,  8, -1,-16, -1, -2,  1,  0,  0,  0 },
    { -4, -4,  4,  6, -1,  2,-16,-10,-15,-10, 21, -2, -6, -2, 14, -7,
      10, -5,-55, 34,-12, 11,-13, -2,  2, 28,-26,  0,  7,  4, 21, -7 },
    {  2,  1, 15,-22, 10, -3, 14, -6, -2, 15, -2, -7, 20,  6,-15, -7,
      23, 10,-60,  8, -4, 29,-22,  2,-13,  9,-10, 12, -1, -3,  4,  7 },
    {  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1,  0, -1, -2, 11, -5,
     -21,-11,-60,-27,-17,-39,  6, 36,  0, -8,  2,  2,  0,  0, -2,  3 },
    {  2, -5,  9,-17, -1,  2, -3, -6,  8, 12,  7, -6,-33,-11,-14,-40,
      10, 36,-46,  0,-19,  5,  0,-10,  3, 12, -6, -8,  6,-12, -7,  1 },
    {  1,  1,  0,  0,  0,  0,  1,  0,  0,  0, -1,  0,  1,  0, -2,  0,
       4, -2,-87, -3, -2,  2, -2, 20,  2,  6, -1,  6,  0,  0,  2, -1 },
    {  1,  0,  0,  0,  0, -1,  0,  0,  0,  0,  0,  0, -1,  0,  0,  1,
       1,  7,-76, 41, -7,-24,  0, -6,  3,  6,  0, -2, -1,  1,  0,  0 },
    {  0, -3,  4,  2,  3,  2,  2,  0,  3, -1,  4,  0, -1,  4, -2, -4,
     -32,-11,-64,-29, -9,-43,  2,-11, -1, -7,  0, -4, -2, -2, -2,  2 },
    { 10,-20,  3, -3, 13, 13,  0, -4,  2,  7, -8,  7, -2,  2,-20,-20,
     -19,  3,-47,-18,-16, -6,-15,-42,-17, 14, -6,  8, 12,-10, 11,-12 },
    { -3, -2, -2, -1, -1,  4, -3, -1, -6, -2,  3,  2, -3,  6, -1, -9,
      10, 13,-68, -9, 26,  3,  5,  3,-21, 10,-15, 21,-22, 19, 11,-14 },
    {  1,  5, 18,-19,-29,-13, -2, 18,-10, 20,  2, 10,-10, 11,  1,  8,
     -16,-17,-41, 10,-14,-25,  0,-14,-19, 17,  7,-12, 14,-11, 14,  5 },
    {  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  3, -1,-43,  5,
       6,-12,-48, 19,  8,-38, -8, -3, 22,-21,-10, 15, 20, -9, -5,  8 },
    {  0,  0,  0,  0, -1,  1, -1,  0,  0,  0,  0,  0,  0,  0,  6, -3,
      22,-14,-71,-24, -2,-33, 23,  7, -8,  7, -3,  2, -4,  1, -8, -2 },
    {  1,  0, -1,  2,  0, -2,  0,  0, -1,  0,  4,  0, 26, -1, 10,-11,
     -17,-32,-58, 14,-14,-11, -2, 15,  2, -8, 12, 10, -9, 13,-33,-14 },
    { 15,-17,-19,  7, -8,-15,-32,-22,  7, 12, 18,  0,  0,-15, -4, 16,
      37, -2,-46, 11,  2, -8,-10, -8, 14,  9, -4,  5,  7,-17,  4,  3 },
    {  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -2,  0,
      -5,  3,-85, 23, -9,-17, -2, -2,  0,  0,  0,  0,  0,  0,  0,  0 },
    {  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -2,  0,
      -5,  3,-85, 23, -9,-17, -2, -2,  0,  0,  0,  0,  0,  0,  0,  0 },
    {  1,  0,  0,  0,  0, -1,  0,  0,  0,  0,  0,  0, -1,  0,  0,  1,
       1,  7,-76, 41, -7,-24,  0, -6,  3,  6,  0, -2, -1,  1,  0,  0 },
    {  0, -3,  4,  2,  3,  2,  2,  0,  3, -1,  4,  0, -1,  4, -2, -4,
     -32,-11,-64,-29, -9,-43,  2,-11, -1, -7,  0, -4, -2, -2, -2,  2 },
    { 10,-20,  3, -3, 13, 13,  0, -4,  2,  7, -8,  7, -2,  2,-20,-20,
     -19,  3,-47,-18,-16, -6,-15,-42,-17, 14, -6,  8, 12,-10, 11,-12 },
    { -3, -2, -2, -1, -1,  4, -3, -1, -6, -2,  3,  2, -3,  6, -1, -9,
      10, 13,-68, -9, 26,  3,  5,  3,-21, 10,-15, 21,-22, 19, 11,-14 },
    {  1,  5, 18,-19,-29,-13, -2, 18,-10, 20,  2, 10,-10, 11,  1,  8,
     -16,-17,-41, 10,-14,-25,  0,-14,-19, 17,  7,-12, 14,-11, 14,  5 },
    {  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  3, -1,-43,  5,
       6,-12,-48, 19,  8,-38, -8, -3, 22,-21,-10, 15, 20, -9, -5,  8 },
    {  0,  0,  0,  0, -1,  1, -1,  0,  0,  0,  0,  0,  0,  0,  6, -3,
      22,-14,-71,-24, -2,-33, 23,  7, -8,  7, -3,  2, -4,  1, -8, -2 },
    {  1,  0, -1,  2,  0, -2,  0,  0, -1,  0,  4,  0, 26, -1, 10,-11,
     -17,-32,-58, 14,-14,-11, -2, 15,  2, -8, 12, 10, -9, 13,-33,-14 },
    { 15,-17,-19,  7, -8,-15,-32,-22,  7, 12, 18,  0,  0,-15, -4, 16,
      37, -2,-46, 11,  2, -8,-10, -8, 14,  9, -4,  5,  7,-17,  4,  3 },
    {  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -2,  0,
      -5,  3,-85, 23, -9,-17, -2, -2,  0,  0,  0,  0,  0,  0,  0,  0 },
    {  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -2,  0,
      -5,  3,-85, 23, -9,-17, -2, -2,  0,  0,  0,  0,  0,  0,  0,  0 },
    { 16, 65, -2, -2,  4,  3,  0, -7,  3,  1,  3,  1,  0,  5,  1, -5,
       0,  2, -1,  3,  0,  0, -1, -2,  6,  0, -2,  0,  0, -1,  1,  1 },
    {  5, 37, -4,  8, -4, -1,  9, 17,  6, -7,  5, -1, 11,  6, -4,  7,
      -2,  4,  1, -3, 11,  3,  3, -9,  6,  0, -2, -4, -5,  4,-12,-11 },
    { 15, 24,-14,  2,  6, 17, 26,  5,  8, 11, -9, -7, -6, -8,  3, -5,
       9, 10, -3, 10,  0,  1,  4, -9,  4,  9,  3,  0,  4,  0, -5,  3 },
    {  9, 36, -9, -8,  7,  7,  4,  3, -1,-16, -2,  7, -5, -6,  6, 12,
     -11,-12,  9, -1, -3, -9, 12,  6, -6,  2,  2,  5,  0,  5,  6, -6 },
    { 25, 39, -5, 24,  3, 10,  3, -6, 13, -8,  3, -7,  2,-10, -5,  2,
      -2,  3,  5, -2,  1,  5, -2,  3, -4,  1, -5, -4,  0,  1, -2,  0 },
    { 16, 27, -1,  0,-14,  6,  4, -5,  7, -2, -6,  0, -3, -5,  2, -1,
      -1,-19,  5, -8,  0, 11, 12,  5,  0,  3, 10,  6,-14, 14,-13,-15 },
    { 12, 23,-14,  2,  1,  4, -3, 16,  7, -8,  2, -8,  8,  6, -8, -7,
      -3,  0,  2,  8,-13,  7, 13, -6, -4,  6,-13,-16, 14, 11, -7,  5 },
    { 16, 28, -7, -1,  6, -3,  9,  0, -7,  3,  0,  3,-12, 20,  8,  9,
       8, 23,  8,-13, -2,  4,  9,  3, -5, 13,  5, -2, 12, 14,  5, -1 },
    { 19, 37, 19,  5,  7,  5, 10,  5, 19, 10, 14,  0,  2,  5,  1, -4,
      -4,  2,  2, -5, -2, -1,  2, -6, -4, -4, -5, -3,  2, -2, -2, -2 },
    { 24, 21,  1,-11,-10, 17,-14, 14,  6, -1, -6, -1,  0,-13, -1,-12,
      -2, -5,  6, -4,-12, 14,  5, -2, -8, -8, 15, -7,-30,-12,  4,  0 },
    { 11, 26, -3,  3,  5, -1, -2,  3, -2, 10, 15, -4, 10,-28, 10,-17,
      -8,  1,  2, -7, -1, -6,-15, -1,  4,  5, -7,  9,  0, -5, -4,  4 },
    { 18, 32,  1,  2, -7,  4, 15,  2, -9, -2, 12,-11,  7, 11, 13,  2,
       0,  5,  9,-10, 16,  3, -3,  5, -9,-23,  2, -2, -1,  5,  2, 11 },
    { 35, 24,-20,  2,  4, -1,  5, 14,-10, -9,  8, -7,  0,  5, -7, -7,
      11,  1,  5,  3,  2,  0, -2,  3,  0,  1,  4,  0, -2, -8,  0, -4 },
    {  9, 35, -1,  2, -1,-19, -3, 12, -1,  8,  8,-13, -1, -2,  2,  5,
      -8, -1, 13, -2, 11,  1,  0,-10,  0, -3, -7,  2,  1,-12,  3, 12 },
    { 20, 27,-12,-12,  7,  4, -1,-13, -1, -9,  2, 13,-11,  5,  7, -9,
       9,  1,  1,  8, -9,  0, -6,  7,  4,  2, -2,  7,  3, -2,  1, -9 },
    {  8, 37,-20, -5,  0,-21, 10, -8,  3, 19, -9,  7, -3, -8, 10, -2,
       0,  5,  6, -4, -2, -1,  0, -7,  6,  1,  0,  4, -5,  6, -8,  2 },
    {  8, 27,  1, -3, -5,  1,  6,  0, 15,  2, 17, -1,  3,-17, 10,  5,
       5, -6, -6,  6,-10, 18, -5,  0,  0, 13,  7, 10, -5, -6, -2, -4 },
    { 14, 29,-20, -4, -3,  1, -5, -1,  2, 12,-10, -3,  4,-18,  4, 14,
      -4, -1, -9, 15, -2,  2, -5, -3,  2,  9, -2,-14, -3,  4, -4, -7 },
    { 23, 23,-23,-11, 27,  4,  4, -1,  7,  0, -5,  9,  2,-11,  3,  7,
      -2, -5,  2, -7, -7, 13, -3, -6,  2,  3,  3, -4, -1, -8,  5, -2 },
    { 16, 26, -6,  8, -9, -1, -2, -1, -8,  4, -2,  0,-12,  9, -1,  0,
     -17, -9, 30, -5,-15,-16,-13,  0, 10,-11, -7, -3, -1,  0,-11, -2 },
    { 12, 32, -4, -5, 10, 19,-10,  4,-12,  5, -6,  9,-12, -6, -6, -8,
       4,  1,  3,  0,  8,  0, -3, -4, -7, -4, 10,  8,  6,  5, -1,  4 },
    { 46, 42, -3,-14, -2, -6,  6, -2, -5, -1, -3, -3,  1, -1,  3,  1,
       1,  4, -1,  2,  3,  1, -2,  6,  0, -1, -2,  4, -2, -1,  2,  2 },
    {  9, 33,-13,  4,-11,  3, -8, 22, 12, -2,  4,  0,-16,  5,  4, -1,
       7, -6, -9,  1,  7,  5,  0, -5,  5, -1, 10,  3, -2, -1,  3, -2 },
    {  9, 30,  6, -3,  6,  1, -7,  5, 11, 14,  7,  1,  0,  2,  2, -1,
       8,  7, -6,-13,-10, -2,  1, -6, 10,  7,  6,  5, -2, -5, -1,-16 },
    {  9, 28,-11,-10,  9,-10, 15,  8,  4,  9, -4, -7,  0, -5,  9,  8,
      -7,  2,-15,-23,  4, -4,  4, 16, -8, -3,  0, -8, 14,  5, -3, 15 },
    { 17, 26, -5, -5, -1, -8, 20, 18, -7, -2,  4, -7, -8, -5, -4, 16,
       0,  0, -7, -2,-13, -5, -2,  3, 12,  1,  3, -5,  2,  2,  0, -1 },
    { 11, 37,  7,-23,  6, -1, 15, 13,  4, -9,  7,  5,  3, -3, -5, -8,
      -2,  3, -5, -1, -8,  7,  2, 13,  1,  3,  0, -3, -1,  2,  0, -2 },
    { 21, 33,  7, 20, 21,-10,  6, -5, -5, -6, -9,  2, 10,  0,  8, -4,
      10,  2, -2, -2,  0,-10, -6, -2,  0, -5,  3,-11,  3, -9, -3,  1 },
    {  6, 30,-15, -8, 16,  1,  4,  6,  4,  5,  8, -3,  8, -9, -1, -6,
       8,  2, -2,  4, -2,  5, 11,-21,  3,-10, 16,-11, 24, 10, 14, -6 },
    { 15, 36, -3, -9,-20, 12,  0, -7,-18, -4, -8, -9,  9, -7, -3, -1,
       2,  7, -5, -8,  6,  2,  2, -1,  7,  1,  1, -3,  3, -4, -8,  1 },
    { 16, 34, 21,  3, -9, 10,  7,  9, -7,  1, -4, -9, -4, -5, -5,  3,
       3,-19,  1,  5,  4, -2, -6, -5,-10,-11, -8, -2,  2, -5, -8, -7 },
    { 28, 29, -3, 18, -2,  0, -6, 12, -2, 10,-11, -4,-13,-12, -6, -4,
       0,  4, -1, -8,  6,  4, 12, 11, 10, 10, -3, -6,  1,  2,  1,  7 },
    {  3,  8, 22, -8,  3, 36, -8, -1,  9,  6,-13,-14,  8, -1,  1,  2,
      -2, -8,  0,  3,  1,  2, -1,  5, -1, -8,  0, -2,  2,  2, -1,  1 },
    {  0,  6,  0,  0,  4, 13, -7,-16, -6, 15,-14,-21, -9,-10,-10, -6,
     -21,  5,  4,  2, 12,  4, 12, 11, -4, -6, -6,-10, -7,-18,  1,  4 },
    { -1,  3, 10,  1, -1, 15,  4, -7,-16,  3,  0,-22, 10,  2, -3, -2,
      13,  5, -8, 16, -5,  4,  0,-11,-10,-22,  0, -4,-17,  5,  2,  1 },
    { 12,  8, -4, -9, 14, 40,-21,  0,  1,-15,-10,-12, 12,  6,-10,  2,
       8,  6,-12,-10,-11,  1,  0,-11,  2,  1, 13,  0,  6,  3,  8,  4 },
    {-10,  3,  5, -4, -3,  3,  0, -9,  2,  8,-22,-23, 17,  8,-17, -3,
      14, -8, -4,  1, -8,  3,  0,  5, -1, -3, -2, -4,  1,-10,  0, -2 },
    {  0, -1,  5, -7,  4, 12, -2,  0, -7,  2,-16,-15, 12, 21, -7, -4,
       7, -7,-11,-15, -7, -9, -5, -8,  0, -6,  8, -3, -8, 22, -7, -9 },
    {  7, 19,  4, -9, 24, 22,  2, -6,  8, 13,-14,-20, -4, 11,  8, -4,
      -1,  2,  0, -7,  5,-17, -3,  3, -6,  5,  3,  4, -5, -7, -3, 14 },
    { -2,  6,  2,  8, -2,  5, -4, -2,-10,  3,-45,-30, -3, -3,-12, -4,
      -3, -3, -1,  9, -6, -6,  5, -4,  0,  5, -1, -2, -1,  0, -6, -1 },
    { -3, 14,-16,-10, 10,  0, -2,-40, -9, 12,  2,-19, 15, -4,  4,  3,
       3, -4,  7,  1, -4, -5,  0,  4, -1,  0, -9, -2, -4, -1, -2,  0 },
    {  7, 16,  2, -7,  8,  2,  0,  1,  5, 21,-10,-26,  7,  2, -9, -7,
      -3,-16,  8,  5,  5, -6, 10,  4,-14, -6,  5,  3, -2, -2, -4,  1 },
    { -9, 14, -1,  3,  3, 11,  1, -5, -3, 13,-16,-18, 20,  6, -5,  0,
      -3,  2,  8,  4,-19, -9, 12,  0, -8,  2,  2,  1,  6, 13, -7,-11 },
    {  2,  5, 16, -4, 19, 15,  4,  0,-11,  7,-10,-10,-16, 18,-11,-12,
      -9, -4,  7, -4, -4,-17,  1,  1, -8, -3, -3,  5, -2, -6,-11, -5 },
    {  2, 12,  0, -9,-10, 14,  6,  2, -3,  2,-12,-28, 12,  1, -1,  2,
       0, -3, -4,  7, 16,  5, -7,  8, -4, -3, -1,  3,-12,  4,-17, -5 },
    { -4,  7, 11,  6,  1, 14, -4, -6,  5,  5, -6,-24, 23, -9,-15, 13,
      -7, -9,-15, 10, -1,  8, -5,  1, 12,  6,  2,  0,  4, -2,  9,-10 },
    {  1,  5, 11,  3,  6, 12, -3,  8,-21,  5, -7,-20, 12, -2, -9, -3,
      17, -7, -8, -9,-14,  3,-13, 18, -8,  9,  2, -8,  4, -8, -5, -2 },
    { -3, -3, -1,  5, -2, 15,  3,  2,  1, -8,  1,-39, -6, 13,-13,  0,
      -2, -5, -6, -3,  0, -5, -2, 15, -9,  5, -3, -6, -2,  7,  0,-13 },
    {  2,  8,  5,-12,-13, 22,  8,-16, 11,  5, -2,-32, -2, -4, 11,  5,
       5, -6,  1,  3,  1,  5,  3,  6, -5,  4,  4, -8,  8,  4,  1,  3 },
    { 13,  9,  5, -4,  9, 18,-11,  2, -1, 15,-10,-19, -2, 14,  0,-10,
       1,  1,-18,  3,  2, -6, -8, 20,  7, -8, 16,  9,  9,-13, -3, -2 },
    {-13, 11, 11, -9,-10, 13, -3,-18,  2, 10,  5,-21,  6, 15,-11,-21,
       3, 14,  0,-12,  9, -1, -2, -4,  3, -3, -9, -8, -5, -2, -8,  2 },
    {  3,  3, 11,  4,  0, 13,  1, -8, 10, 13, -6,-26,  2, 12, -3, -5,
      12, -2,  1,  8, -7,-17,-19,  5, 10,  7, -3,  2, -3,  0,  5,  0 },
    {  5,  0,  3, -3, -9,  5,-15, -5, -5, 17, -5,-31,  0, 13, 13,  5,
      -1, -6,-14,  7, -8,  9,-14, -2,-16, -4, -4, -6,  6, -6,-10,  6 },
    { 13,  3,  1,  7, -3,  4, -1, -2, -1,  4, -8,-32, -1, -4,  0,  3,
     -10,  7, 10,-10,  4, -1,  6,  2,-16, -9,  4,  3, 13,-23, -3, -4 },
    {  4, 11, -4, -9,  4, 11,-12,-12,-12,  6,  1,-28, -3, 14, 18, -2,
     -12,  7, 15, -3, -5, -7, -3,  2, -6,  4,  4, -2, -5, -3,  2,-13 },
    {  8,  7, -7,  0, 13,  7, -8, -7,  8, 36,-10,-22,  3, 23, -3,-10,
      -3, 11,  1, -7,  3,  3, -1, -7, -4,  2,  3,  2,  5,  3, -4, -1 },
    { -1,  1, 13,  1, -6, -1, -6, -9,-18, 17, -5,-37, -1, -1, -6, -4,
       1, -6,-15,  2, 17, -9,  0, -3,  0,  4,  0, -5,  0,  4,  1, -5 },
    {  0, 14,  5,  0, -7,  2, -6, 17, -6, -9,  7,-16, -5, 23,-14,-13,
       8,-15, 11, 10,-11,-13,-33, -5, -2,  1,  6,  8,  0,-13, -9,  5 },
    { 11,  7, -2, -8,  9, 11, 25,-14,  7,  3, -1,-33, 14,  8, -6,-19,
       3,  3,  2, -1, -3, -1, -2,-10, -3,  1,  2,  1,  4,  2, -3,  4 },
    { -2,  8,  4, -2,  9, 13, -4, -2,-15, -3, 19,-37,  9, 25, -9,  2,
      -5, -2, -2, -4,  4,  2,  2,  0,  3,  3,  3,  5, -2, -3, -4, -3 },
    { 10, 13, -1,-15,  4,  6,-18, -4, 25,  1,-23,-17, 15, 13, -8, -8,
       7,  4, -5,  3,  6,  9, -7,  6,  0, -5,  8,  0, -6, -1, -2, -2 },
    {  1,  3,  9, -5, 27, 15, -9,-31, -1, 23, -2, -9,  1,  8, -1, -7,
      -2, -8, -4, -4, -2, -1,  3,  5,  0,  0, -1,  1, -7,  7, -3, -3 },
    { -8,  7,  3, -6,  8,  3,-11, -2, 36, 14,  1,-30,  6, 10,-12, -6,
      -6, -2, -4, -3, -5,  0,  9,  4, -5, -5, -8, 12,  4, -3,  1, -8 },
    { -2,  9, 33,  0, 12, -3, -7, -4, -4, -1,  6,-25, 11, -6, -9,-11,
      -2, -4, -2,  6, -1, -3, -6, 15, -6,  3, 10, -4,  1,  0,  5,  8 },
    {-22,-21, -9,-19, -5, -7,-12,-15, -8,  9,-19, 14, -7, -4,  5, -8,
      -2,  7,  1, -3,  4, -4,  6, 11,  2,  6, -3, -5,  2, -2,  0, -3 },
    {-32,-13,  3,-24,  3, -8,  4,  1,-10, 14,-15,  0,  4,  6, -1,  6,
       7, -1,  6,  4, -3,-17,  1,  4, -6, -1,  1,  0,  3,  3, -7, -4 },
    {-32,-11,  7, -8,-12, 13, -5,-22, -4, 12,-16,  2,  0,  4,  0,  1,
       0,  6, -5, -8,  2,  6,  5,  0, -3, -6,  5,  6,  5,  5, 13, -4 },
    {-44,-33,  6, -4,  2,  0, -9, 10,  3,  4,  7,  0, -1,  7,  5,  1,
       1, -3,  1,  6, -1,  0,  2,  3, -4,  0,  0,  1,  0, -1, -2, -1 },
    {-30,-18,-24, -8,  5,  0, -2, 14,  7,  0,  1, 12,  6,  4, -9,  7,
       5,  7,-11, -5,  1, -8, -1,  2,  2, -9,  7, -1,  7,  5,  6,  6 },
    {-22,-20,-13, -9, 20, -3, 10, -8,  6, -4,  2, -7, 10,  8,  0, -1,
       2, -3,  6,-19,  2,  4,  3,  3, -7,  2, -1, -6,  1,  1,  6, -2 },
    {-27, -8, -1,  3, -1,-11, 24,  4, -1,  1, -8,  8,  5,-11, 15, -3,
     -15, -1, -1,-13, -1,  1, -5,  5,  2,  3, -9,  0,  4,  3, -7,  6 },
    {-33,-16, -1, -8, 10,-23,  6, 13, -1, -3, -9,  0,  5, -7, -5,-12,
      -2,  3,  3,  6, -2, -3,  2, -3,  9, -6, -3, -2,  0,  5, -3, -4 },
    {-22,-17, 11, -3,  3,  1, -1, -5, 17,  2,-15, -2, 10, -9,  6, 14,
     -16,-12, 20, -1, -7,  6, -3,-12,  1, 10,-10, -1,  7, -3, -1, 10 },
    {-28,-13,  1, -3, -1, -1,  0,  3,  3,  5,  1, 10,-10, -3,  7,  2,
       4, 19, -1, -1, 10,  5, -8,  1, 11,-15, -4, -3, -5,  4,-13,  3 },
    {-22,-13, 42,-20,  5,-13,  7,-11,  1,  1, -1,  1,  6,  3,  6,-11,
       3,  3, -2,  0, -4,  4, -3, -1, -5,  2,  0,  0, -9, -1,  4,  4 },
    {-26,-15, -2, -6, -4, -2, 16,  8, 21,  8,  1, -3,-10,  7, -8,-12,
      -5, 12, -9,  3, -2, -3, 18,  1,-12,-15, -4,  5, -3,  0, 12,  7 },
    {-26,-16,  5,  6, 14, -3, 15,  6,  1, -7,-13, 16,-15,  5, 11, -2,
       9, -7, -4, -2,  0,  0, -2,  7, -8, -6, -5,  2,  7, -3,  2, 12 },
    {-31,-17, -8,-30,  4, 14,  6, -6,  6,-11,  0,  3, -4,  0,  0, -4,
       0, -4,  1,  4,  3,  4,  0, -5,  3,  2,  2,  0,  2,  1,  3,  5 },
    {-61,-10,  4, 10,  4,  7,  0, -3,  0,  1,  0, -3,  0,  1,  0, -2,
      -1,  1,  2, -2,  4, -3,  1,  1, -1,  1, -2, -4, -4,  4,  0,  0 },
    {-28,-13, -8, -4,  3, -3,  2,  1, 11, 14,  3,  9,  1, 13,  3,  5,
      -3, -2, -2,-12,-14, -9,-11,-15,-12, -5, -4,-12,  3, -3,  0, -5 },
    {-41,  0, 12,-24, 13,  4,  5, 16, -5, -4,  0,  0, 13, -4,  1, -9,
       9, -6, -1,  6, -2,  5,  2,  9,  6, -9, -8,  8, -2, -3, -6, -4 },
    {-26,-19, -2,-15,  4,-14,  6,  0, 26, 20,  8,  9,  9,  3, -4, -5,
      -8,  1,  0, -1,  5,  9,  3,  4,  4,  7,  1,  3, -2, -2,-10,  0 },
    {-29,-18,  9, -4,  1, -5,-14,-12,  5,-10, -5,  4, -5,  0, -1, -1,
       4, -5,  7,-16,-11,  2,  7,-15,  2, -4,  6, -4, -6,  7, -3,  7 },
    {-27,-16,  9,-14,  3, -8,  9,  0,  7, -4, -3, -7,  0,-10, -1,  2,
       1, -2, 15,-10, 14,  7,  6, 17,  3, -4,  3,-10,  8, -8,  3, 11 },
    {-21,-20, -8, -8,  4,  5, -3, -2,  0, -5, 14,-10, 11, -4, 13,  0,
       5,-11, 19,-18, 18,  3, -5, -3, -4, -8, 11,-10, 10,  3,  4, -9 },
    {-35,-15, 13,-12,  4,  0, -2, -4,-12, -3, -8,-24, -7,  1,  7,  8,
      -3,  0, -2, -1,  3, -2, -2, -6,  8,  1,  0,  1, -6, -1,  2, -6 },
    {-19,-14, 13,-10,  9, -1,  1,  3,-12,  5,-16,  7, 13,  9,  4, -4,
       6, -5,  4,  9, -3, 17, -4, 12,-11, -6, -5, -6, 13,  2,  7, -9 },
    {-34, -8, -4,  1,  2, -1,  3,  6,-20,-11,  8, -1,  4,  2, -9,  4,
      -4, -5, 16, 10, -4, 14,-13,  1, -6,  0,  2,-10,  0, -3, -3,  7 },
    {-36,-10, -8, -3,  2, -2, 14, -4, -1, -7, -4, 10, -1, -3, 15,-11,
       0,  2,  3, -1,  4,  0,  8, -1,  0, 18,-11, -5, 15, -5, 13,-12 },
    {-22,-13, 14,-20, 15, 25, 16, 10,  8, -2,-10, -5, -1, -8, 11,  8,
      -1, -2, -4,  1,  2, -1, -7,  0,  0,  0, -3,  0,  2, -1,  0,  2 },
    {-31,-22,  7,  6, -2,  5,-20, 14, -6,  7,  0, 14,  3, -7,  3, -6,
      -2,  1, -3, -5,  1,-10,  1,-24,  6, -2,  3, -7,  1, -7,  8,  7 },
    {-25,-20, -3, -9, 10,  6, 12,  7,  5,  4, -3,  6, -1, -5, -6, -8,
       3,  5,  6,  5,-10, 10, -4,-15,-15, -2, -9,  2, 18,  1,  8, 12 },
    {-24,-19, -2, -4, -7, 11,  6,  9, 16,  2, -7, 18,  6, -7,  6,  6,
      -2, -9,  3, 12, -2,  3, -1,  6,  7,  8,  0,  8,-11,  8,  4,  2 },
    {-26,-20,-12,-12, -2, -3,  1, -5, -1, -2,  0,  3,  7,  9, -2,  2,
       9, 22, 13,  4, -4, -1, -2,-14,  5, 15, -8, -5, -7,-11,-14, -6 },
    {-21,-18, -1, -4,  0,  3,  7, -2, 10,  8, -8, -1, 15,  1, -9,  3,
       1,  3, -5, -2,  2,  4,  0, -1, 10,  2,-19, -8,  8, 30, -7,  8 },
    {-25, -6, 26,  4, -8,  4, -2, 21,  5, -4,-16,  5, 13,  4,-10, -1,
      -6, -2,  2,-10,-13,  1,  3, -3, -6, -8,  2, 11,  1, -7,  0,  5 },
    {  0, -1, -2, 19,-12,-48, -6, 11,  8, -2, -4, -2, -7,  5, -3,  2,
      -2, -1, -1, -7,  0, -3, -3, -4, -4,  4,  1,  3, -3, -1, -2, -5 },
    {-11, -8,-28, 18, 16,-24, -8, 19,  4,  8,-12,  9, -4, -2,  4, -7,
       6,  2,  3,  3, -4,  0,  1, -6, -4, -2,  2,  6,  0, -3,  1,-16 },
    { -9, -5,-26,  7, -3,-37,-16, -2,  2, -7,  4,-13,  0, -4, -6, -5,
      -6, -4,  0,  3,  4, -3, -4, -4,  4, -3,  9, -4, -2,  2,  7, -4 },
    {  2,  9,-18,  7, 29,-24, -1,  7, 14, 10,  3, -3, -2, -5,  6,-10,
      -6, -3, -8,  0,  5,  1,  4,  3,-12,  2,  6,  1,  3,  4,  1, -3 },
    {-20,  2,  8, 20, -9,-24, -4, 18,  3, 11, -1,-11,  6,  9, -1, -3,
       1, -1,-15,  3, 15,  9,  3,  2,-13,  2, -8,  8,  1, -1,  1, -8 },
    {-12,  5,-11,  6, 19,-26,-17, -6,  4, 14,  6, -8,  9,  5, -6, -5,
       2, -1, 20,  1,-11,-10,-18, 20, -7,  0, -3,  4,  2,  0, 10,  4 },
    {-15,  1, -2, 13, -8,-21,-22,  4,  4,  3,  3, -7,-31,  4,-10,-14,
       0,  8,  4,  5,  8, 11,  2, -8,  6,  7,  0, -2,  6,  8,  8,  7 },
    {-13,-10, -9, 12, 19,-16, -3, -2,  9,  2, 11,-29, -1,  9,  4, -3,
       1,-10,-10, 16,  1,  7, -7, -6, -4, -1, -5,  3,  6,  0,  3,  1 },
    {-17, -1, -5, 19, 12, -9,-21, -5,  2, 12, -7, -7, -3,  8,  7, -2,
       6, -9, -9,  1, -4,  1,  1,  3,-14,  2, -8,  0, 10,  1,-12, -6 },
    {-13, -5,  8, 15,  0,-20, -2, 20,  8, -8,  8,-19, 12, 10,  2,-11,
       0, 12,  1,-11,  0,-11,-15,  5,-11,  2,  4, -4,-11,  5, -4, -5 },
    {  3,-11, -7,  8,  0,-17,-26, 15, 19, -7, 10, -9, -5, -5, 14,-25,
       0, -8,  2, -9, -3,  9,  1, -6,  4, -4,  3, -9, -1,  6,  2,  2 },
    {-12,  5,  5,  9, 14,-18,-19,  4,  2, 16, 14,-21,-15, -9, -1, 16,
      12,-11,-10, -5, -7,  4, 15, -8, -5, -1,  1, 14, 13, -7, -1, -4 },
    {-10, -5, -1,  8,  7,-23,-10, 14,  6, 11, 10,-16, -3, 16,  6,  0,
       0,  9,  6, -2, -7,  1, 22,  5,  3, -8,  0,  3, -2,-10,  3,  0 },
    { -2,-14,  2, 16, 15,-17,-17,  6, 19,  4,-10,-15, -1, 15, 11,-14,
      -8,  5,  8,  8, -2, -8,-11, 10, 10, -8,-14,  2, 13,  4, -2,-12 },
    {-10,  3,  6,  4, 19,-23,-19,  1,  4, -9,-30,  3, -6, 18,  0,  2,
       0,-11,  0,  3,  7, -2,  8,  5,  2, -3,  6, -9,  1, -4,  7, -6 },
    {  9,  5, -2, 21, 20,-33,-13,  7,-10,  8,  8,-15, -6, -4,  1,  5,
       3,  7, -2, -9, -1,  4, -6,  1,  0,  9, -1, -5,  2,  1, -3,  3 },
    { -9, -3,  3, 15, -3,-30, -7, -7,-25,  6,  2, -6,  1, 19,  1,-12,
       1, -8,-13,  9, 13,  1,  8,  2,  5, 15, -2,  3, -9,  0, -4,  4 },
    { -6,-12,-17, 25, 22,-13,-10,  9,  2, 11, -7,-16,  4,  6,  1,  0,
       0, 18, -4, -5,  4, -2, -1, -5,  0, -4,  6,  1,  6, -1,  7,  0 },
    { -1,  0,-10,  8,  8,-27,  0, -2, 29, 16, -2, -4,  9, -1,  2,  0,
       6, 10,  6,  4,  2, -7,  9,-18,  3,  3,  3,-10, 17, 10,  9, -6 },
    { -3,-12, -6, 11, 20,-32,  5, 21,  3, -4, -9,  2,-10,  1,  7, -4,
       5,  0,  0, -1, -8, -9, -7,  4,-10,  5,  0,  2, -5,  4,  9,  1 },
    { -5, -1, -5,  1,  2,-19,-13,  1,  6, 12,  2,-16,-17, 11, 10, 13,
      16,-12,-11,  3, -6,  0,  6,  4, -3,  1,  8,  2,  5,-11,  3,-14 },
    {-19,  5, 10, 11,  2,-23, -9, 16, -2,  7,  0,-11, -7, 10,  6, -7,
      26,-15, -4,  8,  6, -4,  7, -9,-15,  1,  8, -4,  4,  2,-12, 16 },
    {-11,  1, 11, -4,  1,-31,-13, -1,  8,  5,  4, -2,  0, 13,  7,-17,
       7,-10, -6,  1,  4, -1,  2, -9, -4,  9,  3,  3, -4, -5,  3,  4 },
    { -3,  1, 10, -1,  0,-15,-22,  4, 40,-11, -4, -3,-14,  9, 11, -1,
       9, -1, -6,  6,  3, -6,  0,  0,-12,  7, -2,  0,  9,  3,  1,  3 },
    { -1, -1, -1, 14,  8,-24,-14, -8,  5,  8,  5,-12,-17,  8,  2,  7,
      10, -8,  0,  4, -6, -6,-10,  8,  4,-12,  3, -9,-12,  5,  4, -3 },
    { -5,  1,-11,  8,  9,-24,  0,  2,  2, 14,-12,-13,  1,  6,  7,  0,
       7, -6,  9, 26, 11,-14,  8, 10,  1,  9,  0, 11, -2,  6,  2,-10 },
    {-13,  1,  4, 34, 19,-17,-15,  0,  3, -2, -7, -1,  0, -3, -3, -1,
       1, -1,-10,  8,  5,  0, -8,  4,-17,  9, -2,  0,  0,  6,  2, -3 },
    { -6, -4,  1,  2,  2,-14,-29,  0,  9, 34, -3, -5,-14,  6,-10, -9,
      -5, -1,  0,  3,  3,  0,  1, -1, -2, -1, -1, -3, -3, -4,  3, -3 },
    { -4,  6,  3, 14, 14, -8,-29, 31, 11, 14, -4, -5, -6, 10,  6, -9,
      -1,-11, -7,  1,  7,  4,  1, -6,  4,  0, 10, -7, -5, -1,  2,  4 },
    { -4, -4, -2, 14,  6,-32, -6,-14, 14, -5,-11, 10,-18, -4,  6, -8,
       9,  5, -4,  1, -4,  5, -2, -9,  3,  5,  2,-10, -6,-17,  3, 17 },
    {-16,  9, 21, 19,  4,-20,-17, 14,  9, 15, -6,-17, -1,  1,  6, -3,
       1,  1,  8, -3, -6,  6,  9,  4,  9, -9, -5,  1, -1,  0, -1,  2 },
    { -7, -5,  3, 19,  1,-20, -9, 14, 21, -7,-18, -9, 26, -7,-17, -7,
      12,  6,  0, -9, -6, 14,  9, -9, -8,  4, 15, -7, -9, -1,  9,  1 },
    {-20, 30, -6, 11, 24, -4,  0, -6, -2,  8, -4, 12, -8,-17,  0,  5,
      -4,  1, -1,  3, -3,  5,  3,  3,  7, -2, -3, -2,  4,  0,  0, -1 },
    {-35, 17,  6,  1, -9, -1,-16,  3,-20,-13,  8,  7, -4, -7, -4,-20,
       7, 12, -5,  5, -5,-11, 12, -1, 15, -9, -6, 16, -4, -9,-13,  4 },
    {-21, 36,-19,  9,  0, -7, -8,  9, -4, -3,  3,  0,  7, -8, -2, -2,
     -11, 13, -1,  5, -3,  7,  2,  3, -1, -2, -5,  1, -1, -2, -5, -3 },
    {-12, 33, -4,  1,-12, -9,  0,-13, -1,  2, -8,  4,-10,  6,-16, -7,
      -1, -4,-10, 15, -1,  0, -5, -8,  5,  5, -3,  0,  2, -7,  1, -7 },
    {-14, 32,  5, -7,-15,  3, -5,  8, 14,  5,  9, 13,  3, 18, -3,  7,
       4,-10,-10, 10, -1,  2,  0, -2,-11,  5, -3, -4,  2,  2,  7,  4 },
    {-14, 34,  1, 20, -1,-12,  0, -3, -7, -4,  7, 18,  9, -3, 14, -7,
      -9,-20, -7, -4,-13, 12,  1, 12,  5, -6,  2, -4,  0,-15,  1,  3 },
    {-21, 23,  7, -8,  3,-13, -3,  0, -6, -2, -7,  6,-12,  9, -6, -2,
      -2, -4, -1,  6,  9,  5, -9, 15,  0,  8, -8,  7,  6,-15,  3, -5 },
    {-27, 32, -1, -4, -2,  4,-10, 12, -3,  8, 13,  7,  0,-15,  4, -2,
       3,  5,  7, -4,  9,-12, -1, -2, -1, -4,  0, -4,  2, -5,  6, -6 },
    {-17, 29, 15,  0, -1, -4,-10, 13, 12, -1, -8,-10,-10,  4,  7, -2,
       6, -5,-13, 19,  6,  1, -7,  2, -9, -2, 12, -4, -8, -3,  2,  4 },
    {-38, 27, 16,-15, -6,  3, -7, -4,  0, -1,  6, -2, -3, -6,  6, -6,
      -3,  0,  2,  0, -4,  6,  1, -1,  0,  4, -1,  3,  4,  1, -2,  5 },
    {-33, 40, -4,  2,  1,  0,  0,-10,-14,  0, -7,  4, -1,  3, -2,  5,
       7,  6, -1,  4,  1,  3,  1, -7,  1, -4,  5,  7,  0,  4,  3, -4 },
    {-20, 25, 12, -4, 16, -4,  2,  2,-14, -2, -3, 29, -1,  1,  3,  1,
       9, -5,  2, -8, -3,  1, -7, -2, -7,  1,  0,  4, 16, -2, -1, -1 },
    {-10, 30, 17,  3, -5, -2,  0, -5,-22,  4,  5,  5, -3,-18, -6, 10,
      -5, -7,  2,  8,  7, -7,-11, -2,  0, -3,  3,  2, 11, -4,  4, -4 },
    {-11, 30, 11,  4, -3, -8,  1, -2,  4, 18,  3,  1, -1,  0, -8, -4,
      -3, 10, 13, 14,  5, -5,  1,  1,-10,  2, 15,  4,  9, -1, -5, -3 },
    {-17, 32, 18,-18, -3, -5,  6, 10,  1,-15, -5,  9,  8,-12,-10, -6,
      11,  9, -5, -8, -7, 10,  5,-10,-14, -4, -3,  1,  9,-11,  2,  1 },
    {-13, 28,-11, -1,  2,-16, -2,  7,-24,  0,  3,  6,  3, -1, -8, -7,
     -12,  2,  2,-20, 10,  4,  0,-13, -2, -2,  1,  8,-14,  0,  4,  1 },
    {-14, 23, 12,  8,  8,-26,  2, -4,-14, 13,-14, 15,  3, -9, -1,-13,
     -10, -2,-10,  6,-16, 12,  8,  0,  9,-10, -7, -4, -4,  7, -8,  8 },
    {-20, 45, 10,-14,  4, 16,  8, -9,  1, -8, 10,  5, -7, -2,  2, -5,
      -1,  0, -5,  4, -6, -2,  4,  1,  3,  4, -4,  2, -2, -2,  5,  1 },
    {-20, 26, -4,  1,  7,  4, -8,  1, -5,-13,  2, 13, -7, -3,  6, -6,
      22,  0,  5, 11, -4,-11,  8, -9,  2, -2, -4, -2,  2,-13, -4, -8 },
    {-28, 18, 17,  3, -8,-23,-16, -6,  5,-10, 14, 10,  5, -1, -8,  4,
      -2, 13, -3, -2,  3,  4,  3, -2, -3, -4,  0,  1,  3,  4,  0,  4 },
    {-12, 32, -6,-16, 18, 12,-16,  0,  7, 13, -4,  5, -8, -1, -3,  4,
       6, -2, -1,-13,  4, -1,  3, 12, -3,-10,  1,  6,  8,-11, -2,  4 },
    {-18, 26,  2,  5,  0, -9,-17, 14,  5,  1,  7, -3, -8, -3, 11,  7,
      -5,-12, -8,  7,  0, -7,  2,-12, -9, 13,-11,  9,  6,-11, -5, 11 },
    {-24, 22,-15, -9,  8,  1, -7,-12, -9,  3, 11, 15, 14,-11, 12,-15,
      -5,  7, -2,  0, -8,  3,  3, -1,  2, 11,-11, 14, -6, 13,  1, -6 },
    {-20, 28, 18, -4, -6, -5, 12, 14,  2, 10,-13, -6, -8, -6,-13, -1,
     -26, 22, -3,-14,  6,  0, 10,-15,-13, -9,  6, -7,  1, -5, -4, -1 },
    {-19, 26, -8, -3,-14, -6, -9, -4, -8, 15, -8,  3,-12, -4, -2, -7,
      -5,  3, 13, -3, -4,-25,  4, -1,  5,-12, -1,-13,  5,  2,  0,  6 },
    {-18, 43, 14, -8,  1,-23, -2, -2,  1,  3, -7,  0,  0,  8, -1, -3,
      -5,  1,  5,  2,  0, -2, -2, -2,  1, -1, -1, -7,  0,  3, -3,  9 },
    {-11, 30, 10,-14,  3,  1, 10,-11,  1, -7, -4, 14,  2,  1, -9,  1,
     -11, -2, -7,  5,-11,  1,  3, 14,  1,-16, -8,  3, -5,  7, -4,  4 },
    {-18, 24,  6,  3,  8,  7,-22, -7, -7,  3, -8,  4, 23,  9,  3, -1,
       3,  6,  7, -1, -7,  6,  4,  1, -3,  1, -6, -1,  2, -7,  3,  3 },
    {-15, 38, -7, -1,-11,  2,-17,-24, 24,  8,  7, -4, -5,  2,  2, -7,
       1,  4,  0, -9,  5,  0, -1,  1, -1, -5, -6,  3,  0,  7,  8, -3 },
    {-14, 22,  1, -5,  9,-12, -9, -5, -6,  5,  7,  8, -1, -4, -9, -3,
     -33,-16, -9, -1, 12,-11, 17, -7, -3, -1, -7,  3,  2, -3, 16, -4 },
    {-14, 20,  6,  4,-10, -4, -4, -4,  1, -7,  2,  6,  8,-12,  4,  1,
      -1, 12, 10,  3,-14,-10, -3, 18, -2, 33, -5,-17, 17, -5,  9,  7 },
    {-12, 23, 13,  0,-11, -8,-11, 12, -5, -9,-16, 11,  6,  4, 12, -5,
       5,-13,  7,-12, -3,  1,  2, 12,  1, -4, -1,  5,  4, 11,-12, -3 },
    { 15,  2, 14,  7,  1,  2,  1, 12, 10, 23,  4,  6,-20,-10,  4, 26,
      -6, 13,  4,  3,  2,-11,  5, -7,-10,  4,  9,  1, 10, -4, 11,  4 },
    { 17, 15, 31, 17, 18, 16, 11, 24,  2,  4,  2,  3, -8, -3,  7, -3,
      -5, -7, -2, -6, -4, -5, -4, -1, -4, -2, -5, -6,  2, -1,  4, -2 },
    { 16,  8, 15, 14,  3,  7, 21,  9,  8, 15, 21,  6,  8, 12,  5, -5,
       7, -3, 10,  2, -3,  8,  6,  0,  5,  5,  6, -3,  2,  4,  0, -5 },
    {  5, -4,  6, 12,  6, 13, 24, 17, -5, 17, -1, -6, -7,-10, -8,-18,
       3, -2,  2,  7,-15,-11, 12, -3, -2, -2, -4, -7,  2,  0,  5,  5 },
    { 10, -6,  8, 11, 12, 20, 22,-11, -3, 15, -3, 15, -2, -2,  0,  2,
       5, -8,  4, -5, -9, -4, -1,  2, -1, -3,  1,  3, 13, -1,  9,  7 },
    { -5,  8,  5, 11, 14, -5, 14, -9,  2, 35,  8, 15,  1, -2,  2, -2,
       4, -9, -3,-14,-12, -2, -2, -4, -2, -8, -3,  1, -6,  3, 10,  0 },
    { 16,  0, -6, 15, -3,  4,  4,  3,  3, 20,  5, -4, 10,  9, -9, -3,
     -10, -2, -7, 11,-11,-10, 17, -1,  3,-15,  2,  9,-15,-10, 16, 10 },
    { 14,  4, -7, 19,  3,  0, 19,  8, 16, 34, -9,  6,-13, -1,  6,  5,
      -1, -2,  4,  3,  2,  1,  1, -1,  0, -7,  2, -1,  1,  0,  6, -1 },
    {  1,  6,  9, 13,  9, 10, 15, 16, 10, 18, 13, 17,  3, -1, -7,  2,
     -15,-11,-10, -4,-13, -6,-17,-13, -6,-14,  1,-10,  6,  4, -1, -1 },
    { 13,  1,  7, 10, 14, 13, -7,  5,  5, 28, 14, 14, -2,  2,  3, -3,
     -13, -4, 10, -9, 19, -4, -3,  4, -5, -5,  0,  5, -5,  0,  3, -4 },
    {  1,  0,  6, 22,  9, 18, 18, -3,  5, 10, 12, -2,  1, -3, -8,-12,
       9,-10, -7,  1, -1, 19,  0,  2, -8,-11,-10,  9,  6, 11,  0,  3 },
    { 10, 11, 19, 44,  0, 14,  1, -7,  6, 22,  2, -1,  9,  2,  0, -4,
       4,  0, -6, -6,  3,  0,  0, -2,  2, -5,  1, -2,  0,  1,  1,  1 },
    {  5,  7,  0, 32, 30, 26,  5,  4, -7, -3, 15, -6,  3,-10,  7,  6,
      -8, -7,  2,-13, -5, -1, -3,  7,  3, -2, -8,  0,  6,  4,  5,  0 },
    {  9,  8, -2,  4,  2, 11,  4, 29, -5, 14,  8, -5,-14,  8,  0,  9,
       8,-10,  5,-15, -6, -9,  9, -1, 18,-16,  9,-21, -3,-13, -2,  8 },
    { 25,  7, -9, 23, 20, 18,  6, 16, -9,  8,  8, -5, 11, 13, -8,  7,
       4, 10, -2, -1, -7, -9, -7, -9, -4,  1,  1, -5,-10,  8,  4, -5 },
    {  9,  2, 16, 14, -5, 14,  1,  0,-21, 17, -1,  9, 12, -3, -3,  4,
      -4, 14, 10,  3,  0,-10,  7,  4,  4,-11,  2,  4, -1, -3,  9, -1 },
    { 17,  8, 11, 26, 15, -3, 14, -1, 12,  9, 10, -8,  8,-18,-11, -3,
     -14, -7,  7, -3, -3, -4,  1, -7, -3,  2, -3, 16, 10,  0,  9,  6 },
    {  9,  8,  3,  8, 18, 14, 11,  1, 10,  6,  1, -4,-16, -2, 14, -2,
       1,  8, 12, 14,  3, -3,  8,  8, 12,-15,  3, -3,  3, -2, 14, 10 },
    { 22, -3,-11, 13, -7, 11,  4, 11,  3, 14,  0, -6, -2, -9,  4,  2,
      -2,  0, -5,-27,-10,  3, -1,  5,  8,-24, -3,-11, -3,  2, 11, -1 },
    { 19,  2,  8, 36,  5, -6,  3, 15, -3, -4, -5, 14,-10,  1,-12,-10,
      -3, -4,  3, -2,  1, -8,  4,  3,  5, -3,  0,  4,  8, -2,  8,  4 },
    {  8, 14, 15,  9, -4, 10,  5, 11,  9, 10,  8,  9,-15, 15,  6, -8,
     -10,-13,  5, -8,-20,-13, -6,-11, -1, -3, -6, -4, -1,  0, 13, 15 },
    { -2, -1,  9, 12,  2,  2, 13,  3,-23, 33, 15,  2, -4, -1,  3,  8,
       8,  6,  6, -7,  8,  6,  9, -1,  3, -8,  0, -4,  1, -8, 11, -1 },
    {  6,  5, -6, 16,  2, -3, 31, 21, -9, 12,  0, -1, -4,  1,-12,  3,
     -13,-18,  2,-11, -9,  2, -8, -6, 11, -3, -1,  0, -1,  0, 13,  5 },
    {  5, -1,  2,  0, 25,  5, 10, 16, -5, 21, 14, 12, 13,  2, -5,  5,
       5, -3, -2,-14,  0,-12,  7, 11, -1, -7, 19, -1, -1, -1,  8, -1 },
    { 10,  7,  3, 11,  0,  8, 22,  3,  3, 19, -4, 12, 15,  9,  5, 15,
       2,  1,  2,-10,-10,  0,  2, -1,  0,  1,-12, -1, 21, 16,  9, -7 },
    { 11, -4, -5, 24, -7, 11, 20, 11,-15, 18,  5,-13,-15,  0, -5,  9,
       1,  0, -1, -9,  4, -8,  6, -8,  1, -2, -7, 20,  9,  3,  9,  3 },
    { 20,  0,-12, -6,  9, 31,  9, 12,  8, 27, 15,  7,-16,  5, -3, -7,
      -1, -9, -2, -7, -3,  4, -8, -3,  3, -6, -2, -2, -3, -6, -1,  2 },
    {  6, -6, 48,  8, -3, 19, 12, 11, -7,  2,  3,  0, -1,  1,  8, -4,
       4, -6,  0, -4, -4, -3,  3,  6,  3,-13, -8,  5, -3, -7,  8,  5 },
    {  7, -2,  6, 11, 12,  2, 14,  4, -5, 12,  2,  9,  4,  2,  0, -1,
       2,  0,-15, -9,-16, -2,  8,-17, -5,-22,-19, -5, -1,-10,  1, -2 },
    { 11, -9,  3, 12,  6,  6,  1, 17, -6, 19, 14,  7, -7, -1, -1, -9,
       9,-11,-17,  0, -6, 16,  0,  1,  9,-24,  3,  3, -9, -3,  3, -2 },
    {  9,  0,  1,  8,  1,  7,  2, -5, -3,  8, -1,  7,  2,  6, -3, -6,
       5, -2,  6, -2, -4, -3,  0, -3, 13,-50,  1, -2,  2,  4,  4,  3 },
    {  7,  0, 26, 21, -4,  2, 17,  8,  7, 11, -7,  1, -1,-15, -1,-15,
     -11, -4,-17, -4,  1, -7,  3,  6,  3, -9,  2,  3,  6, 10,  6, 12 },
    {  1, -2,  2, -1,-10, -4,  6, -3, -5, -2, -8,  2,  2,  2,  8,  0,
       1,  1,  6,  0, 11, 13,  3,  4,  0,-12, 11, -5, 19, 20,  2,  5 },
    {  5,  3,-13, -2,  1,-12, 11, -7,-12,  7, 10,  0,  7,  0, -2,  4,
      -6, -9,-11,-12,-23, 12, 10, -3,  0,  6, 19, -1, 24, 18,  9, 12 },
    {  6, -3,  2,  5,  2,  2, -2, -5, -8,-11, -4,  3, -8, -4,  5, -3,
     -16, -4,  3,-12, -4,  3, 32,  7,  2,  8, 32,-18, -1, 12,  1,  7 },
    {  0, -8, -1,  0, -8,  7, -8, -1, -1,  4,-12, -1,  3,  0,  1,-18,
       8,  8,-14,-10,-11, 19,  9,  5, -7,  6,  8, -4, 26, 12, -1,  6 },
    {  3,  5,-14,  7, 14,  8, 20,-13,-16,-10, -2, 17, -7,  4, -8, -9,
      14, -5,  3, -4,-12,  7, 14,-10,-19,-20, 35,  8, 13, 14, -2,  9 },
    { -2, -4, -1,  1, -3,  0, -1,  1,  2,  2,  6,  0,  0,  4,  5, -2,
       3,  3,  3, -2, -7, -3, -3, -1,  6, -2, 29, 22, 13, 34,  0, 14 },
    { -3, -9,  3,  1,  5, -4,  2,  0,  7, -9,  0,  2, -5, -3,  0,  6,
      -1, -1, -1,  2,  2,  4,  8,  7, 20, -6,  7, 16, 33, 20,  6, -1 },
    {-11,  1, -3, -3,-11,  3, -9,-25, -1,-16,  4, -8, 15,  1, -2,  7,
       8, 23,  2, 18,-13, 16,  3, -7,  6,  3, 16, -8, 12, 16,  3,  4 },
    {  0,  5,  5, -5,  1, -1,  2, -3, -2,  1,-13,  2,  2, 10,  6,  7,
      18, 18,  7,  9,  8,  9, 21, 14,  7, 12, 15, 14, 15, 12, 11,  5 },
    {  1, -5, 11, -2, 17,  8,  3,  0, -1,  6, 11, -7,  6,  6,  7,  5,
     -15, 14,  1, 11,  4, 10, 12,  1,  2,  4, 30,  1, 11,  1,  6, 13 },
    {  2,  4,  3, -7,  5,  8,-11,  7, -5,  9,-10,  6,  8,-10, -3, 10,
       1,-29, -4,-26,  5, -8, 13,  4,  3,  6, 35,  1,  3,  6,  3,  0 },
    { -2,  1,  0,  0, -1, -3, -7, -3, -9, -3, -1, -6,  3,  4,  4,  0,
       5, -1, -2, -2, -1, -4,-10,  8,  0, -6, 10, -4, 46, 12,  2, 28 },
    {  4, -1,  4,  1,  0,  4, -2, -2, -2, -1,  2, -4,  1,  5,  0, -3,
       1,  1, -2,  0,  1, -2, -1, -1,  3, -6, 35,-11, 13, 53, -3, -1 },
    { -5, -2,  0,-13,-16,  5,-12,-11,  1,-30,  3,-18,-24, -8, -5,-19,
       1, -3, -8,  7, -7, -8, 15,-19,  4, 10, 30, 24,  6,  1, -9, 10 },
    { -4,  8, -7, -4, -6, 12, -1, -9, -4,  2, -9,  3,  2, -2,  4,  2,
      22,  9,  4, -5,  0,  5, -2, -9, -3,  1, 18,-12, 18, 16,  4, 16 },
    { -5, -8, -3, -5, -3,  6, -7, -3, -2, -5, -3,  1,  2,  2,  4, -6,
      10,  3, 12, -3, 20,  0, 27, -4, 16,  5, 18, -3, 23,  4, 12, 11 },
    {  0,  1,  0,  1, -2,  1,  2,  1, -1,  0, -2,  2, -2, -4,  1, -2,
      -2, -1, -5, -2,  0,  0, -2,  2,  9,  7, 63,  5, 12, -1,  1,  0 },
    {  4, -3, -7, -5,-11, -5,-12,-10,-10,-12,-15,-12,-14,-14,  1,  1,
      10,-10, 16,  6,  2,  9, 11,  9,  9,  8, 12, -1, 13, 12,  6,  3 },
    {  7, -3, -2,  4,  6, -8,  2, -3,-12, -5, -9, -8,-10, 15, -2, -4,
       8,  9,  7,-13,-18, 34, -5,  7, 12, 22, 16,-11, 13, 25,-15,-11 },
    { -3, -2,  0, -4,  1,  0, -3,-13, -7, 13, 12, -7,-10, 13, 19,  6,
      16, 15,-12,-15, -3, 34,  1,  5,  1, -9, 11, 21,  8, 17, -5, -6 },
    {  3, -5,  0, -4,  0,  4,-11,  4, -7, -3, -1, -8,  3, -2,  2,  1,
      11,  5,  6, 14, -3,  2, -4, -7,  0, 31, 15, -2, 24, 11,  5,  4 },
    { -1, -4, -9,  5, -8,-18, -4, -9,-20,-18,  7,-14,-16,  3,  8, -3,
      29, 11,-13,-13,  7,  1, 17,  6,  6, 21, 11,  1, 14, -8,  2,  5 },
    { -3,  8,-10, -6, 12,  2,  1,  3,  3,  3,  3, -6, -8,-14, 15, -5,
      16,  4, 16,  0,  7, -1,  0, 16,  2,  1, 22,  4, 19, 13,-11,  1 },
    {  2, -3, 10, 20, -4, -1, -8,  5, -8, -9, -6, -2, -4, -7,  8,-10,
       0,  8, -6,  1, -8, 14, 13,  5, 17, -6, 26, -1,  7, -1,  0, 12 },
    { -4, -7,-31, -2, -7, -1,  5, -5, -5,-12,  4, -7, -6,  3, 15, -2,
       5, -2,  7, -1, 10,  7,  8, -1, 14, 20, 14,  9, 16, 16,  8, 24 },
    { -7,  0, -3, -6,  1,  3,-13, -6, -4, -4, -5, -9, -1,-10, -4, -8,
       2,  0, -1,  1, 24, 24, 21, 31,  5,  2, 11, 12,  7,  4,  3,  6 },
    { -3, -5,  6, -4, -3, -1,  2, -1, -2,  1,  0, -8, -1,  2,  0, -4,
       6, 22, -1, -5,  8, 12, -1, -2, 28, 27, 20,-27, 14,  1,  2, -3 },
    {  1, -5, -2, -2,  6, -2,  9,  1, -2, -5,  3,  4, 11,  5,  2,  8,
      -3, -1,  1, -2, -3, -5,  5,  8, 49, 12,  8, -3,  9, 20, 12, 17 },
    { -6,  0,  1,  7,  0,  9, -2, -4,  8,  0, -2,-10,  0,  7, 21, -1,
       0,  1, 17, -7, -5,  2,  4, 16, -2, 17, 14,-20, 15, 14,  4, 15 },
    {  0,  3, -4,  9, -4,  0,  6,  4, -6, -6, -5, -7,  2, -9,-10, -2,
      -5,  0, -3,-21,  9, 14,-11, 13, 29,  2, 25,  4, 22, -1,  2, -3 },
    {  2, 12,-11,  2, 16,  9, -4,  7,  1,-10,-15, 11, -4,  3, -2,  4,
       4, -5,-10,  1,  4, 19,-15,  6, -4, -2, 30, -7, 11, 21,-12,  5 },
    { -2, -3, -2,  4, -1, -5, -3, -7, -5,  1,  0, -6,  1, -6,  7,  0,
       8, -7, -3, -2,  2, 14,  2, -3,-26, -1, 26, 22, 32,  1, -2,  6 },
    {  1,-38, -1,-20, -2, -3, -6, -4,  2,  2,  7,  0,  3,  5,  3, 10,
       6,  1, -3, -5,  7,  5, -5, -4,  8,  3,  1,-14, -1, -9, -5, -4 },
    { -5,-26, -7,-19,-10, -5,-11,  5,-11,-25, -8,-14, -9,-16, -8, -6,
     -17,-14, -1, -1,  6,  2,  2,  2,  3,  0,  2,  8, -8,  3,  0, -3 },
    { 17,-49, -3,-23, -1, 11,  7,  3,  4, -4,  0,  0, -1,  4,  2,  4,
      -2, -4,  2, -2, -1, -2,  2,  0,  0, -1,  0,  0,  1,  2,  0,  0 },
    {  4,-34, -6, -9,  1, 21, -7,  3, -2, -1, -3, 18,  2,-16,  7, -3,
       8,  7, -5,  7,  2,  4,  8, -6, -7, -2, -5, -1,  4,  1,  2, -4 },
    {  5,-29, 13, -2,-14,  3,  1, 18,-15,  4, -8,  8,-10,  8,  2,  1,
      -8, 15,  3,-10, -4, -4, -2,  0, -3, -4,  2, -3, -4, -3, 12, -6 },
    { 13,-20,  3,-18,-17,  4,-14, 13, 28, 11, -8, -6, 16,  6,  0, 10,
       3,  4, -9, 13,  5, -7, 12, -5,  0, -7,  5,  1,  3,  3,  2,  1 },
    {  3,-27, -5,-11,-21,-11,-12,  0, -5,  7,-22,  1,  3,  5,  0, -5,
       8,  7,  1, -5, -7,  2, -5,  4,  1,  3, -8, -2,  0,  4, -2,  6 },
    { 31,-45,  0, -1,-12,  1,  2, -6,  4,  3, -1,  3,  3,  0,  5,  3,
      -5, 12,  4,  6,  2,  1, -2,  1,  3,  2,  5,  2,  2,  2,  3, -1 },
    {  9,-45,  6,  5, -1,-17, -2, 18, -3,  2,  0,  1,  0, -1, 10,  8,
      -7, -2, -5, -8,  6, -1,  0,  4,  6, -3, 12, -1, -2,  0,  5, -7 },
    {  3,-26, -2,-12,-12,  2,-10, 16, -3, 12,  4,  5, 11,  8,-16,-17,
      -2, -3, -3,  2,  5, -9, 13,  1, 10, 11,  3,  5, -2,  2,  2, -7 },
    {  8,-26, 32, -7, -5, 22,  2, 14,-10, -8, -7,  3,  3,  7,  0, -5,
       0, -1, -3,  0,  8,  4, -5, -7,  6, -1,  4,  8,  1,  1,  7, -6 },
    {  4,-31,  2,-14,  2,  0,  1,  8, -6, -1, 17, -3, 13, -6,  5,-10,
      -2,-10, -2,-10, -3,  7,  1,  5, -8,  8,-14, -3,-15,  7,-10, -6 },
    { 16,-27, 13, -4,-23,  7, -9,  6, -7,  5,  4,  2, -1, -3, 23,-18,
       7,  0, -3,  4, -3,  9, -6, -2, -1,  8, -6,  2,  6, -3,  2, -2 },
    { -1,-35, -2, -8, 11, -1, -7, -3, -2, 11,  7,  6, -6,-10,  9,  6,
      -3, -5, -6, -3,  9, 16,-16, -9,-20, 12,  3,  5, -3,  1, -9,  4 },
    {  2,-24,  1,-12,-16,  5, -4,  3, -4, -1,-11,-11, -8,-14, 14, 10,
      -8, 20,  8, -3,-11,  1,  1, -4, -4, -7, -3, 15,  2, -6, -2,  7 },
    {  9,-21,  2,-19, -7, -5, -8, 25,  3, 17,  5, -3,  9,-12,  8,  2,
      -4,  3,  3,  1, 11, -9, -4, -3,  4,  3,-22,  6,  4,  6, 11, -5 },
    { 16,-23, 13,-17,-21,-12,  5,  9,-20,  7,  6, -6,  0,  2, -9,  6,
      -6,-13, -7, -1,  5, -3,  5, -7,-10,  1,  0,  8, -9, 11,  0, -8 },
    { 10,-26, -9, -7,-19, -4,  6, 16, -7,  5, -4,  4,  8,  0,  4, -1,
       6, -7,  1, -8,-11, 10,-14,  0,-16,  6, -3,  5, -1, 14, 12,  1 },
    {  8,-27, 12,-14, -1, -1,-19, 10,-11, 21,-14,  9, -8, -3,  8, -1,
      12,-13,  3, -4, -2,  0, -9,  0, -7,  2, -3, 12,  1, -3,  3,  1 },
    { 18,-20,-14,-14,-16, -3,-24,  6,-17,  2, -3,-11,  2, -3, 12, 10,
      10,  1, 10,  7,  8,  5,  5,  4, -1,  7,  2,  2,  0,  4,  7,  0 },
    {  0,-30,  9,-16,-18, 15, 12, -3,  4, -4, -5,-11, -4,-12,-10,  0,
       2, -2, -4, -1,  2,  0, -1, -6,  2, -3,  4, -5,  7,  3,  5,  7 },
    { 25,-24, -1, -6, -9,  6,-13, -2,  3, 15, -3, 11,  4, -8,-11,  2,
       0, -9, -2,  7,  4,  8,  5, -8,  5,  6, -1,-11,-15, -5,  0, 11 },
    {  0,-34, -7,-11, -7,  9, -3, 19,  4, -8,  3,-11, 11, -3, -9, 12,
       9,  9,  2,  1, -7,  1, -3,  0, -6, -2, -1,  3,  0, -7, -2, -5 },
    {  6,-34, -4, -5, -3, -9,  2,  9, -1,  9, -5, -3,-26,-12,  8, -6,
      -7, 11, -8,  4,  4,  1, -1,  0,  8,  9, -4,  7, -1,  1, -3, -1 },
    {  3,-30,  5,  6,-10,  3, -7,  6,  3,  3,-26,-19, -3,  1,  7,  5,
      -4, -5,  6, 10, 13,-10,  4, -7, -4,  5, -3,  9, -6,  3,  9,  5 },
    {  4,-24,  9,-19,  2, -4, -5,  8, -3,  2,  0,-15, -1,  9, -4, 22,
       6,  9,  3,  7, 11, -9,  0, -3,  4,  5, -5, 10, -8,  5, -7, -3 },
    {  8,-27,  7, -3, -1,  2, -9, 13,  7, 12, -4, -6, -6,  5,  0,  7,
       5,  1, 15, -3, -4,  0, -5, -2,  7, -5, -7,  1, -2, 13, -8, 13 },
    { 17,-22,-15,-11, -8, 16,-14, 18,  2, -1, 14, -7, 14, -6, -6, -7,
      -8, 17,  6,  4,  4, -7, -5, -9,-14, -6, -1,  9, -3,  1,  6, -5 },
    { 25,-30,  2,-12,-13, 18,-18, 16,  8, -3, 10, -8, -3, -1, -6,  3,
      -5, -7,  4,  6,  7,  1,  1,-11, -5,  6,  2, -4,  9, -1, -5, -2 },
    {  7,-23,  7,-15, -1, -3, -1,  0,-10, 12,  2,  5, -4,  0,  4,  6,
      -1,  5, -9, -1, -1, -7,  1, 17,  9,-17,-16,  8,  4,-14, 11, 14 },
    {  0,-31,  7,-13,  3,-11, -7,  6,  1,-11,  8, -7, 15, -3, 16,-11,
      -1,-15, 16, -3,  5,  0, -2, -2, -6, 11,  5,  6,  5, -5,  6,  3 },
    { 13,-24, -2,-20,-10,  7, -3, -1, 15,  2,  6, -5, -7,-10,-20,  1,
      -4, 14,  8, -2,  3,-13, -3,  1, -4,  1, -3,  2,  8, -7, 16, -4 },
    {  1, -2, -2, -3, -4, -7,  0,  3,  6,  7,  3,  2,  1, -2, -1,  0,
      -6,  4,  2, -4, -3, -4,  5,  9,  5,  0, -3, -3, -4, -7,-31,-50 },
    { -1, -3,  7,  2, -1,  2,  4,  6,  0, 10, -2,  0,-20, -6, -3,  9,
     -20,-22, -1, -1, 15,  9,-12, 10,-13,-20, 12,  3,  5,  6, -7,-26 },
    {  0,  4, -2,-14,-12,  6,-13, 11,-10,  3, 22,  6, 16, -2, -5,  1,
      -3,-11,  0, -7,  5, -5,  0,  1, -1, -6,  8,  8, 10,  9, -5,-27 },
    { -5, 10, -2,  7,  9, -9,  5, -9,  5,  4,-15, 14,  1,  3,-10,  5,
       0, -2,  7,  3,-13,  6,  9, -6,  5,-14,-17, -1, 11, 14, -2,-26 },
    {  0,  6, -3,  0, -8,  6,  0,  1,  4, -8,  2, -5,  4,  7, 15, 11,
       9, 19, -2, 14, -8,  7, -1,  3, -3, -3,-10, -2, 12, -2,-12,-29 },
    {-12, -5,  0, -3, -2,  6,  3, -3,  2, -2,  1, 11,  2, -7,  5,  1,
       2, -2,-14,  0, -1, -5,  3,  8,-28,-26,  6, -6,  3,  8,-10,-27 },
    { -1, -3,  6,  2,  4, 15,  1,  0,  2, -2, -2, 13,  3,  6,  0,  6,
      -1, -4, -1, -5,  8, -1,  5, -5,-15, 11, -8, -5, 14, -6,-14,-29 },
    { -5, -6,  0,  1,  0,  6, -3,  2, -5, -1,  5, -3,  2,-10,  3,  4,
       3,  0, 13, -3, -1,  4, -4, -6,  2,  9,  8,  2, -3, 28,-11,-31 },
    {  1, -4,-10, -9, -4, -3,-15, -6,  1,  5, -3, -6,  5, -6,-22, 27,
     -13,  5,  3, -7, -4, 20, -7,-12, -1,-24, -4,-13, -8,-11,-15,-21 },
    { -6, -4, 19, -6,  2, 11, -6,  1, -3,-10,  9, -9, 12,-10,  2,  1,
      -9,  1, 15,  7, -5,  5,-29,-35,  4,-30,  9,  9, 19, 17,  2,-17 },
    { -3,  3, -3,  1,  2,  5, -1,  5, -2, -3,  1, -3, -8,  3, -4, -2,
      -4, -1, 12,  0,  2, -8, -6, -4, 16, -1,-14, -2, 25, -6,-15,-36 },
    {  0, -1,  3, -4, -4, -1,  7, -4,  8,  0, 10,  9, -4,  1, 10, -1,
      -3,-13, -5, -4, -1, -4,  8, 11, 14, -7, -5, 16, 12, 13, -1,-28 },
    {  1, -2,  2, -3, -8, 10,  4,  9, 12,  3,  5,  0,  8, -3, -6,  2,
      16,-11, 11,  0,  1,  6,  1, 18,-10,-16, -1, -4,  5,-14,-15,-20 },
    {  1,-12,  5,  4, -7,  8, -1,-17, -2, -9,-14,-11,  6, -9,  5, -4,
       3, -2,  7, 18, -5,  5,  6, -1,-11, -2,-10, -3,  8, -3, -2,-32 },
    {-12,  5, 20, -5, -6,-11, -6, -6,-13,  4, -6, 19, -8,  2,  3, -9,
      -4, -4, -1,  9, -1, 21, -1,  7, 15,-10, -1, -3,  9, -3,  2,-24 },
    {  0, -3,  2, -6,  4, -1, -9, -2, -1, -3,  6, -1, -5, -6, -5, -8,
       0, -2, -6,  9, -4,  3,  2,-13,  1, -7, 23,-13,  4, -3,-15,-33 },
    { -7,  2,-15, 11,-10, 14,  0,-11,  3, -1, 12, -4, -4,  9, 11,-13,
     -13, -3,-14,  1,  3,  6, -5,  8,  0,  5,  5,-10,  4,  5, -6,-30 },
    { -6,  4,  0, -5,  4,  1, -1, -1,  3,  6,  5, -2, -5,  0, -2,  5,
      -4, -2, -4, -2,  4,  7, -7, -1,  1, -4, -3,-19, 37, 12, 10,-40 },
    { -7,  2, -7,-12, 17, 11, -7,  2,  2,  3,  1, -1,  3,  4, -2, -5,
       9, -9,  6,  4,  9, 12, 11, -5,  2, -1,  0,  9,  5, -7, -2,-24 },
    { -7,  6,  1,  3,  1,  0,  6,  0,  4,-12, -2, -2,  1, -9, 10, -2,
      11, -1, 21,-12, 15, -5, 10, -5,  5, -5, 14, -6,  5, -7, -3,-29 },
    { -2,  0, -5, -2, -3,  1, -3,  0,  4,  2,  3,  0,  2, -2,  7, -2,
       3, -5,  2, -1,  6, -4,  0, -3,  8,-11, 19, -8, 22,-34, 13,-35 },
    { -1, -3, -1,  9, 11, -3, -3, -1,  7, 18, 11, -5,  2,-12,-11, 18,
       9, -5,  1, -6, -9, 12,  1, -3, -3, -9,-14,  9,  9,  8, -6,-26 },
    {  0,  5, -5, -1, -1, -2,  4,  6,  8,  2, -1, -2,  5,  1, -5, -4,
       1,  1, 18,  1,  7,-10,  3, -2, 12, -1,-15,  9, 12,-14, 13,-38 },
    {  3,  0, -8, -1,  0,  8, -9, -3, -8, 16,  3, 16, -5, -9,  0, -1,
      -7, -1, -4, 13,  7,  0,  1,  2, -1,-16,  0, -2,  1,  8, -8,-28 },
    {  7,  9, -5, -3, -2,  2,  0,  3, 11, -6, -4, -2, -2, -5, 28,-18,
      -6,  2, 15,-10,-15,-10, -2,  0, -2, -2,  4, -3,  7, 11,  5,-30 },
    {  9,  0, -7, -1, -4, -7,  2,  2,  9, -2,  2,  3, -8, -6, -6,  3,
     -10,  4, 10,  5, 21, -4, 14,-18,  1,  3,-10, -2,  6, 14, -8,-26 },
    {-14, -1,  2,  3, -3,  7,  1,-22, -1, -1,  0,  1, 12,-14,  3, -5,
       0, 10, -3,  1, -5, 12, -3, 10, -8,-22,-11,-13, -7,-10,-13,-25 },
    { -2, -5, -4, -4, -9,-18,  9, -3, -5, 17, 13,  5,  6, 11,  3,  8,
      20,  4,  2,  9,  8,  5,  6,  1,  7, -7, -6, -2, -7,  0,-17,-23 },
    { -5, -5,  2,  0,  6,  2, -2,  2, -3,  4,  4,  0, -5, -2, -4,  6,
       8, 10, -1,  1, -5,  5,-14, -2,-11,  8,  6, 25,  7, -1,  0,-43 },
    { -4,  0,  4, -2,  7,  0,  3, 17,  5,  2, -5,  1, 21,  3, -2,-10,
     -16, -9,  7,-12,  9, -8,  2,  5, -5,-10, -2,-11, -5, -1, -9,-30 },
    { -2,  3,  1, -4, -1,  0,  8,  1, 12,  4, -1, -1,  3,-17, 13,  9,
       0,  7, -6, -5,  9,  1,  5,  4,-10,-18,  0, 14, 11, -4,-16,-28 },
    { -1,  0,  2, -1,  4,  1, -1,  1, -1, -2, -1, -2,  3,  0,  0, -1,
      -1,  1,  2, -2,  3,  3, -2,  4, -2, -1, -6,  1, -1, -1,  6,-70 },
    {  7,  3,-11, -1, 12, -4,-14,  4,  4, -4,  4, -2,  2,-12, -4, 15,
     -17, -4, -3,  6,  8, -5, 22,-22,  5,-11, 15, -4,  4, -1,-21, -1 },
    { 10, -2,-13, 11,  4, 14,  4,  9,  8,  8, 19, 15, 14, 15,  5, 10,
       8, 15, -5,  4, 14, -8,  1,  1,  2,  1, -1, -3, 21,  8,-29, 13 },
    { -6,  0, -6,  6, -1,  2,  8, -4, -5,  4, -4, -5,  0, -2, -4,  0,
       9, -2,  1, -2, 26,-19, 21,-10,  4,  1, -8,  5, 22,-10,-13, 15 },
    { 11, -5,  1,  0,  6,  3,  7, -2, -2, -3, -5, -1, -2, -6,  1,  1,
      -8, -5,-13, 13, -2, -3, -1, -9,-28,  4,  2,-11, 18,-20,-24,  9 },
    {  7,  4, -3,  6,  6, -6, -7, -5, -7, -4, -4,  0, -7, -5, -6, -5,
       2,-13,-12,  2,  0,  5, 18, 15,-13, -7, 13,-20, 16,-10,-19,  6 },
    {  5, -8, -1,  5, 10,  2, -1,-10,-11, 23,  8, -5, -8,  4, -5, -4,
      -5, -5,-11, -8,  5,  1,  7, -9, -9, -6, 12, 14, 17,-12,-22,  3 },
    { -5, -8, -3,  3, 12, -1,  0, -4, -5,  1,  1,  6,  1,  5, -5,  7,
      -2,  7,  1,  6,  6,  2,  0, -5, 17, -4, -5,-24, 13,-20,-27, 14 },
    { -1,  2, -3,  1, -3,  1, -3,  0, -2,  3, -2,  1,  2, -1, -2, -1,
      -2, -5,  5, -2,  0, -7,  1, -6,  8,  8, 11, -5, 24,-43,-13,  2 },
    { -2,  4,  7, -3, -4,  4, 13, -4,  0,  0, -2,  9,  0, -3, -6,  1,
      -7,  1, -1, 10,  0,  5, -1,-24, 25,-15,  7,  2, 22,-10,-21,  0 },
    { -5,  2,  6, -2, 13,  3,  5,-12,-11, 16,  6, 10, -5,  0, -3,  6,
       5, -5, -5, 10, 12, 10, 11, -7,  8,-14,  2,-15, 13,-14, -8, -3 },
    {  5,  6, -7, -5,  5,  2,  9,  5,  0, -1, -4,  2,  8,  0,  3,  5,
     -12,  3, -3, -6,  2, -1, -5, 14, 11,-20,-21,-25, 24, -1,-10,  6 },
    { -5,  5, -2,  9,  4, -4, -1, -6, 11, -6,  5,  0,  2, -3,  6, -1,
     -17,-18, -4,-13,  9, -1,  9, -7, -4, -8,  2, -3, 12,-31,-18,  5 },
    { -7,-11,  6, -8,  4, -3,-12,  0, -1, -6, -3,  0,  5,  9,  7,  2,
       1, -8, -6,  8,  2, -5,  7, -1, 16,-10, 16,-12, 18, -1,-25,-12 },
    {  3,-12,  1,  2, -2,-18, -8,-15,-10, -9,  2, -7, 11,-11,  2, -1,
      -1, -1, -9, -6,  3,-14, -2, -1,  2,-13, -7, -9, 19, -5,-17,  2 },
    {  7,  1, -8,  7, 17,-13,-10,  5,  7,  1, -6,  4,  9, -4,  0,  3,
       8,  1,-14, -9,  4,  7, -9,  0,  6, -5,-12, -2, 25, -2,-19,  1 },
    {  7, -3,  6, -3,  1,  6, -7,  0, 10,  0,  4, -5,-17, -4,  4, -1,
       0, -3, -7, 19, 24, -1, 21,  8, 10,  9,  8, -1, 23, -2,-18, -2 },
    {  3, -3,  0,  5,  8, -2, -9,  2,  9,  6, 19,  8,  2,  6, -9, -2,
      -4, -3, -8,  7, -7, -8,  5,  4, 26, -6,  7, 18, 24,  0,-13,  4 },
    {  0,-13,-11, -1,  3, -9,  5,  4, -7,  3,  0,  2, -1,  4, -5,  2,
       9, -2,-11, 15,  1,-21,  1, -1,  0,  4,-14, -4, 24,-16,-13,  1 },
    {  1, -9, -8,  0,  0, -4, 11, -1, 14, 16,  0, 17, -2, -9,-12,  0,
      -1,-14, -9,-14,  0, -2, 19,  4,  6,  4,  4,-11,  8,-17,-19, -5 },
    { -3,  1,  2, 12, -4,-18, -1, -4, -7, 14, -3,  2,  0, -7, -8, 12,
      -5, -9, 14, 12, -9, -2,  4, -6,  4, 18, -1,-25, 22,  2,-23, -5 },
    { -2,  0,  0,  0,  1,  3,  5, -1,  5, -2, -2,  2, -3,  0,  1,  2,
       0, -1,  2, -1, -9, -6, -7, -4, -2,  4, -7, -5, 64, -3,-25,  4 },
    { 12, -2, -3,  0,  8, -9, 13, -7,  6, -3,-12, 12, 15, -9, -4,  2,
       9, -4,-12,  3, 14,  1,  7,-15, 15,  0, -6,-12,  0, -3,-20,  6 },
    {  2, -1, -4,  5,  9,  6, -7,  2, -2, -7, -2,  0, -1,-18, -4, -6,
     -15, -5, 11,  5,-10, -1,  2,  7, 12,-19, -7,  8, 21, -4,-15,  4 },
    {  4,  2,  5,  5, -5,  1,  3,  2, -8, 13,  0, -5, -2,-14,-11,  6,
       2, 17,  8,-13, 26, -2,  5,-15, -4,-14, 12, -9, 13,-21,-23, -4 },
    {  2, -3, -2, -3,  3, -2,  6,  9, -9, 13,  4,  2, 12, -3, -3,  1,
     -17,-22, -3,  4,  3, -2,  1, -9,  1, -6, 11,-13, 14,  0,-15,  6 },
    {-16, -4, 17, -2,-20,-11, 11, 10,  5, -8, 16,  2,-17,-14, 11, 11,
      -6,-11, -7, 12, 12,-10, -6,  5,  8, -4, -2, -5, 28,  3,-13,  4 },
    {  0, -3,  3, -7,  6,  8,-12, 20,-19, 18,-11, 10, -5,  0, -9, 11,
       3,  0, -2,  9, -7, -5, 18,  3, -2,-16,  1,  6, 12, -7,-16,  1 },
    {  4,  1,  5, -5, 15,  2, -8,  3,  5,-11, 15, -3,  8, -8, -1,  7,
       4,  7, -2,  6, -9,  5, 12,  2, 33, -2, -6,-18,  4,  0,-18, 11 },
    {  3, -1,  1, -1,  0,  1,  4, -1, -5,  0,  1,  0,  4,  2, -1,  4,
      -3,  2,  0, -2,  4,  6, -1,  6, 42, 19, -4,-37, 19,  1,-15, -4 },
    {  2,  0, -5,  0, 10,  0,  0, -5,  3,  0,  0, -3, -3,  0,  2, -4,
     -10,  2, -6,  4,  4,  1, 27, -7, 17,-34,  5, -9, 15,-16, -7, -5 },
    { -2,  7,  7, -2,  9, -2,-15, 11, 11,  7,  5,  1, 15,  1, -9, 31,
       2,-15,  2,  4,  3,  4, -1, -8,  2, -7,  6,-17, 11,-14,-11,  2 },
    {  1,  1,-11,  9,  9, -6,-14,-11,-10,  8, -3, 11, 16, -9, -8,-13,
      -8,  9,  0,  6,  6, -2, 13, -8, -2,  3, 13, -3, 10, -6,-17,  4 },
    { 14,  5,  4, -6,-12, 10, -7,  8, 21, -8,-30, 15, -2,  1, 11, -9,
      -5,  1,  0, -1, -1, -6, -2,  3, -5,  7,  9,  5, -5,  2,  0,  1 },
    { -1,  2, 20,-17,-15,  3,  3,  7, 11,-17,-13, -6, -3, 18, 17,-15,
      -4, -4, -5, 22, 14,-14, -2,-10, -7, 11,  8, -7, -3,  0, -7, 11 },
    {  7,-11, -7, -8,-14, 22,  5,  2,  6, 13,-12, -2, 10,  3,  0,-21,
      -4, 20,  3, 10, 21,-10,-12,  8, 11,  2, -5,  2,  1,  3, -1, 15 },
    { -1, -2, -1, -2,-13,  8, -4,  0,  7, -2,-17,  8, 18,  5,  3,  8,
      -8, -2,  3, -4, 14,-18,-13, 14, 15,-13, -1, -2,  4, 11,  1, 12 },
    { 13, -6, -4,-16,-17, 16, 21, -2,  5,-11, -9, 19, 21,-17, -3,-17,
       3, 12,  8,-12, -6,  1, -7,  9,  9, -7, -5, -1, -3,  5, -6, -4 },
    { 11,  5, 12,-20, -6, 10,  4, 12,  8, -5,-10, 15, 13, 14, 10,-15,
     -13,  1,  6, 14, 15,-17,-13,  4, -5, 10,  7, -6, -8, -3, -4, 12 },
    { 25, -1,  7, -5, -7, 11,  1, 17, 13,-15,-14, -4,  5,  3,  8, -3,
      -2,  2,  0,  6, 16,-12, -6, -4,  4, -3,  7,-10, -3, -7,-13,  7 },
    { -8, 10, -3,-13,  5,  2,  4,  9,  9,-17,-13,  2, 11,  1,  6, -4,
       8,-10,  4,  1, 19,-15, -4, 12, 31,  7, -5,-17, -4,  9, -2,  7 },
    { 14, -6, -6, -6,-14, 13, 17, -5,  4,-14, -9,  7,  7, -9,  3,-16,
     -15, 11, 11,  6,  4,-11,-19,  3,  5,  8, 13,-14,-14,  3, -4, 12 },
    { -2, -4, 10, -4, -7, -1, 27,  5,  2,-16,-18,  4, 12, -2, -3, -2,
      -1,  1, -8,-12,  3, -4,  8, 15,  2,  4,  9,-13,-14,  9, -7,  5 },
    {  4,  2,-10, -5, -7,  2,  1,  4, -1, -6,-15,  6,  1, 10,  5,-10,
      -9, -1, 13, -3,  5,-21,-11,  8,  8,  5, 27,-21,-18, -5, -1, 15 },
    { 11,  1,-16, -8,-11,  0,  5, -8,-12,-13,-17, 22,  4, -6, -1,-18,
     -10,  0, 19,  2, -2, -8, -7, -3,  2, -2, -9,-17, -5,  4,  4, 10 },
    {  8, -6,-19, -5, -4, 12, 14, 15, 10, -9, -1, -9, 19, 12,  0, -1,
       2,  4,  7,  9, 16,-16,-14,  9, -4,  3,  1,  0, -2, 10, -1, -1 },
    { 12, -8, 12, -9,  0, 25,  7,  9,  2,-31, -9, -4, 15,  4, -5,  1,
     -10, 11,  8, 10,  0, -6,  5, 11, -1, -6,  4,-10, -9,  6,  4,  5 },
    { 14,  6,-17, -2, 17, 12, -9,  2,  0,-25,-14,  5, 20, 14,  8,-20,
       5,  2, -2, -3,  9,-13, -3, -1, -6,  3,  7, -6,  0,  2,  3,  1 },
    {  8,  4,-15, -3, 10, 18, -4, 13,  8,-22,-10,  9, 19,-15,  7, -5,
     -13, 12, -4,  9,  2, -9, -6,  0,  2,  1, -9, -6,  6,  1, -1, 11 },
    {  4,  1,  4, -5,-10, 18,  7,  2, -4, -9,-11,  0, 32, -7,  4,-16,
      -1,  0,  6,  3,  6, -3,-14, 16,  9, -2,  7, -1,  0, -5,  5, -3 },
    { -3,  2,  3, -8, -6,  4,  6,  2,  4,-12,-15,  2,  8,  8,  9, -3,
     -18,  6, 34, 11, 12,-15, -1,  2,  9,  2, -4, -4,  2,  4,  2, -3 },
    { 18, -6,-12, -8, -1, 15, 20, -4, -1,-11, -5,  6,  6,-11,-15, -7,
       3,  7, 10,  2,  8,-10, -5,  8, 15, -5,  5,-17,-13, 13, 11,  7 },
    {  8, -4, -6, -1,-14, -3,  6, -2,  1, -5, -1, 10, 10,-15,  5,  0,
     -10, -4, -3,  7, -4,-19,-15, 27, 11, 18,  3,-19, -2,  6,  0, 12 },
    { 12,  0, -5,  0,  4, -5,  1,  5, 10, -7,-11, 21, 29,  1, -2,  1,
      -4,-11, -1, 13, 11,-20, -1,  4,  4,  4, -5,  6,-13, -2, 11,  9 },
    {  2, -7, -7, -3,-10, -1, 20, 12,  1,-19,-19, -1,  5,  4, -7,-25,
      14,  1, -3,  2, 12, -4, -3, -3, -2,  6,  1,  0,  3,  2,  5, -1 },
    { 12, -8,  3,-12,-10, 10, 13,  0, 23,-14,-18, 10,  0, 15,  3,-12,
      -3, -5,  5, -4,  2,-14,-10,  8,  2,  9, -1,-11, -3,  5, 13,  2 },
    {  9, -6,  7, -7,-30, 17,  6, 13,  1,-14,  0, -1,  6, -9,  8,  3,
      -4,  0, -1, -7, -5,-13,-19, -3, -4,  4, -6, -2,-13,  1, -2,  3 },
    { 10,  1,  3,-18,-26, 17,  4,-16,  4, -3,-13, -4, -6,-11, -4,-21,
       7,  8,  2,  5, 13, -6,  1,  5,  8,  7,  9, -6, -6,  1, -1,  2 },
    { -3, -1,  0, -2, -2,  0, -1,  3,  4,-14, -8, -9, 13,  2, 50,-23,
      -8,  8,  7, 11, 16,  3, -7,  0, -2,  6,  5, -1,  1, -2,  4,  3 },
    {  1,  3,  1,  1, -6,  3,  6,  6,  2, -2, -3, 10,  2, -8, -5, -5,
       5,  4,  4, -2, 10, -8,-40, -1, 21,  8,  3, -4, -1, 13,  4,  7 },
    {  2,  0, -4, -8,  5,  2,  7, -5,  5, -8, -4, -1, 12,  2, 12,-13,
      -9,  0,  1,-12,  9,-43,  1, -5, 12,  1,  3,  6,  1, -1,  3, -2 },
    {  6, -2, -1,  1,  0,  4,  8, 14,  4, -7,-23, -5, 23,-17, -6,-15,
      -8,  7, 10, -1,  7,-16,  4, -6,  2,  3, -3, -3, -1,  8, -1,  4 },
    { 10,  4, -4,  1,  7, -3,  2, 11,  4, -6, -3,  8,  5,  4,  1,-45,
      -6, -4,  4,  2,  1,-14,-10,  1,  1,  6,  2, -8, -1, -3,  3,  3 },
    {  1, -1,  2, -3, -8,  9,  3,  3, -2, -5, -8,  8,  7, -7, -4, -6,
       5, -9, 11, -2, 46, -5, -1,  9, -2,  0,  3, -5, -3, -5,  7,  0 },
    { -4,  1, -2, -1,-11, 11,  8, -3, -2,-10,  0,  4,  9,  9,-17,-17,
     -34, -4, -5, -7, -3,-12, -3, 11, 18,  3, -2, -5,-18, -5, -3,  6 },
    {  7, -5, -3,  1, -4, -3, -5, -1,  2,  5, -2,  3,-10, 12,-18, -5,
     -10, 12, -9,  4, -6,  2,  0, 16,-17, 15, 14,-12,-10, -2, -9, -1 },
    {  4, -5, -3, -5, -3, -1,  7, 18, -7, 12,  3,  5, -8, -4,-20,  1,
     -25,  1, -8, 13,-10,  8,-19, -1, -8, 10,  6, -9, -1,  0, 12,  4 },
    { -4,  5,  0, -1,  2,  5, -8, -2, -6,  4, -8,  9,  3,  2, -7,  4,
     -25, 13,-23, 10, 14, 15,-11,  3,-18,  4, 16, -4,  1,-10,-10,  3 },
    {  5, -3, -1, -3,  4,  1, -3, -4, -5,  1,-12, 14, -7, 11,-15,  6,
      -6, 24, -4, 13, -1, 15,-13,  8,  3,  7, -5,  2,  2,  0,  3, -7 },
    { -3,  1,  0,  8,  6, -1,  6,  5, -5, -2,-12,  4,  0, -2, -3,  5,
      -6,  0, -8,  9,-10,  4,-28, 12,-20, 11,-13,  7,-18,  1,-11,  1 },
    {  1, -4,-15,  5,  0,-13, -5, 13,-11,  4, -4, -5,  5,-14,-16,  0,
     -14,  5,-20, 12, 10, -7, -5,  6,  6, 22,  6, -4, -2,  3,  8, 11 },
    { 13,-11, -2, 16, 16, -7,  0, 20, -7, -1,  0,  5, -9, 12, -2, -5,
     -22,  5,-10, 12, -6, 11,  9, 21, -8, 15,  4,  0, -8, -4, -4, 10 },
    { 18, -4,-13,  0,  1,-15, -1, -3,  2, 10, -1,  6,  1, -4,-20, -5,
      -8,  6, -8, 17, -5,  5,-10,  8,-22,  6, -5, -2,  8,-17,  8,  2 },
    {  1, -2, -9,  6,-31, -8, -8,  8,  0,  5, -9, -4,  2,  3,-12, 11,
     -18, 10, -5,  3,-11, 13, -6, 11, -3, 12, -7,  3, -9, -1,  2, 11 },
    { -9, -6, 21, -8,-15,  4,-11, 12,-11, 17, -1,  2, -6,  0,-15, 13,
     -12, 19,  0,  2, -6, -3, -9, 10,  3, 17, -2,  5,-10, -3,  0,  1 },
    {  4, -6,  5,-10,  1, -5,  1,  0,  0,  0,  2,  7, -2,  2, -2,  0,
      -4,  3, -4,  1,-12,  6,-49, 16,-10, 13,  0, -2,  8,  6,  1,  8 },
    {  5, -8, -7,  9, 13, -5,  7,  0, 10, 11, -4, -3, -1, 13,-14,  6,
     -15, -6,-14, 16, 15,  1,-18, -4,-20, 20, -7, -1, -9, -2,-10, 10 },
    {-12,  4,  0, 10,  0,  3,  8,  4,-27, -1, -2, 19, -4,  2,-13,  3,
       1,  9,-12,  1,-22, 19, -5,  4, -9, 12,  2, -9, -8, 11, -3,  7 },
    {  4, -5, 11, -6, 17,-17,  5, -4, -2, -6,  1, -5,  2,  4,-14,  6,
     -20, 19,-20, 12,-21,  5,-14, 13, -2, 11,  4, -3,  0,-10, -4, -2 },
    { -2, -1, -3,  8, -9, -7,-22, -3,-24, 13, -2, 10,-15,  5, -9,  4,
      -7,  0, -5, 15, -8, 11,-13,  6, -4, 19, -8, 12, -4,  6,  9,  7 },
    {  2, -3,  2, -1,  0,  3,  1,  2,  1, -4, -2, -3,  1,  5,-12,  6,
     -16, 14,-23, 10,-14, 17,-15, 16, -2,  9,-25,  9,-10, 16,  4,  9 },
    { -3,  7, -8, -3,  2,  2, -4, -8, -9, 10,  3,-11, 25,-10,-28, 27,
      -9,  7,-13,  9, -2,  4,-12, -8,-14,  6,  7,-10,  3,  3, -3,  5 },
    { -8, -3,  1,-10,  8, -3, -9, -4, 13,  7,  2,  4,-10,  4,  3,  7,
     -18,  2,-22, 15,  4, 20, -7,  5, -6, 13, -1,  4, -7, -6,  6, 13 },
    { -2,  3,  0,  2, -4, -2,  0,  0,  1,  2, -2, -5,  0,  1, -4,  0,
      -2, -3,  1,  2, -1,  2, -8, -1,-24, 68, -3,  8,  3,  3, -1, -1 },
    {-15, -2, -9, -7, -1,  8,-14,  8,  3,  6,  0, -1, -8,  8,-23,  2,
     -14, 17,-15,  8, -4,  7,-18,  0, -8, -3, -1, -4,-10,  4, -1,  4 },
    {  8,  0,  2, -7,  0,  5,  1,  3,-11,  4, -8, 14,  3, 20,  1, 26,
     -11, 13,-13, 20, -2,  0, -8,  2, -6,  6, -1,  9,  3, -6, -3, 10 },
    {  5,  0, -1, -7, 10,  1, -3,  5,  4,  7, -5, -1, -3, -1, 12, -3,
     -15,  7, -9, 22,-19,  8, -9,  4,-23, 13,-14,  6, -6,-14, -4,  7 },
    { 14, -5, -8,-10, 25,  3,-23, -7,-28,  0, -1, -9,  4,  1,-13, 20,
      -8, 10,-16,  8, 12,-13,-21,  5,-13, 11, -2,  1, 12, -7,  2,-10 },
    { -5, -4,  9,  5, -6, 35, -7,  8, 15,  2, -1, -9, -6,  2,-18,  7,
     -15,  6, -3,  2,  8, 12,-30,  7, -4, 20,  2,  6, 13, -6, -4,  0 },
    {  1,  8, -9,  9, -5, 12, -9, 16, -9, 16,-17, 14,-13, 15,-18, 14,
     -15, 17,-12, 14,-13,  7,-16, 13, -9,  5,-11, 10, -9,  6,-12, 13 },
    {-10, -4,  5,  3,  1,  6,  8,-14, -5, 15,  7,  4,  8,  7,-22,  8,
      -7, -8,-15, 26,  1, 13, -3, 17, -5,  9, -2,  4, -6,  3, -8,  9 },
    {  8, -3,  2,  3,  3,  1, -2, -1,-11,  8, -4,  0, -6, -5, -1, 13,
     -37,  9,  1, -6,-10, -2,-10, 11,  8, 13, -3, -2, -6,  8, -4, 13 },
    {  3,  2, -3, -4, -4,  7, -8,  9, -8,  9,-20, 12,-19, 15,-18, 17,
     -15,  7, -1, 20,-11,  6, -6,  3,  1,  9,  2,-14, -2, -2,  2,  1 },
    { -7,  1, -1, -3, -6,  4,  4, -3,  3, -1,  5, -4,  3,  2, -1,  9,
     -59,  5, -4, 30,  3,  3, -2, -3, -1,  2,  2,  1, -1, -1, -2,  1 },
    {  0, -3,  2,  0, -1, -8,  0,  2, -3,  4, -4,  1, 10,  6, -6,  8,
      -7,  4, 10, 11,-41, 27,-20,  3, -3,  8,  1, 11, -5, -8,  0,  4 },
    {  5,  1,  4, -2,  1,  2, -1,  6, -7,  2, 11,  4,  0,  0, -8,  7,
     -10,  0,  0,  8,  2, 10, -1,  1, -2, 44, -2,-21,-12, -3, -1,  2 },
    { -4,  4, -2, -2,  6, -8,  2,  1,-10, 14,  8,  6,  5,  1, -2,  4,
     -13,  4,  2,  5, 10, -2,-21, 32, -3, 18,  9, -6, -9, -9, 10,  2 },
    {  9,-16, -6, -2,  1,  4, 22,  2, -2,  1, -3, -2, -9,  3, 16, 19,
     -24, -6, -6, -5, -8, -7,  8, -7, -1,-12,  5, -3,  0,  4,  2, -3 },
    { 10,  3,-16, -4, -1, 13,  4,  4,  1, -3,  1, -6,-14, 18,  3,  8,
      -8,-28,-16,  4,  4,  2, 12,  7,  9, -4, -4,  5, -1, -1,  2,  2 },
    { -5,-13,-22, -3, -8, 21, -2, -9, 21, -4, -9,  5, -8, 15,  5,  1,
      -5, -9, -7, -2, -5, -5, -1, -5, -5, -5,  3, 10, -4,  0, -7, -2 },
    {  5,-10,-18,  2, 20,  4, 13,-10,  8,-15,-11, -3, -1, 16, 10,  9,
      -8,  6,  7, -5,  6, 11,  5, 17, -4,  7,-11,  5, -3, -6,  2,  1 },
    {  3, -5,-19,  1,  1, -3, -2,-25,-11,-17,  0,-13, -4, 10, 10,  2,
      -5,  4,  0,  3, -3, -5,-10, -2, 13,-22,  0,  3,-11, -5,  7, -1 },
    { 12,-14,-29,  6, -1, 10,  7,-17,-12, 14,  3,  9, -9,  9,  7,  6,
      -3,-13,  0,  5,  3, -1, -6, -1,  0,  2,  4,-12, -5, -1,  2, 11 },
    { 12,-15, -7, -2,-12, 17, 20,-16, -2,-12, -6, 15, -6, 12, 11,  9,
       7, -6,  7, -4,-19,  6,  2,  2,  3,-11,-10, -4, -5, -3,  3,  2 },
    { 11,-22, -6,  0,  8, 18,  3,-11, -4, -7,-15,-17,-12,  6, 16,  4,
      -9,  4, -5,  3,  6,-16, 10, -7, -7, -3,  5,  0,  1,-15, -4,  5 },
    { 12,-22,-16,  5, -6,  8, 12, -4, -9,-17,-11,  3,  5,  8,-17,  0,
      11, -4,-13, -6,  2, -1, -1,  3,  3,-11,-12, -1,  1,  1, 12, -2 },
    {  8,-10,-33, -5, -3, -6,  1, -7, -8, -4, -6, -1,  5, -4, -6,-12,
     -16, -8, 11,  8,-14,  7, 12, 11,  4,-14, -3,  6, -7, -5, -3,  3 },
    {  0, -8, -7,  2, -4, 24,  2, -9,-11, -3, -7, 11,-12, 17,  1, -1,
       3, -5, -7, 12,  4, 11,  0,  3,  2,-18, -3,  4,  7, -6,  3, 15 },
    { 10,-15,-16, -2, -4, -9,  7,-15, -6,  2,-16, 13, -8,  7, 19,-21,
      -4,-12, -9, -3, -3,  6, 11, -3, -1,-19,  3, -7, -9, -4,  3, -6 },
    { -5,-10,-21,  0, -3, -7, 18,-21, 15, -5,-12, -4,-13,  2,  6, -9,
      -9,-11, -4, 13, -3,  6,  4, -1,  7, -9, -4,  9,  5,  2,  6,  3 },
    { 15, -1,-27, -2, 10,  3,  7, -8,  9, -2,  7,  1, -2, -5, 18,  9,
     -11,-17, -2,  7, -9, 11, 10,  0, -8,  6,-16, -3,  2, -7,  3, 11 },
    {  4, -9,-39, 19,  6,-13, 13, -5, -5,-15, -2,  9,  0,  4, 14,  6,
     -10, -4, -5,  2, -4, -2,  5,-11,  3,  3, -2, -2, -7,  9,  7,-10 },
    {  5,-11, -8, 10, -2, 12, 16,  0, 12, -2, -6,  8, 14,  8,  7,  1,
      18,-30,  4, 10, -4, -6,  2,-11,  9,-10, -8,  5,  0,  0, -7,  6 },
    { -1,-16,-10, 11,  0, 13, 12, -4, -4, -5,-21, 12,  4, 13, 14, -7,
       6,-16,-13,  8,  2,  9, 15,-12,  1, -9,-22, 10, -9,  9,  9, -7 },
    {  4,-12,-27,  1, -2, 11, 15,  3, 14,-14, -9,  0, -9, 16, 22, 10,
      16,-10,  5, -5, -9,  1,  1,  6,  6, -4,  2,-17, -5, -6,-15, -1 },
    {  7,-12,-17,  1, -9,  5, 20, -7,  3, 23, -8, -8, -8, -1, 13, 17,
      -7,-13,  4, -4,  7, 14,  8, 11, -3, -3,  4,  0,  4,  6, -1, -9 },
    {  7,-15,-15, -4, 10, 12,  3,-13,  6, 14,  9, -8,-15, 14, 23, -5,
     -10, -5,  1, 15,-10, -7,  1,  9,  4,-13,-10, 10,  7, -3,  2,  3 },
    {  4,-10,-14,  0,  3,  4,  0, -9, -3, -4,-11,  2,-17,  8,  2, 15,
       6,-12,-12, 15, -5, 17, 18,  3, -3, -3, -4, -6, -8, 13,  4, 10 },
    { -2,-18,-26, 10, -4, 10, 13,  4, -4,-16, -7,-17, -3,  5, -4,  2,
     -15,-10, -1, -8, -7, -3,  2,  2,  8,-10, -7,  2,  2, -4,  4, -1 },
    {  4,-19, -5, -1, -1, -6,  2, -8, 10,-16,-28, -6,  8, -1, 11, 28,
       2,-10, -4,  6, -6,  6, 11, 15, -4, -2,  7,  3,  7, -7,  4,  1 },
    { -3, -6,-10, -5, 13, 18, 10,-15, -5, -3,-13,  5,  1,  2, 18, -5,
     -10,-10, -7,  4,  2,  1,  5,  4,  2,  5,  4,  8, -9,-17,  7,  7 },
    { 20,-12, -2, -4,  5, 14,  7,-11, -1,-16, -6, -4,-11, 17, 14,  0,
      -8,-10, -8, 10,  3,  5, 10,-16,  3, -8,-14, 10,  3,  9,  0,  3 },
    { 12,-10,-36,  0,  7, 15,  2,-16,  2, -1,  0, -1,  5,  4,  5, -3,
       1,-10,  5, -1,-15, -3,-12, 12,  2,  5, -1,  5,  6, -3, -2,  2 },
    { 17,-15,-31, 23, -4, 15, -2, -3,  6, -7, -5,  1,-12,  4,  6,  8,
     -10,  8,  3,  5, -4,  1,  5,  3, -1, -4, -3,  1, 10, -4, -2, -2 },
    {  6,-18, -5, 12, 10, 12, 14,-11, 15,  2, -9, -6, -5, -2, -9,  4,
      -5,-28, -4, 14,  0,-16,  9, 14, -1,  3, -4, -4,  2,  1,  0,  4 },
    { -5,-14,-31,  8, 16,  7, 13,-13,  5,  6,-16, 10, -5,  2, -2,  2,
      14, -5,  8, -5,  7,-16,  6,-13, -5,  0, -5,  8, -3, -1,  4,  3 },
    {  1, -2, -1,  0,  6,  5,  2, -4, -3, -1,  0,  1,  4,  2, 43, 28,
     -12,-35, -2, -2, -7, -1,  0,  2, -1, -2, -2,  1, -4,  0, -2,  3 },
    {  2, -9,-22, 12,  3,  3, -7, -4,-19,-22,-14, -4, -1, 21,  9, -3,
     -15,-16,-13,  1,-11,  4, -9,  1, -7, -1, -1,  0, -2,  9,-13, -3 },
    { -1, -3,-23,  0,  2, 12,  3, -9, -4,  7,  3,  9,-10,  1, 27, 28,
       0,  9,-15, -2, -2,  1,  6,  8, -8,  7, -3, 20,  0,  0, -1, -6 },
    { -1, 11,  8, -2,  1,  5, -6, -1,  4,  2, -4,  0, -1, -5,  4, -6,
     -10,-12, 19,  1, -7,  9, -8, -9,-16,-11, -2, 12, 14,  4,  4, 34 },
    { 17,  7, -6,  1,  4,-10, -5,  4,-11,  3,-18,  4, 14,-13, -3,  1,
       0,  0,-11,  0,  7,-17, -4,  4,-11, -6, -8, 18,  0,  0,  0, 26 },
    { -6, -7, -1, -1, 11, -8,  1,  3,  2, 11, -6, -6, 10, -3,  1, -3,
       7,  4,-12, -8,  0, -9,  8,-22, -5,  0, -6, 22, -2, 11,-13, 24 },
    { -3,  4,  0,  3,  9, 10, -1,  3, -9,-12,  1, -5, 18,  0, -3,  8,
      25, 15, -8,  2,  2, -2,  4,  8,  9, -1, -5, 10, -3,  1, -1, 23 },
    { -5,  2, -9, -1, -3,  0,  3, -1,-10, -4,  0,-13, 16,  9, -1,-14,
       2,  6, -2, -6, -5, -2, -7,  7,  5,  3, 11, -2,-14,  0, -9, 30 },
    {  4,  6,  6,  5, -3, -1,  4,  5, 10,  0,  5, -4,  7,-11, 14, 14,
       7, 34, -9,  0,-10, 22, -7, -1,  7, -9,  2, -8,  0, -7, -5, 29 },
    { -4,  3, -1, -4, -3,  5,  1, -4,  0,  2,  4,  2,  1, -1,-10,  1,
       6, -6, -4,  1,  4, -3, -3, -5,  0,  3,  7,-12,  0, -2,-10, 55 },
    {  5,  9, -1,  0,  4,  9,-21, -9,  4,  2,  6, -7, 11, -7,  1, -5,
       0, -4,  2, -3,-13, -8,  0, -9, -4,  2, 16, -2,-15, -7,-11, 31 },
    {  8,  2, -1,  0,  3, -5, -5,  5,  1, -1, -9,  1,  0, -6, -2, -1,
       5,  2,  0,  0, 12, 20,-19,  1,  8,-12,-11,  0,  6, -5,  2, 31 },
    { -1, -1, -2,  1, -1,  3, -9, -5,  8, -2,  5, -1,  0, -2,  4, -2,
      -3,-12,  0, -2,  3,  0,  9,  4, -1, 21, -8,  3, -4,  9, -6, 30 },
    { -4,  0, -7, 17, 10,-12, -2,-10,-12, -3, 10,  0, 11, -4,-13, -3,
       5,  6, 10,  7, -8,  0, -7,-13,  1,  0, -2,  7,-12,  4, -3, 24 },
    {-13,  9,  4, -2,  2, -4,-14, -1, -3, -5,-10,  4, 13, -2,  5, 13,
       8,  3, -2,  1,  5, -6,  7,-18,-10,  1, -1,  5,  4,  1,  0, 25 },
    { -5, -1, 18, 12,  8,  8,-16, -1,  1,  1,  1, -4, -5,  3,  3,  4,
       4,-11,-12,-16, -6,  2, 12,-13,  0,  9,  7,  9, -9,  0,-10, 24 },
    { -4,  1, -3,  0,  2, -4,  4,  1,  5,  0, -3,  2, -3, -2,  2, -1,
       1,  4, -1, -2, -2,  1, -1, -1, -4, -1, -4, -2, -6,  6, 12, 69 },
    {  8,  5, 11,  0,-15, -4, 13,  6,  0, -4,  9,  1, -5, -3, 15,  0,
       1,  6, -5,  0,  1,  6,  5,  8,  0,  7,  1, -1, -4,-11, -9, 41 },
    { -4, -9, 32, -6,  0,  7, -4,  6, -6,  1, -6, -2,  4, -8, -5, -3,
     -16, -1, -2, -6,  1, 15,  0, 21,  3, -3, -4,  3,-12, 16,  2, 27 },
    { -6, -5,  1, -9, -5,  3,  7, -3,  5,  5, 14, 13, 20, -7, -1, 12,
      -1, 10,-11,-11, -7, -4,-14,  7,-14, 13, 22, 18, -1,  0, 14, 28 },
    { -8,  3, -2,  0,  5,  6, -1, -4,  1,  3, -7,  3,  1,-15,  4, -9,
      22,-10, -9, -4,  1,  8, -4,  9,-15,  2, -6, -4,-16, 12,-10, 23 },
    {  0,  0,  2,  0, -1,  3, -3, -1,  3, -5,  7,  1,  5, -5, -8,  1,
      13,-15, -5, -7, 12, -6, -2,  3, 10, -5, -8, 17, -5,-11,-14, 23 },
    { -7, -4,  6, -4,  5, -6, -5,  2, -4, 11,  9, -4,  2, -2, -4,  6,
      15,  3, -3, 18,-15, -2, -6,  3,  3,-20, 17, 11, -4,  2,  3, 29 },
    {  6,  1, -6,  2,  3,  0,  0, -3,  3,  3, -1,  3, -4, -6, -6, -7,
      -3, -2, -7, -2, -4,  5,  3, -5,-20,-13, -4, 10,-14,-29, 14, 37 },
    {  3,  4,  3, -6, -4,  5,  0,  3,  2,  3,  0, -2,  4,  0, -3, -5,
      -4,  4, -4,  4,  4,  3,  1, -4, -4, -9,-14, 20,-30,  3,-18, 33 },
    {  0,  2,  5, -2, -4, -2, -1,  2, -6, -3, -2, -2,  2, -5, -1,  4,
       3,  2, -3,  0, -1, -1,-10, -7,  2, -4,-18,  2,-37, -1, 12, 40 },
    { -7,  2, -1,  0, -2,  4, -8,  1, -4, 12,  7,  4, 15, -7,  1, -9,
      18,  0, 12,-17, -3, -1,  0,  0,  0,  2, -6,  0, -4, -3, -1, 26 },
    { -6,  4,  8, -5, -6, -2,  2, -1,  1, -1,-15,  8,  7, -1,-17, -4,
       1,  5,  6,-11, -6, 14, 17, -5,-15, 11,  8,  0, -3,-15, -6, 28 },
    { -1,  0,  0,  0,  1,  0, -1,  0,  1,  3,  2, -2,  3, -1, -1,  2,
       2, -1, -1, -7,  1,  2, -9,  0, -1, -4,-18,  7,-10, 49,-13, 32 },
    { -1, -3,  4,  1,  2, -5,  1, -7, -1,  5, -9,  4,  4, 25,  1, -1,
       2, -5,  2, -7, 17, -2, 10, -5,  0,  2,-15,  3, -9,  7, -9, 30 },
    { -5, -1,  0,  2,  1, -1,  2,  5,-33,  3, -5, 14, 11,  7,  5, -3,
       2, -8, -4, -2, -7, -6,  4, -8, -1, -8,  2, -2, -8, -1, -4, 27 },
    { -1,  0, -1, -2,  1, -1, -2, -1,  2,  0,  1,  2,  2,  4,  1,  3,
       4,  2,  1, -7, -4,  1, -3, -4,-35,-25, 17, 10, -3,-26, -7, 32 },
    { -5,  1,  6, -2,  6,  6, -9,  3, -1, -4,  5, -4, -2, -2, -9,  2,
      -5,  2,  2,  4,  3,  5, -5,-16,-31,-12,-11,  2,-19, 20, -2, 21 },
    { -5,  2,  7, -7, -7,  5, -7,  2,  0,  0, -4,  3, -1,  0, -1, -2,
       0, -3,  5,-11, -8, -3, -7, -7, 28,-11, -7,  0,-16,-11, -4, 29 },
    {  2,  1, -3, -2, -1,  3,  4,  0,  1,  0, -1, -5,  4, -5,-12,  2,
      -2, -5,-22, -2, -1, 11,  8, -7,-12,  0,-34,  6, -5, 11, -8, 19 },
    { -1, -3,  5, 11, 18, -2, -2, -5, -2,  4, -1,  8,  5, -6,  1, -1,
       2,  8,  4, -5, -8, -2,  5,-18,  7, 12,  7, 19,-18,  2, -6,-13 },
    {  9,  0,  0,  5,  4,  3, -6,  4,  1, -4,  5, -1, -4,  8,  8,  6,
      -8, -6,  0,  6, -3,  3,  5, -3, 17, 31, 16, 10,-13,  0, -9,-19 },
    { 12,-10,  2, -2, -2, -1, -3,  6,-12, -5, -2, 14,-16,  4, 12, 12,
      17,  4,  7,-16,  7, -6, 11,  7,  7,  2,-25, 23,-24,  5, -7, -9 },
    { 10,  4, 13, 10, 10,  3, -6,  3,  3,  2, -1, -6,  8,  4, 10,  0,
       1,  2, -4,  2, -3, -8,  0, -1,  9,  9,-10, -3,-29,  1, -1,-27 },
    {  2,  2,  0,  7,  9, -2,-10, -1, -1,  1, -9, -5,  8,  4,  1,  2,
     -10,  1, 13, 12, -3, 15, -9,  2, -7,  1,-10, 23,-20,-18, -9,-15 },
    { -3, -5, -1,  8,  0, -5, -1,  4,  7, -1, -7,  2, -8, -5, 11,  7,
      -6,  3, -3, -9,  7,  9,-22,  1,  6, -4, 14, 27,-25,-14,  3, -5 },
    {  1,  3,  8,  4,  7,  6, 12,-17,-15,  1, -8,-10,  7,-14, -8,  6,
      -2, -2,-11,-11, -7, 13, -2, -2,  4,  5, -5, 13,-23, -6,-17, -8 },
    { -5,  4,-14, -5, -4, -5,  6,  5, -8, -5, -2,-11, -7,-12,  3,-11,
       2, -6,  4,-10, -5, -7, 14,  5, 23, 11,  7, 12,-16, -6, -4,-16 },
    {  5,  6,  2,  5, -2, -5, -5, -6, -5,-19,-13, -1, -3,-13,  5,  0,
       6, -2, -2, -6, -7, -7, -1, -9,  4, 14, 17,-12,-27,  3,  0, -1 },
    {  7, -1,  9,-10,  8,  2, -7, -2,  5,  2, -3, -7,  3,  0,  6,  4,
      12,  5, 11, 14,-13, -1,  8,  1, 13,  9, 12, 12,-18,-14,-11,-16 },
    { -7, -5, -6, -5,  0, -1, -3,  2,  2,  1,  4,  9,  2,  3,  5, -2,
       2,  1,  8,  0,  3,  0, -2,  2,  1,  7, 29,  0,-36, -5, -9,-21 },
    { 14, -6, -9,  0, -1, -8, -8,-11,  2,  2, -9,-12, 12, -4,  5,  3,
      -5, -9, 11, -1, -3, 12,-21, -3, 12,  5,  3, 11,-18,-15,  1, -2 },
    { -1,  3, -9, -3,  7, -7,-18,  2,  4, 12,-10,  2,  8, -3,-14, 13,
      17, -5,  5, -9, 13, -3, -7,-18, 17, -2,  5,  7,-20, -3, -6,-11 },
    { -3,  3,  3, -1,  1, -6, -5,  1,  5, -3,-14, -6, -5, -8, 14, -6,
       7, -1,  5,  1, 15, -1, -7, -4,  6,-11,  9, -2,-37, 16, -7, -3 },
    { -1,  0,  6,  1, -3, -9,  0, 11, -8,  2, -2,  0,  5,  2, 12,-10,
      10, 13,  2,  7, -6,  2,-10,-10, 21, -5,  5,  5,-12,-23,  3,-14 },
    {  6,  0, -2,  1,  0,  1,  0, -4,  1,  1,  8, -2,  2, -5, -2,  1,
       8, -4, -1, -1,  4, -1,  2,  6, 32,  1, -5,-20,-40, -4,-18,-14 },
    {  2,  2, -7, -2,  4,  4, -1,  2,  0, -2, -4, -7,  3,  5,  0, -5,
       1,  2, -6,  4, -1, -2, -1,-15,  8,  3,  9, 46, -7,-18,  6,-11 },
    {  5,  5, 16, 21,  3,-11, -4, 11,-12,  2,  4,-12, -1, 11,  8,  1,
      -4, 11,-11,-21,  1,  1,-11,  3, 13,  1,  5, 12,-25,  1, -3, -2 },
    {  1,  6, -7,  4,  2,  3,  1, -5,  8,  9,-15,  3, -3,-14, 17,  4,
      -8, 14, -2, -8, -4,  5,  8, -7,  8,  9,  7,  6,-29,-17,  8,  4 },
    { -7, -7,  4,  0, 13,  1,  0,  4,  4,-16,-10, -7,  5,  9,-15,-10,
     -10,  8, -4, -1,-11, -1,-10,-15,  3,  3, 14, 10,-19,  2,-18,-12 },
    { -4,  0,  2,  0,  5, -2, -9,  0,  4, -4,  2, -1, -2,  2, -4,  9,
       2, -6, -4, -2, -1, -3, -3, -1,  2,  5, -1, 11,-24,-44, -9,-15 },
    { -1,-10,  6, 21, 11, 15, -7, 10,-14, -9, -8, -8,  4,  6, 19,  1,
      -6,  1, -5,-17, -8,-10,  9,  5, 11, 18, -1, 10,-16, -7, -9, -8 },
    {  3, -5,  0,  0, -2, -2, -6,  4, -4,  1, -1,  0,  7, -3,  4, -4,
      -7,  7, 17,-20,  6,  4,  1, -6,-12, 31, 13, 19,-14,-10, -7, -2 },
    { -2,  6,-10,  3,  9,  6,-14, 15,  2, -5,  2,-11,  9, -8,  4,  6,
      20,-15, -3, -3, -1, 32,-21,  6,  1,  9, 11, 17,-19,  6, -1, -3 },
    {  8, 10, -2,  0, -8,-16,  7,  7,  6, 10,  4,-14,  7, -6, 21, -7,
      10,  5,  5,  0, -7,  2, -6,  0, -7, 11, -9, 15,-20, -7,-11,  2 },
    {  0, -7,  5,  2,  0, -3, -6, -4, -2, -1, -4, -5,-13, -1, 27, -9,
      -6,-11, -7,  1, 11, -4, -4,-14, -2, 11,  6, 10,-19, -6,-15,  2 },
    {  0,  7, -1,  2, -7,-15, -2, -3, 13, -5, -5, 12,  3,  0,  5, -5,
     -22,  2,  7, 22, 13,  0, -1,  2,  3,  2, -7,  7,-27, -4, -4,-12 },
    { 11,  1,-16,  6,-15,  1,  3,  2,  0,  2, -3,  2,  5, -2, -5,  9,
       5, -3,  3, -2,-11,  3,  9,  6,  9,  3, -1, 12,-41,  8, -6,  9 },
    {  3, -7,  3,  2,  5,  5,  0, -1,  1,  3, -5, -2,-13,  7, -1, -2,
      -2, -6,  4, -6,  0,  2, -2,  2,  4,  1, -4,  1,-47,-21,  7, -6 },
    {  3, 16, -7, 13, -4, -2, 10, -3, -1, 18,-13,  7,-13, -4,  8,  4,
       8,  9, -5, 13,  8, -5,  3, -6,  7, 18, -8, 10,-25, -3,-12,-12 },
    {  1, -1, -1,  0,  2,  5, -5, -3,  0, -5, -1,  0, -4, -8, -2,  3,
       2, -2,-17, -6, -4,  1, 33, -6,-20, -6,  8, 31,-26, -8, -1, -4 },
    {  3, -3, -3,  5, -3, -2,  1,  7,  0,  3,  6,  3,  6, -2,  9, 15,
     -10, -3,-15, -5, -3, -4, -6,-30, 17, -8, -2,  2,-20,  0, -8, -2 },
    { -2, -1, -1, -1,  3, -5, -2, -3,  4, -2,  0,  5,  8, -3,  1, -4,
       1,  1, -3,  4,  4,-14,  3, 11, -5,  3, -3,  7, -3, 13, 23,-16 },
    {  2, -6,  1, -3,  5,  0, -6,-11, -7, -4, -1,  2, -7, -1, -1,  7,
       1, -2,  6, 12, -6,  8,-13, 17, 25,-23,-19, -7,-12,  9, 16,-17 },
    {  9,  4,  4,  4, -3, -1,  6, -2, -3,  0, 13, -4, -7, 14,  1, -7,
       0, -5,  3,-19, -3,  5,  3,  9, -1,  9,-13, 13,-17,  4, 21,-26 },
    {  0, -5,  0,  0, -4, -5,  2, -6, -4,  5, -7, 10,  0,  2,  0, -2,
      -2,  0,  4, -6,  7, -2,  6,  5, -5,  2,-12,  1,-29, 29, 27, 12 },
    {  9,-10,-22,  6, -1, -1,  9,-14,-12, -2,  1, -1, 10,-11,-16,  0,
       3, 11, 13,-14, -9, -2, -1,  6,  4,-14,  0,-10, -2, 16, 17,-11 },
    {  2,  0, -1, -2,  4,  3, -6, -2,  1, -1,  1,  3, -4,  1,  3, -4,
      -1, -1,  4, -1,  1,  0,  1,  6, -5, -7,  2,  1,-47, -3, 50,-17 },
    {  8, -4,-11, -7, 11, 11, 14, -7, 12, -7,  6,  2, 13, -6, -3, -2,
     -14,  6,  6,  6,  0,  2, -1,  5,-20,  2, -1,  4, -5,  6, 21,-11 },
    { -2, -9,  3,  0, -6,  7,  8, -8,  1, -3,  4,  1,  5, -2, -3, -7,
       4,  7,-12, -9, -2, 10, -6, 13,  6,  5, 20,  2,-15,  9, 28, -7 },
    {  0, -5, -6, -6, -6,  1, -6,  6, -2,  4,  8, -3, 12, -1, -4, -2,
       6, 16,-14,  9,-14, -2, -8,-27, -3, 18, -1, -7, -3,  8, 23,-23 },
    {  1,  4, -9, -1, -5, 10, -2,  1,-11,  1, -9,  4,  7, 14, -9, -2,
      -3,  2, -5, -1, -6,-10, -7, 11, 20,  2,  3,-19,  3, 15, 30, -9 },
    {  7,  2,-14, -4,  0, -2,  5,  2,  5, -2,  8, -3, -7,  6,  6,-11,
     -14,  1, 10, -1, -7, -8,  1, 10,  3, -6,-15,-12,-17,  4, 30, -6 },
    {  4,  2,  1, -2,  3,  0,  1,  0,  2,  0,  1,  6, -7,  0,  3,  4,
       4, -4, -2, -5, -2,  2, -1, -2,  0, -2,-11, -7, -3, 42, 24,-14 },
    {  4,  1,  3,  2,  0, -2, -3, -2,  2, -1,  4, 11, -2,  2,  3, -4,
      -5,  9,  2, -4, -9,  5,  8, -1, -7,  1, 24,-13,-28, 20, 15,-22 },
    { -3,  7,  6,  3, -2, -5,-10, -2, -2, -1, -6, -6, -2,-14,-16, -6,
      -5,  0, 18,  0,  9,  1,  7,-13, -5, -6, -9, 11,-15,  9, 22,-11 },
    {  9, -2,  6,  5,  2,  9,-10,  1,  1,  5, -4, 12,  2,  2,-10, -7,
      -4, -6,  7,  9,  6, 15,  6,  6,-10, 10,  5,-13, -5,  6, 24,-12 },
    {  1,  3, -3, -3,  8,  1, -6,  2, -5, -3,  7,  2, 14,  6,  9, -6,
      -5, -4, 27,  7, -3,  8, -6,  3, -8,  8, 22, -5, -6, -2, 22,-17 },
    { -2, -2,  3, 10,  9,  9, 12,-15, -1,-11,-13,  3, -2,  1, -3,-11,
       7,  9, 16, -3,-10, -5, -5,  1,  8, -3,  9,  9, -5,  3, 31,-12 },
    {  7, -5, 10, -4, -8,  2, 16, -2, 10, 10, -3, -2,  3, -8, -3,  3,
     -13, -6, 15, 20, -9, -3,-12,  1, -2,-16,  8,  8, -1, 16, 22, -5 },
    {  5, -3,-15, -2, 12, -8,  8, -5,  2, -8, 20,-18, 14, -4,  3,  3,
       7,-13,-16,  1,-10,  7, 16,  7,  4,-14, -4, -5, -9,  8, 23, -6 },
    {  5, -4, -5, -4,  1,  8,  4, -7, -5,  8, 10,  6, -6,-10, -2,  6,
       9,-17,-14, 11, 12, -3,-13, -7,  2, 18,  3,-25,-16, 18, 22, -5 },
    {  5,  6, -7,-20, -4,  2,  8,  4,-24, -4,  1,  4, -5, -2,  1,-10,
      -2,  9,  3, -4, -3, -4, -4, -4, 10, 10,  3,  0, -6, 25, 21,-11 },
    {  0,  7, -1, 14, -6, -4,-10,  5,  4,  4,  4, -5,  3,  4, -1, -7,
       8,-19,  0,  6,  2,  3,-18, -3, -6,  2,  8, 14,-26, 22, 27,-13 },
    { -2, -6,  7, -5, 12, -7,  8, -1,  3, -2,  4,  1,  8, -2,  0, 14,
       6, -5,  6, -4, -7,  7,-21,  8,  1,  8, -9, -4, -3, 11, 25,-13 },
    {  4,  4, -1, -6,  4,  9, -8,  1, -3,-10, -2,  0, 15, -9,-16, 11,
       1,  1,  6,  3, -9, -5, 16, 26,  1,-14,  1, -3,-14,  7, 15, -9 },
    {-12, -2, -9,-13,  2,  6, 14,  0,  1,  0, -1,-13,  0, 10, -1,  6,
       9, -7,  8,  8, 19,  6, -1,  9, 10, -4,  1, -7,-22, -2, 29, -7 },
    {  2,  4, 13,-12, -8, -4, -5, 13, 12, -5, -3, -3, -4,  1, -1, 10,
      15, -6, -1,-11,-30,  4, 15, -1,  9, -7,  0, -2, -7, 10, 25,-16 },
    {  7,-15, -7, -7, -1, -5, -5,-11,-20, 10,  3,-10, -3,  5, 20, -4,
       0, -2, -2, 17,  2,  0, -3,  3,  6,  5, -1,-12, -3, 15, 22,-16 },
    {  4, -1,  3,  4, -5,  0, -1, -5,-24,-29,  4, -9,  1, -3,  0,  0,
       0, -4,  7, -4, -4, -4,  3,  1, -6,  5, -3, -5,-10,  3, 25,-10 },
    { -2, -1, -1,  4,  4, -1,  2,  0, -4, -4,  2, -1, -3, -1, -2, -2,
       1, -3, -5, -1,  2, -3, -4, -4, -3,  5, -9,  1,-11,  7, 46,-46 },
    {  0, -9,  3,  4,  4,  3, -5, -6,  5, -4,  4, -2,  1,  7, -4,-10,
      13,  1,  3, -6,  4, -4,  7,  2,-19,-25, -3,-16,-12, 16, 20, -1 },
    { 18,  6,  4,-12,  0,-14,  9, -6, -1, -4, -5,  2,  1, 12,  4,  2,
       7,  0,  2,  5,-11, -5, -2,  2, -4, 10,  0, -9, -7,  9, 25, -8 },
    {  5,  0, -6,  5,  6,  3,  3,-10, -5,  1, -1,  4,  3,-11, -8,  5,
       4, -5,  5, -5, -7, -5, 11,  5, 20, -8,-16, 21, -4, 27, 23, -5 }
};

/* FIR filter coefficients, they can be cut on half and maybe use float instead of double */

DECLARE_ALIGNED(16, const float, ff_dca_fir_32bands_perfect)[512] = {
    +1.135985195E-010,
    +7.018770981E-011,
    -1.608403011E-008,
    -5.083275667E-008,
    -1.543309907E-007,
    -3.961981463E-007,
    -7.342250683E-007,
    -3.970030775E-007,
    -4.741137047E-007,
    -6.022448247E-007,
    -6.628192182E-007,
    -6.982898526E-007,
    -7.020648809E-007,
    -6.767839409E-007,
    -6.262345096E-007,
    -5.564140224E-007,
    +7.003467317E-007,
    +8.419976893E-007,
    +9.742954035E-007,
    +1.085227950E-006,
    +1.162929266E-006,
    +1.194632091E-006,
    +1.179182050E-006,
    +1.033426656E-006,
    +9.451737242E-007,
    +1.975324267E-006,
    +1.190443072E-006,
    +5.234479659E-007,
    +2.014677420E-007,
    +7.834767501E-008,
    -6.702406963E-010,
    -1.613285505E-009,
    -2.682709610E-009,
    -3.399493131E-009,
    +1.314406006E-008,
    +7.506701927E-009,
    +2.788728892E-008,
    +1.444918922E-007,
    +3.132386439E-007,
    +1.399798180E-006,
    +2.032118118E-006,
    +2.715013807E-006,
    +3.453840463E-006,
    +4.195037945E-006,
    +4.896494374E-006,
    +5.516381407E-006,
    +6.015239251E-006,
    +6.361419310E-006,
    +8.006985809E-006,
    +8.087732567E-006,
    +7.941360309E-006,
    +7.568834008E-006,
    +6.986399967E-006,
    +6.225028756E-006,
    +5.315936960E-006,
    +4.429412002E-006,
    +3.332600045E-006,
    +8.427224429E-007,
    +4.341498823E-007,
    +9.458596395E-008,
    +2.975164826E-008,
    +6.402664354E-008,
    -3.246264413E-008,
    -3.809887872E-008,
    +8.434094667E-008,
    +6.437721822E-008,
    +1.189317118E-006,
    +2.497214155E-006,
    +3.617151151E-006,
    +3.157242645E-006,
    +2.319611212E-006,
    +7.869333785E-006,
    +9.826449968E-006,
    +1.177108606E-005,
    +1.379448349E-005,
    +1.571428584E-005,
    +1.743183020E-005,
    +1.884208177E-005,
    +1.987093310E-005,
    +2.042970118E-005,
    -3.144468428E-005,
    -3.334947178E-005,
    -3.460439257E-005,
    -3.515914432E-005,
    -3.495384954E-005,
    -3.397853652E-005,
    -3.225446198E-005,
    -2.978993689E-005,
    -2.677291741E-005,
    -1.806914770E-005,
    -1.776598037E-005,
    -1.661818715E-005,
    -1.207003334E-005,
    -6.993315310E-006,
    -5.633860383E-007,
    -9.984935332E-007,
    -1.470520488E-006,
    -1.853591357E-006,
    +7.198007665E-007,
    +3.086857760E-006,
    +6.084746474E-006,
    +9.561075785E-006,
    +1.309637537E-005,
    +2.263354872E-005,
    +2.847247197E-005,
    +3.415624451E-005,
    +3.946387005E-005,
    +4.425736552E-005,
    +4.839275425E-005,
    +5.176846025E-005,
    +5.429694284E-005,
    +5.595519906E-005,
    +4.916387297E-006,
    +9.299508747E-006,
    +1.356193479E-005,
    +1.751866148E-005,
    +2.093936746E-005,
    +2.362549276E-005,
    +2.537086584E-005,
    +2.618136386E-005,
    +2.554462844E-005,
    +3.018750249E-005,
    +2.570833203E-005,
    +1.985177369E-005,
    +1.191342653E-005,
    +2.525620175E-006,
    -1.521241393E-005,
    -1.617751332E-005,
    +1.992636317E-005,
    +1.774702469E-005,
    +4.624524081E-005,
    +5.610509834E-005,
    +6.568001118E-005,
    +7.513730816E-005,
    +8.413690375E-005,
    +8.757545584E-005,
    +9.517164290E-005,
    +1.020687996E-004,
    +1.084438481E-004,
    +1.140582463E-004,
    +1.187910311E-004,
    +1.224978914E-004,
    +1.250260248E-004,
    +1.262027217E-004,
    +1.226499153E-004,
    +1.213575742E-004,
    +1.180980107E-004,
    +1.126275165E-004,
    +1.047207043E-004,
    +9.417100227E-005,
    +8.078388782E-005,
    +6.447290798E-005,
    +4.491530854E-005,
    +2.470704203E-005,
    -1.714242217E-006,
    -3.193307566E-005,
    -6.541742187E-005,
    -1.024175072E-004,
    -1.312203676E-004,
    -1.774113771E-004,
    -2.233728592E-004,
    -2.682086197E-004,
    -3.347633174E-004,
    -3.906481725E-004,
    -4.490280990E-004,
    -5.099929986E-004,
    -5.729619297E-004,
    -6.358824321E-004,
    -7.021900383E-004,
    -7.698345580E-004,
    -8.385353722E-004,
    -9.078957955E-004,
    -9.775133803E-004,
    -1.046945457E-003,
    -1.115717343E-003,
    -1.183370827E-003,
    -1.252829796E-003,
    -1.316190348E-003,
    -1.376571832E-003,
    -1.433344092E-003,
    -1.485876855E-003,
    -1.533520175E-003,
    -1.575609902E-003,
    -1.611457788E-003,
    -1.640390139E-003,
    -1.661288203E-003,
    -1.674512983E-003,
    -1.678415807E-003,
    -1.672798418E-003,
    -1.656501088E-003,
    -1.633993932E-003,
    -1.593449386E-003,
    +1.542080659E-003,
    +1.479332102E-003,
    +1.395521569E-003,
    +1.303116791E-003,
    +1.196175464E-003,
    +1.073757303E-003,
    +9.358961834E-004,
    +7.817269652E-004,
    +6.114174030E-004,
    +4.244441516E-004,
    +2.206075296E-004,
    -2.719412748E-007,
    -2.382978710E-004,
    -4.935106263E-004,
    -7.658848190E-004,
    -1.055365428E-003,
    -1.361547387E-003,
    -1.684492454E-003,
    -2.023874084E-003,
    -2.379294252E-003,
    -2.750317100E-003,
    -3.136433195E-003,
    -3.537061159E-003,
    -3.951539751E-003,
    -4.379155114E-003,
    -4.819062538E-003,
    -5.270531867E-003,
    -5.732392892E-003,
    -6.203945260E-003,
    -6.683901884E-003,
    -7.170005701E-003,
    -7.664063945E-003,
    -8.162760176E-003,
    -8.665001951E-003,
    -9.170533158E-003,
    -9.676489048E-003,
    -1.018219907E-002,
    -1.068630442E-002,
    -1.118756086E-002,
    -1.168460958E-002,
    -1.217562053E-002,
    -1.265939046E-002,
    -1.313448418E-002,
    -1.359948888E-002,
    -1.405300573E-002,
    -1.449365262E-002,
    -1.492007636E-002,
    -1.533095632E-002,
    -1.572482102E-002,
    -1.610082202E-002,
    -1.645756140E-002,
    -1.679391414E-002,
    -1.710879989E-002,
    -1.740120351E-002,
    -1.767017506E-002,
    -1.791484281E-002,
    -1.813439466E-002,
    -1.832821220E-002,
    -1.849545911E-002,
    -1.863567345E-002,
    -1.874836907E-002,
    -1.883326657E-002,
    -1.889026538E-002,
    -1.891860925E-002,
    +1.891860925E-002,
    +1.889026538E-002,
    +1.883326657E-002,
    +1.874836907E-002,
    +1.863567345E-002,
    +1.849545911E-002,
    +1.832821220E-002,
    +1.813439466E-002,
    +1.791484281E-002,
    +1.767017506E-002,
    +1.740120351E-002,
    +1.710879989E-002,
    +1.679391414E-002,
    +1.645756140E-002,
    +1.610082202E-002,
    +1.572482102E-002,
    +1.533095632E-002,
    +1.492007636E-002,
    +1.449365262E-002,
    +1.405300573E-002,
    +1.359948888E-002,
    +1.313448418E-002,
    +1.265939046E-002,
    +1.217562053E-002,
    +1.168460958E-002,
    +1.118756086E-002,
    +1.068630442E-002,
    +1.018219907E-002,
    +9.676489048E-003,
    +9.170533158E-003,
    +8.665001951E-003,
    +8.162760176E-003,
    +7.664063945E-003,
    +7.170005701E-003,
    +6.683901884E-003,
    +6.203945260E-003,
    +5.732392892E-003,
    +5.270531867E-003,
    +4.819062538E-003,
    +4.379155114E-003,
    +3.951539751E-003,
    +3.537061159E-003,
    +3.136433195E-003,
    +2.750317100E-003,
    +2.379294252E-003,
    +2.023874084E-003,
    +1.684492454E-003,
    +1.361547387E-003,
    +1.055365428E-003,
    +7.658848190E-004,
    +4.935106263E-004,
    +2.382978710E-004,
    +2.719412748E-007,
    -2.206075296E-004,
    -4.244441516E-004,
    -6.114174030E-004,
    -7.817269652E-004,
    -9.358961834E-004,
    -1.073757303E-003,
    -1.196175464E-003,
    -1.303116791E-003,
    -1.395521569E-003,
    -1.479332102E-003,
    -1.542080659E-003,
    +1.593449386E-003,
    +1.633993932E-003,
    +1.656501088E-003,
    +1.672798418E-003,
    +1.678415807E-003,
    +1.674512983E-003,
    +1.661288203E-003,
    +1.640390139E-003,
    +1.611457788E-003,
    +1.575609902E-003,
    +1.533520175E-003,
    +1.485876855E-003,
    +1.433344092E-003,
    +1.376571832E-003,
    +1.316190348E-003,
    +1.252829796E-003,
    +1.183370827E-003,
    +1.115717343E-003,
    +1.046945457E-003,
    +9.775133803E-004,
    +9.078957955E-004,
    +8.385353722E-004,
    +7.698345580E-004,
    +7.021900383E-004,
    +6.358824321E-004,
    +5.729619297E-004,
    +5.099929986E-004,
    +4.490280990E-004,
    +3.906481725E-004,
    +3.347633174E-004,
    +2.682086197E-004,
    +2.233728592E-004,
    +1.774113771E-004,
    +1.312203676E-004,
    +1.024175072E-004,
    +6.541742187E-005,
    +3.193307566E-005,
    +1.714242217E-006,
    -2.470704203E-005,
    -4.491530854E-005,
    -6.447290798E-005,
    -8.078388782E-005,
    -9.417100227E-005,
    -1.047207043E-004,
    -1.126275165E-004,
    -1.180980107E-004,
    -1.213575742E-004,
    -1.226499153E-004,
    -1.262027217E-004,
    -1.250260248E-004,
    -1.224978914E-004,
    -1.187910311E-004,
    -1.140582463E-004,
    -1.084438481E-004,
    -1.020687996E-004,
    -9.517164290E-005,
    -8.757545584E-005,
    -8.413690375E-005,
    -7.513730816E-005,
    -6.568001118E-005,
    -5.610509834E-005,
    -4.624524081E-005,
    -1.774702469E-005,
    -1.992636317E-005,
    +1.617751332E-005,
    +1.521241393E-005,
    -2.525620175E-006,
    -1.191342653E-005,
    -1.985177369E-005,
    -2.570833203E-005,
    -3.018750249E-005,
    -2.554462844E-005,
    -2.618136386E-005,
    -2.537086584E-005,
    -2.362549276E-005,
    -2.093936746E-005,
    -1.751866148E-005,
    -1.356193479E-005,
    -9.299508747E-006,
    -4.916387297E-006,
    -5.595519906E-005,
    -5.429694284E-005,
    -5.176846025E-005,
    -4.839275425E-005,
    -4.425736552E-005,
    -3.946387005E-005,
    -3.415624451E-005,
    -2.847247197E-005,
    -2.263354872E-005,
    -1.309637537E-005,
    -9.561075785E-006,
    -6.084746474E-006,
    -3.086857760E-006,
    -7.198007665E-007,
    +1.853591357E-006,
    +1.470520488E-006,
    +9.984935332E-007,
    +5.633860383E-007,
    +6.993315310E-006,
    +1.207003334E-005,
    +1.661818715E-005,
    +1.776598037E-005,
    +1.806914770E-005,
    +2.677291741E-005,
    +2.978993689E-005,
    +3.225446198E-005,
    +3.397853652E-005,
    +3.495384954E-005,
    +3.515914432E-005,
    +3.460439257E-005,
    +3.334947178E-005,
    +3.144468428E-005,
    -2.042970118E-005,
    -1.987093310E-005,
    -1.884208177E-005,
    -1.743183020E-005,
    -1.571428584E-005,
    -1.379448349E-005,
    -1.177108606E-005,
    -9.826449968E-006,
    -7.869333785E-006,
    -2.319611212E-006,
    -3.157242645E-006,
    -3.617151151E-006,
    -2.497214155E-006,
    -1.189317118E-006,
    -6.437721822E-008,
    -8.434094667E-008,
    +3.809887872E-008,
    +3.246264413E-008,
    -6.402664354E-008,
    -2.975164826E-008,
    -9.458596395E-008,
    -4.341498823E-007,
    -8.427224429E-007,
    -3.332600045E-006,
    -4.429412002E-006,
    -5.315936960E-006,
    -6.225028756E-006,
    -6.986399967E-006,
    -7.568834008E-006,
    -7.941360309E-006,
    -8.087732567E-006,
    -8.006985809E-006,
    -6.361419310E-006,
    -6.015239251E-006,
    -5.516381407E-006,
    -4.896494374E-006,
    -4.195037945E-006,
    -3.453840463E-006,
    -2.715013807E-006,
    -2.032118118E-006,
    -1.399798180E-006,
    -3.132386439E-007,
    -1.444918922E-007,
    -2.788728892E-008,
    -7.506701927E-009,
    -1.314406006E-008,
    +3.399493131E-009,
    +2.682709610E-009,
    +1.613285505E-009,
    +6.702406963E-010,
    -7.834767501E-008,
    -2.014677420E-007,
    -5.234479659E-007,
    -1.190443072E-006,
    -1.975324267E-006,
    -9.451737242E-007,
    -1.033426656E-006,
    -1.179182050E-006,
    -1.194632091E-006,
    -1.162929266E-006,
    -1.085227950E-006,
    -9.742954035E-007,
    -8.419976893E-007,
    -7.003467317E-007,
    +5.564140224E-007,
    +6.262345096E-007,
    +6.767839409E-007,
    +7.020648809E-007,
    +6.982898526E-007,
    +6.628192182E-007,
    +6.022448247E-007,
    +4.741137047E-007,
    +3.970030775E-007,
    +7.342250683E-007,
    +3.961981463E-007,
    +1.543309907E-007,
    +5.083275667E-008,
    +1.608403011E-008,
    -7.018770981E-011,
    -1.135985195E-010
};

DECLARE_ALIGNED(16, const float, ff_dca_fir_32bands_nonperfect)[512] = {
    -1.390191784E-007,
    -1.693738625E-007,
    -2.030677564E-007,
    -2.404238444E-007,
    -2.818143514E-007,
    -3.276689142E-007,
    -3.784752209E-007,
    -4.347855338E-007,
    -4.972276315E-007,
    -5.665120852E-007,
    -6.434325428E-007,
    -7.288739425E-007,
    -8.238164355E-007,
    -9.293416952E-007,
    -1.046637067E-006,
    -1.176999604E-006,
    -1.321840614E-006,
    -1.482681114E-006,
    -1.661159786E-006,
    -1.859034001E-006,
    -2.078171747E-006,
    -2.320550948E-006,
    -2.588257530E-006,
    -2.883470643E-006,
    -3.208459020E-006,
    -3.565570978E-006,
    -3.957220997E-006,
    -4.385879038E-006,
    -4.854050530E-006,
    -5.364252502E-006,
    -5.918994248E-006,
    -6.520755960E-006,
    -7.171964626E-006,
    -7.874960829E-006,
    -8.631964192E-006,
    -9.445050637E-006,
    -1.031611009E-005,
    -1.124680875E-005,
    -1.223855270E-005,
    -1.329243969E-005,
    -1.440921824E-005,
    -1.558924305E-005,
    -1.683242772E-005,
    -1.813820381E-005,
    -1.950545993E-005,
    -2.093250441E-005,
    -2.241701623E-005,
    -2.395598858E-005,
    -2.554569073E-005,
    -2.718161704E-005,
    -2.885844333E-005,
    -3.056998685E-005,
    -3.230916263E-005,
    -3.406793985E-005,
    -3.583733633E-005,
    -3.760734762E-005,
    -3.936696885E-005,
    -4.110412556E-005,
    -4.280570283E-005,
    -4.445751256E-005,
    -4.604430433E-005,
    -4.754976908E-005,
    -4.895655002E-005,
    -5.024627535E-005,
    +5.139957648E-005,
    +5.239612074E-005,
    +5.321469871E-005,
    +5.383323878E-005,
    +5.422891263E-005,
    +5.437819709E-005,
    +5.425697600E-005,
    +5.384063843E-005,
    +5.310418419E-005,
    +5.202236207E-005,
    +5.056979353E-005,
    +4.872112549E-005,
    +4.645117951E-005,
    +4.373511547E-005,
    +4.054862075E-005,
    +3.686808850E-005,
    +3.267079956E-005,
    +2.793515523E-005,
    +2.264085742E-005,
    +1.676913780E-005,
    +1.030297699E-005,
    +3.227306706E-006,
    -4.470633485E-006,
    -1.280130618E-005,
    -2.177240640E-005,
    -3.138873581E-005,
    -4.165195787E-005,
    -5.256036457E-005,
    -6.410864444E-005,
    -7.628766616E-005,
    -8.908427117E-005,
    -1.024810626E-004,
    -1.164562127E-004,
    -1.309833024E-004,
    -1.460311323E-004,
    -1.615635992E-004,
    -1.775395358E-004,
    -1.939126523E-004,
    -2.106313768E-004,
    -2.276388550E-004,
    -2.448728774E-004,
    -2.622658503E-004,
    -2.797449124E-004,
    -2.972317743E-004,
    -3.146430245E-004,
    -3.318900708E-004,
    -3.488793736E-004,
    -3.655125911E-004,
    -3.816867538E-004,
    -3.972945851E-004,
    -4.122247046E-004,
    -4.263620067E-004,
    -4.395879805E-004,
    -4.517810594E-004,
    -4.628172028E-004,
    -4.725702747E-004,
    -4.809123348E-004,
    -4.877146275E-004,
    -4.928477574E-004,
    -4.961824161E-004,
    -4.975944757E-004,
    -4.969481961E-004,
    -4.941228544E-004,
    -4.889960401E-004,
    +4.814492422E-004,
    +4.713678791E-004,
    +4.586426076E-004,
    +4.431701091E-004,
    +4.248536134E-004,
    +4.036037717E-004,
    +3.793396754E-004,
    +3.519894381E-004,
    +3.214911267E-004,
    +2.877934603E-004,
    +2.508567995E-004,
    +2.106537577E-004,
    +1.671699720E-004,
    +1.204049113E-004,
    +7.037253090E-005,
    +1.710198012E-005,
    -3.936182839E-005,
    -9.895755647E-005,
    -1.616069785E-004,
    -2.272142592E-004,
    -2.956659591E-004,
    -3.668301215E-004,
    -4.405563814E-004,
    -5.166754709E-004,
    -5.949990009E-004,
    -6.753197522E-004,
    -7.574109477E-004,
    -8.410271257E-004,
    -9.259034996E-004,
    -1.011756598E-003,
    -1.098284614E-003,
    -1.185167348E-003,
    -1.272067428E-003,
    -1.358630019E-003,
    -1.444484224E-003,
    -1.529243193E-003,
    -1.612505526E-003,
    -1.693855622E-003,
    -1.772865304E-003,
    -1.849094522E-003,
    -1.922092517E-003,
    -1.991399564E-003,
    -2.056547208E-003,
    -2.117061289E-003,
    -2.172462177E-003,
    -2.222266514E-003,
    -2.265989315E-003,
    -2.303145360E-003,
    -2.333251061E-003,
    -2.355825622E-003,
    -2.370394068E-003,
    -2.376487479E-003,
    -2.373647178E-003,
    -2.361423569E-003,
    -2.339380793E-003,
    -2.307097195E-003,
    -2.264167881E-003,
    -2.210205887E-003,
    -2.144844970E-003,
    -2.067740774E-003,
    -1.978572691E-003,
    -1.877046190E-003,
    -1.762894331E-003,
    -1.635878929E-003,
    +1.495792647E-003,
    +1.342460280E-003,
    +1.175740734E-003,
    +9.955273708E-004,
    +8.017504588E-004,
    +5.943773431E-004,
    +3.734139318E-004,
    +1.389056415E-004,
    -1.090620208E-004,
    -3.703625989E-004,
    -6.448282511E-004,
    -9.322494152E-004,
    -1.232374110E-003,
    -1.544908970E-003,
    -1.869517611E-003,
    -2.205822384E-003,
    -2.553403843E-003,
    -2.911801683E-003,
    -3.280514618E-003,
    -3.659002949E-003,
    -4.046686925E-003,
    -4.442950245E-003,
    -4.847140983E-003,
    -5.258570891E-003,
    -5.676518660E-003,
    -6.100233644E-003,
    -6.528933067E-003,
    -6.961807609E-003,
    -7.398022339E-003,
    -7.836719044E-003,
    -8.277016692E-003,
    -8.718019351E-003,
    -9.158811532E-003,
    -9.598465636E-003,
    -1.003604382E-002,
    -1.047059800E-002,
    -1.090117730E-002,
    -1.132682897E-002,
    -1.174659748E-002,
    -1.215953380E-002,
    -1.256469358E-002,
    -1.296114177E-002,
    -1.334795821E-002,
    -1.372423489E-002,
    -1.408908330E-002,
    -1.444163360E-002,
    -1.478104480E-002,
    -1.510649733E-002,
    -1.541720331E-002,
    -1.571240649E-002,
    -1.599138230E-002,
    -1.625344716E-002,
    -1.649795473E-002,
    -1.672429405E-002,
    -1.693190821E-002,
    -1.712027565E-002,
    -1.728892699E-002,
    -1.743743755E-002,
    -1.756543480E-002,
    -1.767260395E-002,
    -1.775865816E-002,
    -1.782339066E-002,
    -1.786663756E-002,
    -1.788828894E-002,
    +1.788828894E-002,
    +1.786663756E-002,
    +1.782339066E-002,
    +1.775865816E-002,
    +1.767260395E-002,
    +1.756543480E-002,
    +1.743743755E-002,
    +1.728892699E-002,
    +1.712027565E-002,
    +1.693190821E-002,
    +1.672429405E-002,
    +1.649795473E-002,
    +1.625344716E-002,
    +1.599138230E-002,
    +1.571240649E-002,
    +1.541720331E-002,
    +1.510649733E-002,
    +1.478104480E-002,
    +1.444163360E-002,
    +1.408908330E-002,
    +1.372423489E-002,
    +1.334795821E-002,
    +1.296114177E-002,
    +1.256469358E-002,
    +1.215953380E-002,
    +1.174659748E-002,
    +1.132682897E-002,
    +1.090117730E-002,
    +1.047059800E-002,
    +1.003604382E-002,
    +9.598465636E-003,
    +9.158811532E-003,
    +8.718019351E-003,
    +8.277016692E-003,
    +7.836719044E-003,
    +7.398022339E-003,
    +6.961807609E-003,
    +6.528933067E-003,
    +6.100233644E-003,
    +5.676518660E-003,
    +5.258570891E-003,
    +4.847140983E-003,
    +4.442950245E-003,
    +4.046686925E-003,
    +3.659002949E-003,
    +3.280514618E-003,
    +2.911801683E-003,
    +2.553403843E-003,
    +2.205822384E-003,
    +1.869517611E-003,
    +1.544908970E-003,
    +1.232374110E-003,
    +9.322494152E-004,
    +6.448282511E-004,
    +3.703625989E-004,
    +1.090620208E-004,
    -1.389056415E-004,
    -3.734139318E-004,
    -5.943773431E-004,
    -8.017504588E-004,
    -9.955273708E-004,
    -1.175740734E-003,
    -1.342460280E-003,
    -1.495792647E-003,
    +1.635878929E-003,
    +1.762894331E-003,
    +1.877046190E-003,
    +1.978572691E-003,
    +2.067740774E-003,
    +2.144844970E-003,
    +2.210205887E-003,
    +2.264167881E-003,
    +2.307097195E-003,
    +2.339380793E-003,
    +2.361423569E-003,
    +2.373647178E-003,
    +2.376487479E-003,
    +2.370394068E-003,
    +2.355825622E-003,
    +2.333251061E-003,
    +2.303145360E-003,
    +2.265989315E-003,
    +2.222266514E-003,
    +2.172462177E-003,
    +2.117061289E-003,
    +2.056547208E-003,
    +1.991399564E-003,
    +1.922092517E-003,
    +1.849094522E-003,
    +1.772865304E-003,
    +1.693855622E-003,
    +1.612505526E-003,
    +1.529243193E-003,
    +1.444484224E-003,
    +1.358630019E-003,
    +1.272067428E-003,
    +1.185167348E-003,
    +1.098284614E-003,
    +1.011756598E-003,
    +9.259034996E-004,
    +8.410271257E-004,
    +7.574109477E-004,
    +6.753197522E-004,
    +5.949990009E-004,
    +5.166754709E-004,
    +4.405563814E-004,
    +3.668301215E-004,
    +2.956659591E-004,
    +2.272142592E-004,
    +1.616069785E-004,
    +9.895755647E-005,
    +3.936182839E-005,
    -1.710198012E-005,
    -7.037253090E-005,
    -1.204049113E-004,
    -1.671699720E-004,
    -2.106537577E-004,
    -2.508567995E-004,
    -2.877934603E-004,
    -3.214911267E-004,
    -3.519894381E-004,
    -3.793396754E-004,
    -4.036037717E-004,
    -4.248536134E-004,
    -4.431701091E-004,
    -4.586426076E-004,
    -4.713678791E-004,
    -4.814492422E-004,
    +4.889960401E-004,
    +4.941228544E-004,
    +4.969481961E-004,
    +4.975944757E-004,
    +4.961824161E-004,
    +4.928477574E-004,
    +4.877146275E-004,
    +4.809123348E-004,
    +4.725702747E-004,
    +4.628172028E-004,
    +4.517810594E-004,
    +4.395879805E-004,
    +4.263620067E-004,
    +4.122247046E-004,
    +3.972945851E-004,
    +3.816867538E-004,
    +3.655125911E-004,
    +3.488793736E-004,
    +3.318900708E-004,
    +3.146430245E-004,
    +2.972317743E-004,
    +2.797449124E-004,
    +2.622658503E-004,
    +2.448728774E-004,
    +2.276388550E-004,
    +2.106313768E-004,
    +1.939126523E-004,
    +1.775395358E-004,
    +1.615635992E-004,
    +1.460311323E-004,
    +1.309833024E-004,
    +1.164562127E-004,
    +1.024810626E-004,
    +8.908427117E-005,
    +7.628766616E-005,
    +6.410864444E-005,
    +5.256036457E-005,
    +4.165195787E-005,
    +3.138873581E-005,
    +2.177240640E-005,
    +1.280130618E-005,
    +4.470633485E-006,
    -3.227306706E-006,
    -1.030297699E-005,
    -1.676913780E-005,
    -2.264085742E-005,
    -2.793515523E-005,
    -3.267079956E-005,
    -3.686808850E-005,
    -4.054862075E-005,
    -4.373511547E-005,
    -4.645117951E-005,
    -4.872112549E-005,
    -5.056979353E-005,
    -5.202236207E-005,
    -5.310418419E-005,
    -5.384063843E-005,
    -5.425697600E-005,
    -5.437819709E-005,
    -5.422891263E-005,
    -5.383323878E-005,
    -5.321469871E-005,
    -5.239612074E-005,
    -5.139957648E-005,
    +5.024627535E-005,
    +4.895655002E-005,
    +4.754976908E-005,
    +4.604430433E-005,
    +4.445751256E-005,
    +4.280570283E-005,
    +4.110412556E-005,
    +3.936696885E-005,
    +3.760734762E-005,
    +3.583733633E-005,
    +3.406793985E-005,
    +3.230916263E-005,
    +3.056998685E-005,
    +2.885844333E-005,
    +2.718161704E-005,
    +2.554569073E-005,
    +2.395598858E-005,
    +2.241701623E-005,
    +2.093250441E-005,
    +1.950545993E-005,
    +1.813820381E-005,
    +1.683242772E-005,
    +1.558924305E-005,
    +1.440921824E-005,
    +1.329243969E-005,
    +1.223855270E-005,
    +1.124680875E-005,
    +1.031611009E-005,
    +9.445050637E-006,
    +8.631964192E-006,
    +7.874960829E-006,
    +7.171964626E-006,
    +6.520755960E-006,
    +5.918994248E-006,
    +5.364252502E-006,
    +4.854050530E-006,
    +4.385879038E-006,
    +3.957220997E-006,
    +3.565570978E-006,
    +3.208459020E-006,
    +2.883470643E-006,
    +2.588257530E-006,
    +2.320550948E-006,
    +2.078171747E-006,
    +1.859034001E-006,
    +1.661159786E-006,
    +1.482681114E-006,
    +1.321840614E-006,
    +1.176999604E-006,
    +1.046637067E-006,
    +9.293416952E-007,
    +8.238164355E-007,
    +7.288739425E-007,
    +6.434325428E-007,
    +5.665120852E-007,
    +4.972276315E-007,
    +4.347855338E-007,
    +3.784752209E-007,
    +3.276689142E-007,
    +2.818143514E-007,
    +2.404238444E-007,
    +2.030677564E-007,
    +1.693738625E-007,
    +1.390191784E-007
};

/* pre-scale lfe fir coefficients
 *
 * Permuted compared to table D.8 in the spec. First, the table in the
 * spec has twice the number of coefficients, but they are reflected,
 * T[i] = T[511 - i]. Furthermore, the indices are permuted, so that
 *
 *   lfe_fir_64[8 * k + j] = T[64 * j + k]
 *
 * for 0 <= k < 32 and 0 <= j < 8. (This corresponds to adding a
 * leading zero to the 8-bit index, and rotating it right by 3 bits
 * as a 9-bit field). Values with j >= 4 can be reflected, as
 *
 *   lfe_fir_64[8 * k + j] = T[64 * (7 - j) + (63 - k)]
 *
 * so all values in T get included. */
#define SCALE(c) ((c) / (256.0f * 32768.0f))
DECLARE_ALIGNED(16, const float, ff_dca_lfe_fir_64)[256] = {
    SCALE(2.658434386830777e-4), SCALE(9.029330685734748e-3),
    SCALE(7.939263433218002e-2), SCALE(2.425158768892288e-1),
    SCALE(3.430179357528686e-1), SCALE(2.398228943347931e-1),
    SCALE(7.746443897485733e-2), SCALE(8.622321300208569e-3),
    SCALE(8.179365249816328e-5), SCALE(9.450953453779220e-3),
    SCALE(8.134882897138596e-2), SCALE(2.451938837766648e-1),
    SCALE(3.429597318172455e-1), SCALE(2.371159791946411e-1),
    SCALE(7.556436210870743e-2), SCALE(8.229630999267101e-3),
    SCALE(9.439323912374676e-5), SCALE(9.887560270726680e-3),
    SCALE(8.333285897970200e-2), SCALE(2.478559017181396e-1),
    SCALE(3.428434133529663e-1), SCALE(2.343961596488952e-1),
    SCALE(7.369252294301987e-2), SCALE(7.850865833461285e-3),
    SCALE(1.082170274457894e-4), SCALE(1.033949479460716e-2),
    SCALE(8.534456789493561e-2), SCALE(2.505008876323700e-1),
    SCALE(3.426689505577088e-1), SCALE(2.316644787788391e-1),
    SCALE(7.184901088476181e-2), SCALE(7.485736627131701e-3),
    SCALE(1.233371440321207e-4), SCALE(1.080708485096693e-2),
    SCALE(8.738376945257187e-2), SCALE(2.531278133392334e-1),
    SCALE(3.424364924430847e-1), SCALE(2.289219647645950e-1),
    SCALE(7.003392279148102e-2), SCALE(7.133882027119398e-3),
    SCALE(1.397485757479444e-4), SCALE(1.129068247973919e-2),
    SCALE(8.945026248693466e-2), SCALE(2.557355761528015e-1),
    SCALE(3.421461284160614e-1), SCALE(2.261696159839630e-1),
    SCALE(6.824731826782227e-2), SCALE(6.794991903007030e-3),
    SCALE(1.575958012836054e-4), SCALE(1.179065089672804e-2),
    SCALE(9.154383838176728e-2), SCALE(2.583232223987580e-1),
    SCALE(3.417979776859284e-1), SCALE(2.234084606170654e-1),
    SCALE(6.648923456668854e-2), SCALE(6.468691397458315e-3),
    SCALE(1.769922382663936e-4), SCALE(1.230732165277004e-2),
    SCALE(9.366425126791000e-2), SCALE(2.608896791934967e-1),
    SCALE(3.413922190666198e-1), SCALE(2.206395119428635e-1),
    SCALE(6.475970894098282e-2), SCALE(6.154712289571762e-3),
    SCALE(1.981738605536520e-4), SCALE(1.284105982631445e-2),
    SCALE(9.581124037504196e-2), SCALE(2.634339034557342e-1),
    SCALE(3.409290313720703e-1), SCALE(2.178637981414795e-1),
    SCALE(6.305878609418869e-2), SCALE(5.852684378623962e-3),
    SCALE(2.211847313446924e-4), SCALE(1.339218579232693e-2),
    SCALE(9.798453748226166e-2), SCALE(2.659549415111542e-1),
    SCALE(3.404086530208588e-1), SCALE(2.150822728872299e-1),
    SCALE(6.138643622398376e-2), SCALE(5.562345497310162e-3),
    SCALE(2.460231189616024e-4), SCALE(1.396108977496624e-2),
    SCALE(1.001838669180870e-1), SCALE(2.684516608715058e-1),
    SCALE(3.398312926292420e-1), SCALE(2.122959494590759e-1),
    SCALE(5.974265560507774e-2), SCALE(5.283284001052380e-3),
    SCALE(2.726115926634520e-4), SCALE(1.454808749258518e-2),
    SCALE(1.024089083075523e-1), SCALE(2.709231376647949e-1),
    SCALE(3.391972482204438e-1), SCALE(2.095058411359787e-1),
    SCALE(5.812742188572884e-2), SCALE(5.015311297029257e-3),
    SCALE(3.013863170053810e-4), SCALE(1.515355054289102e-2),
    SCALE(1.046593263745308e-1), SCALE(2.733682692050934e-1),
    SCALE(3.385068178176880e-1), SCALE(2.067128717899322e-1),
    SCALE(5.654069408774376e-2), SCALE(4.758012015372515e-3),
    SCALE(3.328395541757345e-4), SCALE(1.577781140804291e-2),
    SCALE(1.069347932934761e-1), SCALE(2.757860720157624e-1),
    SCALE(3.377602994441986e-1), SCALE(2.039180546998978e-1),
    SCALE(5.498242005705833e-2), SCALE(4.511159844696522e-3),
    SCALE(3.658991190604866e-4), SCALE(1.642123050987720e-2),
    SCALE(1.092349365353584e-1), SCALE(2.781755328178406e-1),
    SCALE(3.369580209255218e-1), SCALE(2.011223286390304e-1),
    SCALE(5.345252528786659e-2), SCALE(4.274417180567980e-3),
    SCALE(4.018281470052898e-4), SCALE(1.708412915468216e-2),
    SCALE(1.115593686699867e-1), SCALE(2.805356979370117e-1),
    SCALE(3.361004292964936e-1), SCALE(1.983266174793244e-1),
    SCALE(5.195093154907227e-2), SCALE(4.047499038279056e-3),
    SCALE(4.401875485200435e-4), SCALE(1.776690222322941e-2),
    SCALE(1.139076948165894e-1), SCALE(2.828655838966370e-1),
    SCALE(3.351879119873047e-1), SCALE(1.955319195985794e-1),
    SCALE(5.047753453254700e-2), SCALE(3.830091329291463e-3),
    SCALE(4.812776169274002e-4), SCALE(1.846982724964619e-2),
    SCALE(1.162794977426529e-1), SCALE(2.851640880107880e-1),
    SCALE(3.342207968235016e-1), SCALE(1.927391141653061e-1),
    SCALE(4.903224110603333e-2), SCALE(3.621967276558280e-3),
    SCALE(5.252459668554366e-4), SCALE(1.919330470263958e-2),
    SCALE(1.186743453145027e-1), SCALE(2.874303460121155e-1),
    SCALE(3.331996202468872e-1), SCALE(1.899491697549820e-1),
    SCALE(4.761491715908051e-2), SCALE(3.422776935622096e-3),
    SCALE(5.721592460758984e-4), SCALE(1.993762329220772e-2),
    SCALE(1.210917681455612e-1), SCALE(2.896633744239807e-1),
    SCALE(3.321248590946198e-1), SCALE(1.871629506349564e-1),
    SCALE(4.622544348239899e-2), SCALE(3.232272574678064e-3),
    SCALE(6.222130032256246e-4), SCALE(2.070316113531590e-2),
    SCALE(1.235313042998314e-1), SCALE(2.918621897697448e-1),
    SCALE(3.309969604015350e-1), SCALE(1.843813359737396e-1),
    SCALE(4.486365616321564e-2), SCALE(3.050152910873294e-3),
    SCALE(6.755515350960195e-4), SCALE(2.149021252989769e-2),
    SCALE(1.259924471378326e-1), SCALE(2.940258979797364e-1),
    SCALE(3.298164308071136e-1), SCALE(1.816052496433258e-1),
    SCALE(4.352942481637001e-2), SCALE(2.876190468668938e-3),
    SCALE(7.324148900806904e-4), SCALE(2.229913882911205e-2),
    SCALE(1.284746825695038e-1), SCALE(2.961534857749939e-1),
    SCALE(3.285838961601258e-1), SCALE(1.788355410099030e-1),
    SCALE(4.222255200147629e-2), SCALE(2.710093278437853e-3),
    SCALE(7.928516715764999e-4), SCALE(2.313023805618286e-2),
    SCALE(1.309774816036224e-1), SCALE(2.982441186904907e-1),
    SCALE(3.272998929023742e-1), SCALE(1.760730892419815e-1),
    SCALE(4.094288870692253e-2), SCALE(2.551567042246461e-3),
    SCALE(8.570110658183694e-4), SCALE(2.398385666310787e-2),
    SCALE(1.335003077983856e-1), SCALE(3.002967536449432e-1),
    SCALE(3.259649574756622e-1), SCALE(1.733186990022659e-1),
    SCALE(3.969023004174232e-2), SCALE(2.400433411821723e-3),
    SCALE(9.251192095689476e-4), SCALE(2.486028522253036e-2),
    SCALE(1.360425949096680e-1), SCALE(3.023106753826142e-1),
    SCALE(3.245797157287598e-1), SCALE(1.705732345581055e-1),
    SCALE(3.846437484025955e-2), SCALE(2.256359672173858e-3),
    SCALE(9.974770946428180e-4), SCALE(2.575986087322235e-2),
    SCALE(1.386037617921829e-1), SCALE(3.042849004268646e-1),
    SCALE(3.231448531150818e-1), SCALE(1.678375005722046e-1),
    SCALE(3.726511076092720e-2), SCALE(2.119151875376701e-3),
    SCALE(1.073930296115577e-3), SCALE(2.668286114931106e-2),
    SCALE(1.411831974983215e-1), SCALE(3.062185347080230e-1),
    SCALE(3.216609656810760e-1), SCALE(1.651122719049454e-1),
    SCALE(3.609224036335945e-2), SCALE(1.988604199141264e-3),
    SCALE(1.155023579485714e-3), SCALE(2.762960828840732e-2),
    SCALE(1.437802612781525e-1), SCALE(3.081108033657074e-1),
    SCALE(3.201287388801574e-1), SCALE(1.623983532190323e-1),
    SCALE(3.494550660252571e-2), SCALE(1.864377525635064e-3),
    SCALE(1.240676851011813e-3), SCALE(2.860039286315441e-2),
    SCALE(1.463943719863892e-1), SCALE(3.099608123302460e-1),
    SCALE(3.185488879680634e-1), SCALE(1.596965193748474e-1),
    SCALE(3.382468968629837e-2), SCALE(1.746327499859035e-3),
    SCALE(1.331258914433420e-3), SCALE(2.959549613296986e-2),
    SCALE(1.490248143672943e-1), SCALE(3.117676973342896e-1),
    SCALE(3.169221282005310e-1), SCALE(1.570075154304504e-1),
    SCALE(3.272953629493714e-2), SCALE(1.634211512282491e-3),
    SCALE(1.426893868483603e-3), SCALE(3.061520494520664e-2),
    SCALE(1.516709625720978e-1), SCALE(3.135308027267456e-1),
    SCALE(3.152491748332978e-1), SCALE(1.543320864439010e-1),
    SCALE(3.165979683399200e-2), SCALE(1.527829794213176e-3),
};

/* Permuted compared to table D.8 in the spec. First, the table in the
 * spec has twice the number of coefficients, but they are reflected,
 * T[i] = T[511 - i]. Furthermore, the indices are permuted, so that
 *
 *   lfe_fir_128[4 * k + j] = T[128 * j + k]
 *
 * for 0 <= k < 64 and 0 <= j < 4. Values with j >= 2 can be
 * reflected, as
 *
 *   lfe_fir_128[4 * k + j] = T[128 * (3 - j) + (127 - k)]
 *
 * so all values in T get included. */
DECLARE_ALIGNED(16, const float, ff_dca_lfe_fir_128)[256] = {
    SCALE(0.00053168571), SCALE(0.15878495574), SCALE(0.68603444099), SCALE(0.15492856503),
    SCALE(0.00016358691), SCALE(0.16269733012), SCALE(0.68591803312), SCALE(0.15112841129),
    SCALE(0.00018878609), SCALE(0.16666537523), SCALE(0.68568539619), SCALE(0.14738474786),
    SCALE(0.00021643363), SCALE(0.17068879306), SCALE(0.68533653021), SCALE(0.14369773865),
    SCALE(0.00024667382), SCALE(0.17476719618), SCALE(0.68487155437), SCALE(0.14006754756),
    SCALE(0.00027949660), SCALE(0.17890018225), SCALE(0.68429082632), SCALE(0.13649433851),
    SCALE(0.00031519096), SCALE(0.18308731914), SCALE(0.68359452486), SCALE(0.13297818601),
    SCALE(0.00035398375), SCALE(0.18732811511), SCALE(0.68278300762), SCALE(0.12951917946),
    SCALE(0.00039634691), SCALE(0.19162209332), SCALE(0.68185669184), SCALE(0.12611730397),
    SCALE(0.00044236859), SCALE(0.19596865773), SCALE(0.68081587553), SCALE(0.12277261168),
    SCALE(0.00049204525), SCALE(0.20036731660), SCALE(0.67966115475), SCALE(0.11948505789),
    SCALE(0.00054522208), SCALE(0.20481738448), SCALE(0.67839306593), SCALE(0.11625462025),
    SCALE(0.00060277141), SCALE(0.20931822062), SCALE(0.67701220512), SCALE(0.11308115721),
    SCALE(0.00066567765), SCALE(0.21386915445), SCALE(0.67551922798), SCALE(0.10996460915),
    SCALE(0.00073179678), SCALE(0.21846942604), SCALE(0.67391467094), SCALE(0.10690483451),
    SCALE(0.00080365466), SCALE(0.22311829031), SCALE(0.67219948769), SCALE(0.10390164703),
    SCALE(0.00088037323), SCALE(0.22781492770), SCALE(0.67037439346), SCALE(0.10095486045),
    SCALE(0.00096255314), SCALE(0.23255851865), SCALE(0.66844022274), SCALE(0.09806428105),
    SCALE(0.00105048984), SCALE(0.23734821379), SCALE(0.66639786959), SCALE(0.09522963315),
    SCALE(0.00114431616), SCALE(0.24218304455), SCALE(0.66424828768), SCALE(0.09245070815),
    SCALE(0.00124442333), SCALE(0.24706205726), SCALE(0.66199249029), SCALE(0.08972713351),
    SCALE(0.00135110028), SCALE(0.25198432803), SCALE(0.65963155031), SCALE(0.08705867827),
    SCALE(0.00146482687), SCALE(0.25694879889), SCALE(0.65716648102), SCALE(0.08444493264),
    SCALE(0.00158570008), SCALE(0.26195442677), SCALE(0.65459835529), SCALE(0.08188561350),
    SCALE(0.00171401864), SCALE(0.26700007915), SCALE(0.65192854404), SCALE(0.07938029617),
    SCALE(0.00185023469), SCALE(0.27208462358), SCALE(0.64915806055), SCALE(0.07692859322),
    SCALE(0.00199495023), SCALE(0.27720692754), SCALE(0.64628833532), SCALE(0.07453006506),
    SCALE(0.00214785640), SCALE(0.28236576915), SCALE(0.64332056046), SCALE(0.07218432426),
    SCALE(0.00231004250), SCALE(0.28755992651), SCALE(0.64025616646), SCALE(0.06989086419),
    SCALE(0.00248134881), SCALE(0.29278811812), SCALE(0.63709646463), SCALE(0.06764923781),
    SCALE(0.00266251224), SCALE(0.29804900289), SCALE(0.63384294510), SCALE(0.06545893103),
    SCALE(0.00285378192), SCALE(0.30334126949), SCALE(0.63049703836), SCALE(0.06331945211),
    SCALE(0.00305565330), SCALE(0.30866351724), SCALE(0.62706029415), SCALE(0.06123027951),
    SCALE(0.00326841651), SCALE(0.31401440501), SCALE(0.62353414297), SCALE(0.05919086933),
    SCALE(0.00349264755), SCALE(0.31939238310), SCALE(0.61992025375), SCALE(0.05720067024),
    SCALE(0.00372874714), SCALE(0.32479602098), SCALE(0.61622029543), SCALE(0.05525910854),
    SCALE(0.00397720048), SCALE(0.33022382855), SCALE(0.61243581772), SCALE(0.05336561054),
    SCALE(0.00423829490), SCALE(0.33567428589), SCALE(0.60856848955), SCALE(0.05151961371),
    SCALE(0.00451271003), SCALE(0.34114575386), SCALE(0.60462015867), SCALE(0.04972046614),
    SCALE(0.00480085658), SCALE(0.34663668275), SCALE(0.60059231520), SCALE(0.04796761274),
    SCALE(0.00510312291), SCALE(0.35214546323), SCALE(0.59648692608), SCALE(0.04626038298),
    SCALE(0.00542017492), SCALE(0.35767036676), SCALE(0.59230577946), SCALE(0.04459818453),
    SCALE(0.00575236930), SCALE(0.36320972443), SCALE(0.58805054426), SCALE(0.04298033938),
    SCALE(0.00610029325), SCALE(0.36876192689), SCALE(0.58372318745), SCALE(0.04140623659),
    SCALE(0.00646453211), SCALE(0.37432509661), SCALE(0.57932555676), SCALE(0.03987516090),
    SCALE(0.00684553990), SCALE(0.37989753485), SCALE(0.57485944033), SCALE(0.03838652745),
    SCALE(0.00724391919), SCALE(0.38547745347), SCALE(0.57032698393), SCALE(0.03693958372),
    SCALE(0.00766016589), SCALE(0.39106300473), SCALE(0.56572991610), SCALE(0.03553372994),
    SCALE(0.00809498038), SCALE(0.39665243030), SCALE(0.56107026339), SCALE(0.03416819125),
    SCALE(0.00854881573), SCALE(0.40224379301), SCALE(0.55634999275), SCALE(0.03284239396),
    SCALE(0.00902230106), SCALE(0.40783521533), SCALE(0.55157101154), SCALE(0.03155555204),
    SCALE(0.00951600447), SCALE(0.41342487931), SCALE(0.54673534632), SCALE(0.03030703776),
    SCALE(0.01003060210), SCALE(0.41901078820), SCALE(0.54184508324), SCALE(0.02909611352),
    SCALE(0.01056654565), SCALE(0.42459106445), SCALE(0.53690224886), SCALE(0.02792212367),
    SCALE(0.01112466771), SCALE(0.43016362190), SCALE(0.53190881014), SCALE(0.02678431384),
    SCALE(0.01170534454), SCALE(0.43572667241), SCALE(0.52686679363), SCALE(0.02568206564),
    SCALE(0.01230939943), SCALE(0.44127810001), SCALE(0.52177828550), SCALE(0.02461459488),
    SCALE(0.01293735672), SCALE(0.44681602716), SCALE(0.51664537191), SCALE(0.02358125709),
    SCALE(0.01358995494), SCALE(0.45233830810), SCALE(0.51147013903), SCALE(0.02258131653),
    SCALE(0.01426773332), SCALE(0.45784294605), SCALE(0.50625455379), SCALE(0.02161412500),
    SCALE(0.01497144438), SCALE(0.46332800388), SCALE(0.50100076199), SCALE(0.02067894675),
    SCALE(0.01570170000), SCALE(0.46879136562), SCALE(0.49571081996), SCALE(0.01977507770),
    SCALE(0.01645922661), SCALE(0.47423094511), SCALE(0.49038675427), SCALE(0.01890186779),
    SCALE(0.01724460535), SCALE(0.47964480519), SCALE(0.48503074050), SCALE(0.01805862412),
};
#undef SCALE

DECLARE_ALIGNED(16, const float, ff_dca_fir_64bands)[1024] = {
    /* Bank 0 */
    -7.1279389866041690e-8, -7.0950903150874990e-8,
    -7.9525034321375090e-8, -8.9326871281374790e-8,
    -9.7483190494874640e-8, -1.0586659510721950e-7,
    -1.1527363411865760e-7, -1.2523109645615350e-7,
    -1.3555636137971380e-7, -1.4640931821191640e-7,
    -1.5787424998123710e-7, -1.6994592926571220e-7,
    -1.8264184311018150e-7, -1.9600594587757850e-7,
    -2.1007899706006800e-7, -2.2489759047553620e-7,
    -2.4050245877923400e-7, -2.5693874615777660e-7,
    -2.7425370308549580e-7, -2.9249662541757600e-7,
    -3.1171981968268900e-7, -3.3197864040473180e-7,
    -3.5333134588843600e-7, -3.7583910465412180e-7,
    -3.9956628121669190e-7, -4.2458045296159330e-7,
    -4.5095249563698380e-7, -4.7875659112606760e-7,
    -5.0807031294631050e-7, -5.3897464223335550e-7,
    -5.7155423388793000e-7, -6.0589741192019800e-7,
    -6.4209580087853050e-7, -6.8024432849991400e-7,
    -7.2044165259013450e-7, -7.6279018172077200e-7,
    -8.0739610368280600e-7, -8.5436937020293650e-7,
    -9.0382333224895500e-7, -9.5587472540498280e-7,
    -1.0106437128213760e-6, -1.0682538906146330e-6,
    -1.1288323637155430e-6, -1.1925097184646100e-6,
    -1.2594193438633260e-6, -1.3296974052300520e-6,
    -1.4034829433780260e-6, -1.4809178634979940e-6,
    -1.5621465913980060e-6, -1.6473160377545360e-6,
    -1.7365751553971340e-6, -1.8300749104526120e-6,
    -1.9279680902322050e-6, -2.0304092660990760e-6,
    -2.1375541686662320e-6, -2.2495596253776460e-6,
    -2.3665828120950560e-6, -2.4887811970856540e-6,
    -2.6163120571644820e-6, -2.7493324266391140e-6,
    -2.8879984579501380e-6, -3.0324653639203880e-6,
    -3.1828867865974640e-6, -3.3394147240855820e-6,
    -3.5021985520037540e-6, -3.6713849262294680e-6,
    -3.8471166218015830e-6, -4.0295324657699800e-6,
    -4.2187670209852250e-6, -4.4149505043312320e-6,
    -4.6182073040543900e-6, -4.8286558658894680e-6,
    -5.0464076705003400e-6, -5.2715671455405850e-6,
    -5.5042307535458100e-6, -5.7444868785028750e-6,
    -5.9924142424804800e-6, -6.2480817863072800e-6,
    -6.5115476278604000e-6, -6.7828589565804950e-6,
    -7.0620507545154100e-6, -7.3491456697144900e-6,
    -7.6441525098592000e-6, -7.9470661364365490e-6,
    -8.2578666429227760e-6, -8.5765192154768100e-6,
    -8.9029718900122000e-6, -9.2371554195468400e-6,
    -9.5789825988253100e-6, -9.9283481667264540e-6,
    -1.0285127334608450e-5, -1.0649175634328590e-5,
    -1.1020327046334650e-5, -1.1398393880881060e-5,
    -1.1783166036693560e-5, -1.2174410889256530e-5,
    -1.2571871574469220e-5, -1.2975266858030310e-5,
    -1.3384289978242060e-5, -1.3798608540965960e-5,
    -1.4217863365843780e-5, -1.4641668383496450e-5,
    -1.5069609527679710e-5, -1.5501244597782600e-5,
    -1.5936101341640500e-5, -1.6373677426466900e-5,
    -1.6813441716539560e-5, -1.7254834143068570e-5,
    -1.7697261563607500e-5, -1.8140097714618200e-5,
    -1.8582686308483550e-5, -1.9024340970099240e-5,
    -1.9464340744880170e-5, -1.9901930030141080e-5,
    -2.0336321557464510e-5, -2.0766696417773940e-5,
    -2.1192201630574280e-5, -2.1611950084231910e-5,
    -2.2025021652894630e-5, -2.2430463239517170e-5,
    -2.2827288605503470e-5, -2.3214478371918170e-5,
    -2.3590980216497750e-5, -2.3955708939763280e-5,
    -2.4307547722489910e-5, -2.4645348200903720e-5,
    -2.4967930863542620e-5, -2.5274085082324960e-5,
    /* Bank 1 */
     2.5562569399199020e-5,  2.5832111724562040e-5,
     2.6081413415833200e-5,  2.6309149390520640e-5,
     2.6513966535124460e-5,  2.6694483850464800e-5,
     2.6849297239997120e-5,  2.6976979753245170e-5,
     2.7076082153012070e-5,  2.7145133103663890e-5,
     2.7182642746157720e-5,  2.7187102980536760e-5,
     2.7156990105794550e-5,  2.7090765064735940e-5,
     2.6986876191054700e-5,  2.6843759539035120e-5,
     2.6659843389908530e-5,  2.6433548592482960e-5,
     2.6163291550639320e-5,  2.5847484577439320e-5,
     2.5484540698002960e-5,  2.5072874052447260e-5,
     2.4610903957661350e-5,  2.4097055300584120e-5,
     2.3529763128539150e-5,  2.2907473117403740e-5,
     2.2228647280994560e-5,  2.1491764461832140e-5,
     2.0695325418576370e-5,  1.9837853265502040e-5,
    /* NOTE: The spec has a duplicated row "160 | 1.7934037335534700e-5" ! */
     1.8917898053528080e-5,  1.7934037335534700e-5,
     1.6884884022311310e-5,  1.5769086921577900e-5,
     1.4585334740507920e-5,  1.3332356590263080e-5,
     1.2008929083235120e-5,  1.0613876973541650e-5,
     9.1460801506922300e-6,  7.6044740757265500e-6,
     5.9880523825897000e-6,  4.2958677847708840e-6,
     2.5270494199802070e-6,  6.8080315386959900e-7,
    -1.2435990827839660e-6, -3.2468187066945930e-6,
    -5.3294268968052800e-6, -7.4919036559427900e-6,
    -9.7346407524871100e-6, -1.2057941408635810e-5,
    -1.4462010500550970e-5, -1.6946953848107710e-5,
    -1.9512772389215240e-5, -2.2159361597292710e-5,
    -2.4886504491416160e-5, -2.7693871041862060e-5,
    -3.0581012081916960e-5, -3.3547358765259040e-5,
    -3.6592216716540970e-5, -3.9714765485539260e-5,
    -4.2914052387856640e-5, -4.6188991961724560e-5,
    -4.9538360176954340e-5, -5.2960794000140600e-5,
    -5.6454787620185350e-5, -6.0018691972791050e-5,
    -6.3650708053788550e-5, -6.7348886525882700e-5,
    -7.1111125753722860e-5, -7.4935171411178040e-5,
    -7.8818609806320000e-5, -8.2758867628623800e-5,
    -8.6753213062403400e-5, -9.0798755482139240e-5,
    -9.4892437636254000e-5, -9.9031035415475500e-5,
    -1.0321116057304920e-4, -1.0742926070674010e-4,
    -1.1168161614264270e-4, -1.1596433966772030e-4,
    -1.2027337376184880e-4, -1.2460449089034750e-4,
    -1.2895330270761850e-4, -1.3331525983885840e-4,
    -1.3768563787408340e-4, -1.4205953760578120e-4,
    -1.4643190426323540e-4, -1.5079752791692290e-4,
    -1.5515103314094370e-4, -1.5948687909058970e-4,
    -1.6379937154642740e-4, -1.6808266356446820e-4,
    -1.7233075773584680e-4, -1.7653750646560020e-4,
    -1.8069661584612290e-4, -1.8480164638953740e-4,
    -1.8884602526638050e-4, -1.9282304696531290e-4,
    -1.9672587556804270e-4, -2.0054754547108610e-4,
    -2.0428097498957360e-4, -2.0791896735659270e-4,
    -2.1145421910491280e-4, -2.1487932080488700e-4,
    -2.1818676491655660e-4, -2.2136894706917190e-4,
    -2.2441818635854840e-4, -2.2732672676686290e-4,
    -2.3008674810081220e-4, -2.3269036653525150e-4,
    -2.3512963563507800e-4, -2.3739654799291240e-4,
    -2.3948307023621970e-4, -2.4138114593372380e-4,
    -2.4308272452941820e-4, -2.4457975717687980e-4,
    -2.4586407616017000e-4, -2.4692741569938220e-4,
    -2.4776199025177680e-4, -2.4836046092637280e-4,
    -2.4871461830381660e-4, -2.4881540457247040e-4,
    -2.4865471770248810e-4, -2.4822543333608940e-4,
    -2.4752008219951220e-4, -2.4653081794176570e-4,
    -2.4525003217524440e-4, -2.4367037437341020e-4,
    /* Bank 2 */
     2.4178457457919800e-4,  2.3958543985834340e-4,
     2.3706595356421380e-4,  2.3421928008696620e-4,
     2.3103877007019660e-4,  2.2751796132972940e-4,
     2.2365059385031200e-4,  2.1943061159567880e-4,
     2.1485218733324290e-4,  2.0990972538755550e-4,
     2.0459789739171640e-4,  1.9891164424316180e-4,
     1.9284618337957900e-4,  1.8639701070854740e-4,
     1.7955993578371750e-4,  1.7233108395155490e-4,
     1.6470690840351980e-4,  1.5668419239382670e-4,
     1.4826008597798460e-4,  1.3943210818407590e-4,
     1.3019815804235450e-4,  1.2055651618705740e-4,
     1.1050586906665980e-4,  1.0004531116378850e-4,
     8.9174369623037550e-5,  7.7893006105444910e-5,
     6.6201632959158800e-5,  5.4101115008963200e-5,
     4.1592792755171040e-5,  2.8678484003389880e-5,
     1.5360496520469550e-5,  1.6416297019304470e-6,
    -1.2474801400564880e-5, -2.6984964757674160e-5,
    -4.1884506316022560e-5, -5.7168548169515050e-5,
    -7.2831653374563650e-5, -8.8867825335203200e-5,
    -1.0527052946041040e-4, -1.2203269185287530e-4,
    -1.3914664880465400e-4, -1.5660414556657680e-4,
    -1.7439635977956720e-4, -1.9251390142613000e-4,
    -2.1094678833176480e-4, -2.2968444457655940e-4,
    -2.4871569004884240e-4, -2.6802874127778880e-4,
    -2.8761124019708440e-4, -3.0745025297166030e-4,
    -3.2753221542219500e-4, -3.4784293347385130e-4,
    -3.6836764760397900e-4, -3.8909103335715320e-4,
    -4.0999714822829530e-4, -4.3106943097409840e-4,
    -4.5229073958127750e-4, -4.7364335350426610e-4,
    -4.9510898492628230e-4, -5.1666877822680600e-4,
    -5.3830328700499900e-4, -5.5999247625413450e-4,
    -5.8171579317559550e-4, -6.0345216810040500e-4,
    -6.2517996394886800e-4, -6.4687697763330650e-4,
    -6.6852052145861900e-4, -6.9008742526480550e-4,
    -7.1155400216196750e-4, -7.3289605035434600e-4,
    -7.5408892753413600e-4, -7.7510755325530950e-4,
    -7.9592638676398950e-4, -8.1651942936660300e-4,
    -8.3686029869896850e-4, -8.5692223230171550e-4,
    -8.7667809201044900e-4, -8.9610036547340950e-4,
    -9.1516119514986000e-4, -9.3383238324995840e-4,
    -9.5208547141354700e-4, -9.6989174214394400e-4,
    -9.8722217067251400e-4, -1.0040474297445790e-3,
    -1.0203380430448350e-3, -1.0360643900964910e-3,
    -1.0511966606168980e-3, -1.0657048556038330e-3,
    -1.0795588568782010e-3, -1.0927284332850520e-3,
    -1.1051833075734240e-3, -1.1168931605293980e-3,
    -1.1278276550234160e-3, -1.1379564405661050e-3,
    -1.1472492294891640e-3, -1.1556758000499310e-3,
    -1.1632059885511800e-3, -1.1698096960154200e-3,
    -1.1754570428998680e-3, -1.1801183720212010e-3,
    -1.1837641581852870e-3, -1.1863650153060240e-3,
    -1.1878919433902330e-3, -1.1883163316843900e-3,
    -1.1876097809178440e-3, -1.1857441072376680e-3,
    -1.1826916065321440e-3, -1.1784250628415020e-3,
    -1.1729176690707690e-3, -1.1661430294353880e-3,
    -1.1580752925519680e-3, -1.1486891568700730e-3,
    -1.1379598570807350e-3, -1.1258631690530700e-3,
    -1.1123755320313420e-3, -1.0974740543942100e-3,
    -1.0811365181610850e-3, -1.0633413819325920e-3,
    -1.0440678410743190e-3, -1.0232958338441730e-3,
    -1.0010061160103870e-3, -9.7718026613602800e-4,
    -9.5180072722007060e-4, -9.2485080945546600e-4,
    -8.9631470926558100e-4, -8.6617751390929650e-4,
    -8.3442528375666700e-4, -8.0104505655578600e-4,
    /* Bank 3 */
     7.6602485902806300e-4,  7.2935371157849700e-4,
     6.9102172032639350e-4,  6.5102007948960100e-4,
     6.0934103180698750e-4,  5.6597787244591100e-4,
     5.2092507454409450e-4,  4.7417829303612970e-4,
     4.2573432329476060e-4,  3.7559110239316400e-4,
     3.2374777823389440e-4,  2.7020471361387040e-4,
     2.1496350652513040e-4,  1.5802699163104190e-4,
     9.9399252442146460e-5,  3.9085623279149610e-5,
    -2.2907279717153710e-5, -8.6571556658577800e-5,
    -1.5189801251884410e-4, -2.1887615668780720e-4,
    -2.8749420455512610e-4, -3.5773907519488100e-4,
    -4.2959633884346400e-4, -5.0305021853266900e-4,
    -5.7808367853976800e-4, -6.5467842279709950e-4,
    -7.3281477145293300e-4, -8.1247166091487090e-4,
    -8.9362676824381250e-4, -9.7625651350435400e-4,
    -1.0603359870796300e-3, -1.1458389497708370e-3,
    -1.2327379076720450e-3, -1.3210041125919850e-3,
    -1.4106074964587530e-3, -1.5015166759080980e-3,
    -1.5936991188210680e-3, -1.6871211446584360e-3,
    -1.7817477652804130e-3, -1.8775426862784720e-3,
    -1.9744684954557180e-3, -2.0724866717198650e-3,
    -2.1715575922482770e-3, -2.2716405301701940e-3,
    -2.3726935964105480e-3, -2.4746737418493600e-3,
    -2.5775368629845300e-3, -2.6812378156433060e-3,
    -2.7857306109716460e-3, -2.8909684129992950e-3,
    -2.9969032890820140e-3, -3.1034862132032660e-3,
    -3.2106673881690480e-3, -3.3183962566563890e-3,
    -3.4266214220855450e-3, -3.5352906520539890e-3,
    -3.6443510330122110e-3, -3.7537489709635930e-3,
    -3.8634300521859240e-3, -3.9733390588103910e-3,
    -4.0834204508378470e-3, -4.1936183686073460e-3,
    -4.3038762050876900e-3, -4.4141366059086620e-3,
    -4.5243418977567100e-3, -4.6344341075783170e-3,
    -4.7443549566589130e-3, -4.8540458565358850e-3,
    -4.9634478250060780e-3, -5.0725014978718950e-3,
    -5.1811474713480250e-3, -5.2893263173827950e-3,
    -5.3969785783205650e-3, -5.5040447592375600e-3,
    -5.6104651646665850e-3, -5.7161799140263450e-3,
    -5.8211294443412250e-3, -5.9252545216101900e-3,
    -6.0284959882028950e-3, -6.1307947602360250e-3,
    -6.2320920224565700e-3, -6.3323292492425800e-3,
    -6.4314484717364400e-3, -6.5293922762311450e-3,
    -6.6261035015351700e-3, -6.7215252466922100e-3,
    -6.8156013434509950e-3, -6.9082763630156450e-3,
    -6.9994952920827200e-3, -7.0892035490577050e-3,
    -7.1773476647806300e-3, -7.2638752839566550e-3,
    -7.3487345159041400e-3, -7.4318739341000500e-3,
    -7.5132432154309250e-3, -7.5927931627681400e-3,
    -7.6704755623721500e-3, -7.7462431738117650e-3,
    -7.8200496507836000e-3, -7.8918495587816850e-3,
    -7.9615988430090790e-3, -8.0292548409052500e-3,
    -8.0947760898363200e-3, -8.1581223027535600e-3,
    -8.2192540249893810e-3, -8.2781326752680240e-3,
    -8.3347217911436610e-3, -8.3889870206274740e-3,
    -8.4408946925453690e-3, -8.4904118087845760e-3,
    -8.5375073033623590e-3, -8.5821520672029750e-3,
    -8.6243182341614300e-3, -8.6639791627160300e-3,
    -8.7011097471794110e-3, -8.7356864659172800e-3,
    -8.7676881309490440e-3, -8.7970958152554810e-3,
    -8.8238905039462850e-3, -8.8480531477483150e-3,
    -8.8695681885783190e-3, -8.8884235927356400e-3,
    -8.9046080555685600e-3, -8.9181109318560050e-3,
    -8.9289234995380550e-3, -8.9370390010903460e-3,
    -8.9424522900292110e-3, -8.9451598223763610e-3,
    /* Bank 4 */
     8.9451598223763610e-3,  8.9424522900292110e-3,
     8.9370390010903460e-3,  8.9289234995380550e-3,
     8.9181109318560050e-3,  8.9046080555685600e-3,
     8.8884235927356400e-3,  8.8695681885783190e-3,
     8.8480531477483150e-3,  8.8238905039462850e-3,
     8.7970958152554810e-3,  8.7676881309490440e-3,
     8.7356864659172800e-3,  8.7011097471794110e-3,
     8.6639791627160300e-3,  8.6243182341614300e-3,
     8.5821520672029750e-3,  8.5375073033623590e-3,
     8.4904118087845760e-3,  8.4408946925453690e-3,
     8.3889870206274740e-3,  8.3347217911436610e-3,
     8.2781326752680240e-3,  8.2192540249893810e-3,
     8.1581223027535600e-3,  8.0947760898363200e-3,
     8.0292548409052500e-3,  7.9615988430090790e-3,
     7.8918495587816850e-3,  7.8200496507836000e-3,
     7.7462431738117650e-3,  7.6704755623721500e-3,
     7.5927931627681400e-3,  7.5132432154309250e-3,
     7.4318739341000500e-3,  7.3487345159041400e-3,
     7.2638752839566550e-3,  7.1773476647806300e-3,
     7.0892035490577050e-3,  6.9994952920827200e-3,
     6.9082763630156450e-3,  6.8156013434509950e-3,
     6.7215252466922100e-3,  6.6261035015351700e-3,
     6.5293922762311450e-3,  6.4314484717364400e-3,
     6.3323292492425800e-3,  6.2320920224565700e-3,
     6.1307947602360250e-3,  6.0284959882028950e-3,
     5.9252545216101900e-3,  5.8211294443412250e-3,
     5.7161799140263450e-3,  5.6104651646665850e-3,
     5.5040447592375600e-3,  5.3969785783205650e-3,
     5.2893263173827950e-3,  5.1811474713480250e-3,
     5.0725014978718950e-3,  4.9634478250060780e-3,
     4.8540458565358850e-3,  4.7443549566589130e-3,
     4.6344341075783170e-3,  4.5243418977567100e-3,
     4.4141366059086620e-3,  4.3038762050876900e-3,
     4.1936183686073460e-3,  4.0834204508378470e-3,
     3.9733390588103910e-3,  3.8634300521859240e-3,
     3.7537489709635930e-3,  3.6443510330122110e-3,
     3.5352906520539890e-3,  3.4266214220855450e-3,
     3.3183962566563890e-3,  3.2106673881690480e-3,
     3.1034862132032660e-3,  2.9969032890820140e-3,
     2.8909684129992950e-3,  2.7857306109716460e-3,
     2.6812378156433060e-3,  2.5775368629845300e-3,
     2.4746737418493600e-3,  2.3726935964105480e-3,
     2.2716405301701940e-3,  2.1715575922482770e-3,
     2.0724866717198650e-3,  1.9744684954557180e-3,
     1.8775426862784720e-3,  1.7817477652804130e-3,
     1.6871211446584360e-3,  1.5936991188210680e-3,
     1.5015166759080980e-3,  1.4106074964587530e-3,
     1.3210041125919850e-3,  1.2327379076720450e-3,
     1.1458389497708370e-3,  1.0603359870796300e-3,
     9.7625651350435400e-4,  8.9362676824381250e-4,
     8.1247166091487090e-4,  7.3281477145293300e-4,
     6.5467842279709950e-4,  5.7808367853976800e-4,
     5.0305021853266900e-4,  4.2959633884346400e-4,
     3.5773907519488100e-4,  2.8749420455512610e-4,
     2.1887615668780720e-4,  1.5189801251884410e-4,
     8.6571556658577800e-5,  2.2907279717153710e-5,
    -3.9085623279149610e-5, -9.9399252442146460e-5,
    -1.5802699163104190e-4, -2.1496350652513040e-4,
    -2.7020471361387040e-4, -3.2374777823389440e-4,
    -3.7559110239316400e-4, -4.2573432329476060e-4,
    -4.7417829303612970e-4, -5.2092507454409450e-4,
    -5.6597787244591100e-4, -6.0934103180698750e-4,
    -6.5102007948960100e-4, -6.9102172032639350e-4,
    -7.2935371157849700e-4, -7.6602485902806300e-4,
    /* Bank 5 */
     8.0104505655578600e-4,  8.3442528375666700e-4,
     8.6617751390929650e-4,  8.9631470926558100e-4,
     9.2485080945546600e-4,  9.5180072722007060e-4,
     9.7718026613602800e-4,  1.0010061160103870e-3,
     1.0232958338441730e-3,  1.0440678410743190e-3,
     1.0633413819325920e-3,  1.0811365181610850e-3,
     1.0974740543942100e-3,  1.1123755320313420e-3,
     1.1258631690530700e-3,  1.1379598570807350e-3,
     1.1486891568700730e-3,  1.1580752925519680e-3,
     1.1661430294353880e-3,  1.1729176690707690e-3,
     1.1784250628415020e-3,  1.1826916065321440e-3,
     1.1857441072376680e-3,  1.1876097809178440e-3,
     1.1883163316843900e-3,  1.1878919433902330e-3,
     1.1863650153060240e-3,  1.1837641581852870e-3,
     1.1801183720212010e-3,  1.1754570428998680e-3,
     1.1698096960154200e-3,  1.1632059885511800e-3,
     1.1556758000499310e-3,  1.1472492294891640e-3,
     1.1379564405661050e-3,  1.1278276550234160e-3,
     1.1168931605293980e-3,  1.1051833075734240e-3,
     1.0927284332850520e-3,  1.0795588568782010e-3,
     1.0657048556038330e-3,  1.0511966606168980e-3,
     1.0360643900964910e-3,  1.0203380430448350e-3,
     1.0040474297445790e-3,  9.8722217067251400e-4,
     9.6989174214394400e-4,  9.5208547141354700e-4,
     9.3383238324995840e-4,  9.1516119514986000e-4,
     8.9610036547340950e-4,  8.7667809201044900e-4,
     8.5692223230171550e-4,  8.3686029869896850e-4,
     8.1651942936660300e-4,  7.9592638676398950e-4,
     7.7510755325530950e-4,  7.5408892753413600e-4,
     7.3289605035434600e-4,  7.1155400216196750e-4,
     6.9008742526480550e-4,  6.6852052145861900e-4,
     6.4687697763330650e-4,  6.2517996394886800e-4,
     6.0345216810040500e-4,  5.8171579317559550e-4,
     5.5999247625413450e-4,  5.3830328700499900e-4,
     5.1666877822680600e-4,  4.9510898492628230e-4,
     4.7364335350426610e-4,  4.5229073958127750e-4,
     4.3106943097409840e-4,  4.0999714822829530e-4,
     3.8909103335715320e-4,  3.6836764760397900e-4,
     3.4784293347385130e-4,  3.2753221542219500e-4,
     3.0745025297166030e-4,  2.8761124019708440e-4,
     2.6802874127778880e-4,  2.4871569004884240e-4,
     2.2968444457655940e-4,  2.1094678833176480e-4,
     1.9251390142613000e-4,  1.7439635977956720e-4,
     1.5660414556657680e-4,  1.3914664880465400e-4,
     1.2203269185287530e-4,  1.0527052946041040e-4,
     8.8867825335203200e-5,  7.2831653374563650e-5,
     5.7168548169515050e-5,  4.1884506316022560e-5,
     2.6984964757674160e-5,  1.2474801400564880e-5,
    -1.6416297019304470e-6, -1.5360496520469550e-5,
    -2.8678484003389880e-5, -4.1592792755171040e-5,
    -5.4101115008963200e-5, -6.6201632959158800e-5,
    -7.7893006105444910e-5, -8.9174369623037550e-5,
    -1.0004531116378850e-4, -1.1050586906665980e-4,
    -1.2055651618705740e-4, -1.3019815804235450e-4,
    -1.3943210818407590e-4, -1.4826008597798460e-4,
    -1.5668419239382670e-4, -1.6470690840351980e-4,
    -1.7233108395155490e-4, -1.7955993578371750e-4,
    -1.8639701070854740e-4, -1.9284618337957900e-4,
    -1.9891164424316180e-4, -2.0459789739171640e-4,
    -2.0990972538755550e-4, -2.1485218733324290e-4,
    -2.1943061159567880e-4, -2.2365059385031200e-4,
    -2.2751796132972940e-4, -2.3103877007019660e-4,
    -2.3421928008696620e-4, -2.3706595356421380e-4,
    -2.3958543985834340e-4, -2.4178457457919800e-4,
    /* Bank 6 */
     2.4367037437341020e-4,  2.4525003217524440e-4,
     2.4653081794176570e-4,  2.4752008219951220e-4,
     2.4822543333608940e-4,  2.4865471770248810e-4,
     2.4881540457247040e-4,  2.4871461830381660e-4,
     2.4836046092637280e-4,  2.4776199025177680e-4,
     2.4692741569938220e-4,  2.4586407616017000e-4,
     2.4457975717687980e-4,  2.4308272452941820e-4,
     2.4138114593372380e-4,  2.3948307023621970e-4,
     2.3739654799291240e-4,  2.3512963563507800e-4,
     2.3269036653525150e-4,  2.3008674810081220e-4,
     2.2732672676686290e-4,  2.2441818635854840e-4,
     2.2136894706917190e-4,  2.1818676491655660e-4,
     2.1487932080488700e-4,  2.1145421910491280e-4,
     2.0791896735659270e-4,  2.0428097498957360e-4,
     2.0054754547108610e-4,  1.9672587556804270e-4,
     1.9282304696531290e-4,  1.8884602526638050e-4,
     1.8480164638953740e-4,  1.8069661584612290e-4,
     1.7653750646560020e-4,  1.7233075773584680e-4,
     1.6808266356446820e-4,  1.6379937154642740e-4,
     1.5948687909058970e-4,  1.5515103314094370e-4,
     1.5079752791692290e-4,  1.4643190426323540e-4,
     1.4205953760578120e-4,  1.3768563787408340e-4,
     1.3331525983885840e-4,  1.2895330270761850e-4,
     1.2460449089034750e-4,  1.2027337376184880e-4,
     1.1596433966772030e-4,  1.1168161614264270e-4,
     1.0742926070674010e-4,  1.0321116057304920e-4,
     9.9031035415475500e-5,  9.4892437636254000e-5,
     9.0798755482139240e-5,  8.6753213062403400e-5,
     8.2758867628623800e-5,  7.8818609806320000e-5,
     7.4935171411178040e-5,  7.1111125753722860e-5,
     6.7348886525882700e-5,  6.3650708053788550e-5,
     6.0018691972791050e-5,  5.6454787620185350e-5,
     5.2960794000140600e-5,  4.9538360176954340e-5,
     4.6188991961724560e-5,  4.2914052387856640e-5,
     3.9714765485539260e-5,  3.6592216716540970e-5,
     3.3547358765259040e-5,  3.0581012081916960e-5,
     2.7693871041862060e-5,  2.4886504491416160e-5,
     2.2159361597292710e-5,  1.9512772389215240e-5,
     1.6946953848107710e-5,  1.4462010500550970e-5,
     1.2057941408635810e-5,  9.7346407524871100e-6,
     7.4919036559427900e-6,  5.3294268968052800e-6,
     3.2468187066945930e-6,  1.2435990827839660e-6,
    -6.8080315386959900e-7, -2.5270494199802070e-6,
    -4.2958677847708840e-6, -5.9880523825897000e-6,
    -7.6044740757265500e-6, -9.1460801506922300e-6,
    -1.0613876973541650e-5, -1.2008929083235120e-5,
    -1.3332356590263080e-5, -1.4585334740507920e-5,
    -1.5769086921577900e-5, -1.6884884022311310e-5,
    -1.7934037335534700e-5, -1.8917898053528080e-5,
    -1.9837853265502040e-5, -2.0695325418576370e-5,
    -2.1491764461832140e-5, -2.2228647280994560e-5,
    -2.2907473117403740e-5, -2.3529763128539150e-5,
    -2.4097055300584120e-5, -2.4610903957661350e-5,
    -2.5072874052447260e-5, -2.5484540698002960e-5,
    -2.5847484577439320e-5, -2.6163291550639320e-5,
    -2.6433548592482960e-5, -2.6659843389908530e-5,
    -2.6843759539035120e-5, -2.6986876191054700e-5,
    -2.7090765064735940e-5, -2.7156990105794550e-5,
    -2.7187102980536760e-5, -2.7182642746157720e-5,
    -2.7145133103663890e-5, -2.7076082153012070e-5,
    -2.6976979753245170e-5, -2.6849297239997120e-5,
    -2.6694483850464800e-5, -2.6513966535124460e-5,
    -2.6309149390520640e-5, -2.6081413415833200e-5,
    -2.5832111724562040e-5, -2.5562569399199020e-5,
    /* Bank 7 */
     2.5274085082324960e-5,  2.4967930863542620e-5,
     2.4645348200903720e-5,  2.4307547722489910e-5,
     2.3955708939763280e-5,  2.3590980216497750e-5,
     2.3214478371918170e-5,  2.2827288605503470e-5,
     2.2430463239517170e-5,  2.2025021652894630e-5,
     2.1611950084231910e-5,  2.1192201630574280e-5,
     2.0766696417773940e-5,  2.0336321557464510e-5,
     1.9901930030141080e-5,  1.9464340744880170e-5,
     1.9024340970099240e-5,  1.8582686308483550e-5,
     1.8140097714618200e-5,  1.7697261563607500e-5,
     1.7254834143068570e-5,  1.6813441716539560e-5,
     1.6373677426466900e-5,  1.5936101341640500e-5,
     1.5501244597782600e-5,  1.5069609527679710e-5,
     1.4641668383496450e-5,  1.4217863365843780e-5,
     1.3798608540965960e-5,  1.3384289978242060e-5,
     1.2975266858030310e-5,  1.2571871574469220e-5,
     1.2174410889256530e-5,  1.1783166036693560e-5,
     1.1398393880881060e-5,  1.1020327046334650e-5,
     1.0649175634328590e-5,  1.0285127334608450e-5,
     9.9283481667264540e-6,  9.5789825988253100e-6,
     9.2371554195468400e-6,  8.9029718900122000e-6,
     8.5765192154768100e-6,  8.2578666429227760e-6,
     7.9470661364365490e-6,  7.6441525098592000e-6,
     7.3491456697144900e-6,  7.0620507545154100e-6,
     6.7828589565804950e-6,  6.5115476278604000e-6,
     6.2480817863072800e-6,  5.9924142424804800e-6,
     5.7444868785028750e-6,  5.5042307535458100e-6,
     5.2715671455405850e-6,  5.0464076705003400e-6,
     4.8286558658894680e-6,  4.6182073040543900e-6,
     4.4149505043312320e-6,  4.2187670209852250e-6,
     4.0295324657699800e-6,  3.8471166218015830e-6,
     3.6713849262294680e-6,  3.5021985520037540e-6,
     3.3394147240855820e-6,  3.1828867865974640e-6,
     3.0324653639203880e-6,  2.8879984579501380e-6,
     2.7493324266391140e-6,  2.6163120571644820e-6,
     2.4887811970856540e-6,  2.3665828120950560e-6,
     2.2495596253776460e-6,  2.1375541686662320e-6,
     2.0304092660990760e-6,  1.9279680902322050e-6,
     1.8300749104526120e-6,  1.7365751553971340e-6,
     1.6473160377545360e-6,  1.5621465913980060e-6,
     1.4809178634979940e-6,  1.4034829433780260e-6,
     1.3296974052300520e-6,  1.2594193438633260e-6,
     1.1925097184646100e-6,  1.1288323637155430e-6,
     1.0682538906146330e-6,  1.0106437128213760e-6,
     9.5587472540498280e-7,  9.0382333224895500e-7,
     8.5436937020293650e-7,  8.0739610368280600e-7,
     7.6279018172077200e-7,  7.2044165259013450e-7,
     6.8024432849991400e-7,  6.4209580087853050e-7,
     6.0589741192019800e-7,  5.7155423388793000e-7,
     5.3897464223335550e-7,  5.0807031294631050e-7,
     4.7875659112606760e-7,  4.5095249563698380e-7,
     4.2458045296159330e-7,  3.9956628121669190e-7,
     3.7583910465412180e-7,  3.5333134588843600e-7,
     3.3197864040473180e-7,  3.1171981968268900e-7,
     2.9249662541757600e-7,  2.7425370308549580e-7,
     2.5693874615777660e-7,  2.4050245877923400e-7,
     2.2489759047553620e-7,  2.1007899706006800e-7,
     1.9600594587757850e-7,  1.8264184311018150e-7,
     1.6994592926571220e-7,  1.5787424998123710e-7,
     1.4640931821191640e-7,  1.3555636137971380e-7,
     1.2523109645615350e-7,  1.1527363411865760e-7,
     1.0586659510721950e-7,  9.7483190494874640e-8,
     8.9326871281374790e-8,  7.9525034321375090e-8,
     7.0950903150874990e-8,  7.1279389866041690e-8,
};

DECLARE_ALIGNED(16, const int32_t, ff_dca_fir_32bands_perfect_fixed)[512] = {
           0,        0,       -3,      -10,
         -35,     -105,     -218,     -141,
        -170,     -216,     -239,     -254,
        -257,     -251,     -235,     -212,
        -267,     -317,     -362,     -400,
        -425,     -434,     -427,     -373,
        -339,     -593,     -321,     -120,
         -39,      -16,        0,        1,
           1,        1,       -3,       -1,
          -6,      -38,      -93,     -496,
        -723,     -970,    -1235,    -1501,
       -1753,    -1978,    -2163,    -2295,
       -2891,    -2915,    -2860,    -2726,
       -2517,    -2243,    -1915,    -1590,
       -1192,     -252,     -117,      -22,
          -6,      -13,       12,       14,
          32,       25,      469,      942,
        1403,     1421,     1239,     2838,
        3539,     4259,     5002,     5716,
        6365,     6908,     7311,     7545,
       11680,    12355,    12785,    12951,
       12841,    12453,    11803,    10864,
        9762,     7099,     6725,     5954,
        4284,     2584,      215,      379,
         557,      701,      -29,     -687,
       -1578,    -2749,    -4076,    -7933,
      -10049,   -12133,   -14039,   -15752,
      -17213,   -18400,   -19291,   -19878,
       -1444,    -3005,    -4523,    -5927,
       -7143,    -8093,    -8713,    -8939,
       -8700,    -9481,    -7515,    -5279,
       -2725,       61,     5763,     6113,
        7571,     6735,    17126,    20165,
       23328,    26775,    30310,    32639,
       35464,    38064,    40423,    42512,
       44261,    45632,    46578,    46974,
      -45572,   -45008,   -43753,   -41661,
      -38655,   -34660,   -29587,   -23375,
      -15998,    -7631,     2472,    13757,
       26188,    39942,    49789,    67293,
       84699,   101701,   127325,   148404,
      170391,   193280,   217044,   241451,
      266537,   292144,   318161,   344417,
      370786,   397082,   423133,   448757,
      475085,   499136,   522007,   543516,
      563424,   581467,   597422,   611005,
      621975,   630083,   634996,   636457,
      634311,   628147,   619871,   604524,
      585077,   561301,   529204,   494129,
      453552,   407189,   354920,   296502,
      231916,   161012,    83700,      -86,
      -90377,  -187193,  -290528,  -400329,
      516487,   639054,   767835,   902710,
     1043512,  1190048,  1342100,  1499418,
     1661729,  1828700,  2000071,  2175433,
     2354437,  2536630,  2721120,  2908704,
     3098059,  3288764,  3480801,  3672922,
     3864970,  4056432,  4246767,  4435454,
     4621921,  4805642,  4986073,  5162677,
     5334921,  5502279,  5664239,  5820300,
     5969913,  6112723,  6248225,  6375985,
     6495593,  6606663,  6708832,  6801769,
     6885168,  6958762,  7022294,  7075566,
     7118382,  7150633,  7172314,  7183082,
     7183082,  7172314,  7150633,  7118382,
     7075566,  7022294,  6958762,  6885168,
     6801769,  6708832,  6606663,  6495593,
     6375985,  6248225,  6112723,  5969913,
    -5820300, -5664239, -5502279, -5334921,
    -5162677, -4986073, -4805642, -4621921,
    -4435454, -4246767, -4056432, -3864970,
    -3672922, -3480801, -3288764, -3098059,
    -2908704, -2721120, -2536630, -2354437,
    -2175433, -2000071, -1828700, -1661729,
    -1499418, -1342100, -1190048, -1043512,
     -902710,  -767835,  -639054,  -516487,
     -400329,  -290528,  -187193,   -90377,
         -86,    83700,   161012,   231916,
      296502,   354920,   407189,   453552,
      494129,   529204,   561301,   585077,
      604524,   619871,   628147,   634311,
      636457,   634996,   630083,   621975,
      611005,   597422,   581467,   563424,
      543516,   522007,   499136,   475085,
     -448757,  -423133,  -397082,  -370786,
     -344417,  -318161,  -292144,  -266537,
     -241451,  -217044,  -193280,  -170391,
     -148404,  -127325,  -101701,   -84699,
      -67293,   -49789,   -39942,   -26188,
      -13757,    -2472,     7631,    15998,
       23375,    29587,    34660,    38655,
       41661,    43753,    45008,    45572,
       46974,    46578,    45632,    44261,
       42512,    40423,    38064,    35464,
       32639,    30310,    26775,    23328,
       20165,    17126,     6735,     7571,
        6113,     5763,       61,    -2725,
       -5279,    -7515,    -9481,    -8700,
       -8939,    -8713,    -8093,    -7143,
       -5927,    -4523,    -3005,    -1444,
       19878,    19291,    18400,    17213,
       15752,    14039,    12133,    10049,
        7933,     4076,     2749,     1578,
         687,       29,     -701,     -557,
        -379,     -215,    -2584,    -4284,
       -5954,    -6725,    -7099,    -9762,
      -10864,   -11803,   -12453,   -12841,
      -12951,   -12785,   -12355,   -11680,
        7545,     7311,     6908,     6365,
        5716,     5002,     4259,     3539,
        2838,     1239,     1421,     1403,
         942,      469,       25,       32,
          14,       12,      -13,       -6,
         -22,     -117,     -252,    -1192,
       -1590,    -1915,    -2243,    -2517,
       -2726,    -2860,    -2915,    -2891,
        2295,     2163,     1978,     1753,
        1501,     1235,      970,      723,
         496,       93,       38,        6,
           1,        3,       -1,       -1,
          -1,        0,       16,       39,
         120,      321,      593,      339,
         373,      427,      434,      425,
         400,      362,      317,      267,
        -212,     -235,     -251,     -257,
        -254,     -239,     -216,     -170,
        -141,     -218,     -105,      -35,
         -10,       -3,        0,        0
};

DECLARE_ALIGNED(16, const int32_t, ff_dca_fir_32bands_nonperfect_fixed)[512] = {
         -53,      -64,      -77,      -91,
        -107,     -124,     -144,     -165,
        -189,     -215,     -244,     -277,
        -313,     -353,     -397,     -447,
         502,      563,      631,      706,
         789,      881,      983,     1095,
        1218,     1354,     1502,     1665,
        1843,     2036,     2247,     2475,
        2723,     2990,     3277,     3586,
        3916,     4270,     4646,     5046,
        5470,     5918,     6390,     6886,
        7405,     7947,     8510,     9094,
        9698,    10319,    10955,    11605,
       12265,    12933,    13605,    14277,
       14945,    15604,    16250,    16877,
       17480,    18051,    18585,    19075,
       19513,    19891,    20202,    20436,
       20587,    20643,    20597,    20439,
       20160,    19749,    19198,    18496,
       17634,    16603,    15393,    13996,
      -12403,   -10605,    -8595,    -6366,
       -3911,    -1225,     1697,     4860,
        8265,    11916,    15812,    19953,
       24337,    28961,    33819,    38904,
       44210,    49725,    55437,    61334,
       67398,    73614,    79961,    86417,
       92960,    99563,   106198,   112837,
      119446,   125994,   132443,   138758,
      144898,   150823,   156491,   161858,
      166879,   171507,   175697,   179400,
      182566,   185149,   187097,   188363,
      188899,   188654,   187581,   185635,
      182770,   178943,   174112,   168238,
      161285,   153218,   144007,   133624,
      122046,   109254,    95232,    79969,
       63462,    45709,    26715,     6492,
       14943,    37567,    61350,    86256,
      112242,   139258,   167246,   196143,
      225877,   256368,   287532,   319275,
      351496,   384088,   416936,   449919,
      482909,   515770,   548362,   580539,
      612148,   643030,   673024,   701963,
      729674,   755985,   780717,   803690,
      824721,   843628,   860226,   874332,
      885761,   894330,   899861,   902174,
      901096,   896456,   888088,   875832,
      859535,   839050,   814237,   784966,
      751116,   712574,   669239,   621021,
      567840,   509632,   446341,   377927,
      304365,   225641,   141757,    52732,
      -41403,  -140599,  -244793,  -353905,
     -467840,  -586486,  -709716,  -837385,
      969336,  1105393,  1245366,  1389049,
     1536224,  1686655,  1840096,  1996285,
     2154949,  2315802,  2478547,  2642877,
     2808475,  2975015,  3142163,  3309579,
     3476914,  3643818,  3809934,  3974901,
     4138360,  4299948,  4459303,  4616064,
     4769873,  4920374,  5067219,  5210063,
     5348569,  5482406,  5611255,  5734805,
     5852757,  5964823,  6070729,  6170216,
     6263037,  6348961,  6427777,  6499286,
     6563310,  6619688,  6668279,  6708963,
     6741632,  6766206,  6782623,  6790843,
     6790843,  6782623,  6766206,  6741632,
     6708963,  6668279,  6619688,  6563310,
     6499286,  6427777,  6348961,  6263037,
     6170216,  6070729,  5964823,  5852757,
    -5734805, -5611255, -5482406, -5348569,
    -5210063, -5067219, -4920374, -4769873,
    -4616064, -4459303, -4299948, -4138360,
    -3974901, -3809934, -3643818, -3476914,
    -3309579, -3142163, -2975015, -2808475,
    -2642877, -2478547, -2315802, -2154949,
    -1996285, -1840096, -1686655, -1536224,
    -1389049, -1245366, -1105393,  -969336,
     -837385,  -709716,  -586486,  -467840,
     -353905,  -244793,  -140599,   -41403,
       52732,   141757,   225641,   304365,
      377927,   446341,   509632,   567840,
      621021,   669239,   712574,   751116,
      784966,   814237,   839050,   859535,
      875832,   888088,   896456,   901096,
      902174,   899861,   894330,   885761,
     -874332,  -860226,  -843628,  -824721,
     -803690,  -780717,  -755985,  -729674,
     -701963,  -673024,  -643030,  -612148,
     -580539,  -548362,  -515770,  -482909,
     -449919,  -416936,  -384088,  -351496,
     -319275,  -287532,  -256368,  -225877,
     -196143,  -167246,  -139258,  -112242,
      -86256,   -61350,   -37567,   -14943,
        6492,    26715,    45709,    63462,
       79969,    95232,   109254,   122046,
      133624,   144007,   153218,   161285,
      168238,   174112,   178943,   182770,
      185635,   187581,   188654,   188899,
      188363,   187097,   185149,   182566,
      179400,   175697,   171507,   166879,
      161858,   156491,   150823,   144898,
     -138758,  -132443,  -125994,  -119446,
     -112837,  -106198,   -99563,   -92960,
      -86417,   -79961,   -73614,   -67398,
      -61334,   -55437,   -49725,   -44210,
      -38904,   -33819,   -28961,   -24337,
      -19953,   -15812,   -11916,    -8265,
       -4860,    -1697,     1225,     3911,
        6366,     8595,    10605,    12403,
       13996,    15393,    16603,    17634,
       18496,    19198,    19749,    20160,
       20439,    20597,    20643,    20587,
       20436,    20202,    19891,    19513,
       19075,    18585,    18051,    17480,
       16877,    16250,    15604,    14945,
       14277,    13605,    12933,    12265,
       11605,    10955,    10319,     9698,
       -9094,    -8510,    -7947,    -7405,
       -6886,    -6390,    -5918,    -5470,
       -5046,    -4646,    -4270,    -3916,
       -3586,    -3277,    -2990,    -2723,
       -2475,    -2247,    -2036,    -1843,
       -1665,    -1502,    -1354,    -1218,
       -1095,     -983,     -881,     -789,
        -706,     -631,     -563,     -502,
        -447,     -397,     -353,     -313,
        -277,     -244,     -215,     -189,
        -165,     -144,     -124,     -107,
         -91,      -77,      -64,      -53
};

DECLARE_ALIGNED(16, const int32_t, ff_dca_lfe_fir_64_fixed)[256] = {
     6103,   52170, -558064, 1592440, 6290049, 1502534, -546669, 53047,
     1930,   51089, -568920, 1683709, 6286575, 1414057, -534782, 53729,
     2228,   49794, -579194, 1776276, 6279634, 1327070, -522445, 54228,
     2552,   48275, -588839, 1870070, 6269231, 1241632, -509702, 54550,
     2904,   46523, -597808, 1965017, 6255380, 1157798, -496595, 54708,
     3287,   44529, -606054, 2061044, 6238099, 1075621, -483164, 54710,
     3704,   42282, -613529, 2158071, 6217408,  995149, -469451, 54566,
     4152,   39774, -620186, 2256019, 6193332,  916430, -455494, 54285,
     4631,   36995, -625976, 2354805, 6165900,  839507, -441330, 53876,
     5139,   33937, -630850, 2454343, 6135146,  764419, -426998, 53348,
     5682,   30591, -634759, 2554547, 6101107,  691203, -412531, 52711,
     6264,   26948, -637655, 2655326, 6063824,  619894, -397966, 51972,
     6886,   23001, -639488, 2756591, 6023343,  550521, -383335, 51140,
     7531,   18741, -640210, 2858248, 5979711,  483113, -368671, 50224,
     8230,   14162, -639772, 2960201, 5932981,  417692, -354003, 49231,
     8959,    9257, -638125, 3062355, 5883210,  354281, -339362, 48168,
     9727,    4018, -635222, 3164612, 5830457,  292897, -324777, 47044,
    10535,   -1558, -631014, 3266872, 5774785,  233555, -310273, 45866,
    11381,   -7480, -625455, 3369035, 5716260,  176267, -295877, 44640,
    12267,  -13750, -618499, 3471000, 5654952,  121042, -281613, 43373,
    13190,  -20372, -610098, 3572664, 5590933,   67886, -267505, 42072,
    14152,  -27352, -600209, 3673924, 5524280,   16800, -253574, 40743,
    15153,  -34691, -588788, 3774676, 5455069,  -32214, -239840, 39391,
    16192,  -42390, -575791, 3874816, 5383383,  -79159, -226323, 38022,
    17267,  -50453, -561178, 3974239, 5309305, -124041, -213041, 36642,
    18377,  -58879, -544906, 4072841, 5232922, -166869, -200010, 35256,
    19525,  -67667, -526937, 4170517, 5154321, -207653, -187246, 33866,
    20704,  -76817, -507233, 4267162, 5073593, -246406, -174764, 32480,
    21915,  -86327, -485757, 4362672, 4990831, -283146, -162575, 31101,
    23157,  -96193, -462476, 4456942, 4906129, -317890, -150692, 29732,
    24426, -106412, -437356, 4549871, 4819584, -350658, -139125, 28376,
    25721, -116977, -410365, 4641355, 4731293, -381475, -127884, 27038
};

DECLARE_ALIGNED(16, const int32_t, ff_dca_fir_64bands_fixed)[1024] = {
         -38,      -38,      -43,      -48,
         -52,      -57,      -62,      -67,
         -73,      -79,      -85,      -91,
         -98,     -105,     -113,     -121,
        -129,     -138,     -147,     -157,
        -167,     -178,     -190,     -202,
        -215,     -228,     -242,     -257,
        -273,     -289,     -307,     -325,
         345,      365,      387,      410,
         433,      459,      485,      513,
         543,      574,      606,      640,
         676,      714,      753,      795,
         839,      884,      932,      983,
        1035,     1090,     1148,     1208,
        1271,     1336,     1405,     1476,
        1550,     1628,     1709,     1793,
        1880,     1971,     2065,     2163,
        2265,     2370,     2479,     2592,
        2709,     2830,     2955,     3084,
        3217,     3354,     3496,     3642,
        3791,     3946,     4104,     4267,
        4433,     4604,     4780,     4959,
        5143,     5330,     5522,     5717,
        5916,     6119,     6326,     6536,
        6749,     6966,     7186,     7408,
        7633,     7861,     8090,     8322,
        8556,     8791,     9027,     9264,
        9501,     9739,     9977,    10214,
       10450,    10685,    10918,    11149,
       11377,    11603,    11825,    12042,
       12255,    12463,    12665,    12861,
       13050,    13231,    13405,    13569,
       13724,    13869,    14002,    14125,
       14235,    14331,    14415,    14483,
       14536,    14573,    14594,    14596,
       14580,    14544,    14488,    14412,
       14313,    14191,    14046,    13877,
       13682,    13461,    13213,    12937,
       12632,    12298,    11934,    11538,
       11111,    10650,    10156,     9628,
       -9065,    -8466,    -7830,    -7158,
       -6447,    -5698,    -4910,    -4083,
       -3215,    -2306,    -1357,     -366,
         668,     1743,     2861,     4022,
        5226,     6474,     7764,     9098,
       10476,    11897,    13361,    14868,
       16418,    18011,    19645,    21322,
       23039,    24798,    26596,    28433,
       30309,    32222,    34172,    36158,
       38177,    40231,    42315,    44431,
       46575,    48747,    50945,    53167,
       55411,    57676,    59959,    62258,
       64571,    66897,    69231,    71573,
       73919,    76268,    78615,    80959,
       83296,    85624,    87939,    90239,
       92519,    94778,    97011,    99215,
      101386,   103521,   105616,   107668,
      109673,   111626,   113524,   115362,
      117138,   118847,   120484,   122045,
      123527,   124925,   126234,   127451,
      128571,   129591,   130504,   131308,
      131997,   132568,   133016,   133338,
      133528,   133582,   133495,   133265,
      132886,   132355,   131668,   130820,
      129807,   128626,   127274,   125746,
      124038,   122148,   120071,   117806,
      115348,   112694,   109843,   106790,
      103534,   100071,    96401,    92520,
       88426,    84119,    79597,    74857,
       69900,    64723,    59327,    53711,
       47875,    41818,    35542,    29045,
       22330,    15397,     8247,      881,
        6697,    14487,    22487,    30692,
       39101,    47711,    56517,    65516,
       74704,    84076,    93628,   103355,
      113251,   123311,   133528,   143897,
      154410,   165061,   175843,   186747,
      197766,   208892,   220116,   231429,
      242822,   254285,   265810,   277384,
      288999,   300644,   312306,   323976,
      335641,   347289,   358909,   370488,
      382013,   393471,   404848,   416133,
      427310,   438366,   449286,   460057,
      470663,   481090,   491323,   501347,
      511147,   520707,   530011,   539044,
      547790,   556233,   564357,   572146,
      579584,   586654,   593341,   599627,
      605498,   610936,   615925,   620449,
      624491,   628037,   631069,   633571,
      635529,   636925,   637745,   637972,
      637593,   636592,   634953,   632662,
      629705,   626068,   621737,   616698,
      610938,   604443,   597202,   589202,
      580431,   570877,   560530,   549378,
      537411,   524620,   510994,   496525,
      481205,   465026,   447979,   430058,
      411256,   391569,   370989,   349514,
      327137,   303857,   279670,   254573,
      228564,   201644,   173811,   145065,
      115408,    84840,    53365,    20984,
      -12298,   -46478,   -81550,  -117508,
     -154347,  -192060,  -230638,  -270073,
     -310356,  -351478,  -393427,  -436192,
     -479762,  -524124,  -569264,  -615168,
      661821,   709209,   757314,   806121,
      855611,   905766,   956569,  1007998,
     1060035,  1112658,  1165846,  1219578,
     1273830,  1328580,  1383805,  1439479,
     1495578,  1552077,  1608950,  1666171,
     1723714,  1781550,  1839653,  1897995,
     1956546,  2015279,  2074163,  2133170,
     2192270,  2251432,  2310626,  2369822,
     2428988,  2488093,  2547106,  2605996,
     2664731,  2723279,  2781607,  2839685,
     2897481,  2954962,  3012096,  3068851,
     3125195,  3181097,  3236524,  3291445,
     3345829,  3399643,  3452858,  3505441,
     3557362,  3608591,  3659098,  3708853,
     3757825,  3805987,  3853309,  3899763,
     3945322,  3989957,  4033642,  4076350,
     4118055,  4158733,  4198357,  4236904,
     4274351,  4310673,  4345850,  4379859,
     4412678,  4444289,  4474670,  4503803,
     4531671,  4558255,  4583539,  4607508,
     4630146,  4651438,  4671373,  4689936,
     4707117,  4722905,  4737290,  4750262,
     4761813,  4771936,  4780625,  4787874,
     4793679,  4798036,  4800943,  4802396,
     4802396,  4800943,  4798036,  4793679,
     4787874,  4780625,  4771936,  4761813,
     4750262,  4737290,  4722905,  4707117,
     4689936,  4671373,  4651438,  4630146,
     4607508,  4583539,  4558255,  4531671,
     4503803,  4474670,  4444289,  4412678,
     4379859,  4345850,  4310673,  4274351,
     4236904,  4198357,  4158733,  4118055,
    -4076350, -4033642, -3989957, -3945322,
    -3899763, -3853309, -3805987, -3757825,
    -3708853, -3659098, -3608591, -3557362,
    -3505441, -3452858, -3399643, -3345829,
    -3291445, -3236524, -3181097, -3125195,
    -3068851, -3012096, -2954962, -2897481,
    -2839685, -2781607, -2723279, -2664731,
    -2605996, -2547106, -2488093, -2428988,
    -2369822, -2310626, -2251432, -2192270,
    -2133170, -2074163, -2015279, -1956546,
    -1897995, -1839653, -1781550, -1723714,
    -1666171, -1608950, -1552077, -1495578,
    -1439479, -1383805, -1328580, -1273830,
    -1219578, -1165846, -1112658, -1060035,
    -1007998,  -956569,  -905766,  -855611,
     -806121,  -757314,  -709209,  -661821,
     -615168,  -569264,  -524124,  -479762,
     -436192,  -393427,  -351478,  -310356,
     -270073,  -230638,  -192060,  -154347,
     -117508,   -81550,   -46478,   -12298,
       20984,    53365,    84840,   115408,
      145065,   173811,   201644,   228564,
      254573,   279670,   303857,   327137,
      349514,   370989,   391569,   411256,
      430058,   447979,   465026,   481205,
      496525,   510994,   524620,   537411,
      549378,   560530,   570877,   580431,
      589202,   597202,   604443,   610938,
      616698,   621737,   626068,   629705,
      632662,   634953,   636592,   637593,
      637972,   637745,   636925,   635529,
      633571,   631069,   628037,   624491,
     -620449,  -615925,  -610936,  -605498,
     -599627,  -593341,  -586654,  -579584,
     -572146,  -564357,  -556233,  -547790,
     -539044,  -530011,  -520707,  -511147,
     -501347,  -491323,  -481090,  -470663,
     -460057,  -449286,  -438366,  -427310,
     -416133,  -404848,  -393471,  -382013,
     -370488,  -358909,  -347289,  -335641,
     -323976,  -312306,  -300644,  -288999,
     -277384,  -265810,  -254285,  -242822,
     -231429,  -220116,  -208892,  -197766,
     -186747,  -175843,  -165061,  -154410,
     -143897,  -133528,  -123311,  -113251,
     -103355,   -93628,   -84076,   -74704,
      -65516,   -56517,   -47711,   -39101,
      -30692,   -22487,   -14487,    -6697,
         881,     8247,    15397,    22330,
       29045,    35542,    41818,    47875,
       53711,    59327,    64723,    69900,
       74857,    79597,    84119,    88426,
       92520,    96401,   100071,   103534,
      106790,   109843,   112694,   115348,
      117806,   120071,   122148,   124038,
      125746,   127274,   128626,   129807,
      130820,   131668,   132355,   132886,
      133265,   133495,   133582,   133528,
      133338,   133016,   132568,   131997,
      131308,   130504,   129591,   128571,
      127451,   126234,   124925,   123527,
      122045,   120484,   118847,   117138,
      115362,   113524,   111626,   109673,
      107668,   105616,   103521,   101386,
      -99215,   -97011,   -94778,   -92519,
      -90239,   -87939,   -85624,   -83296,
      -80959,   -78615,   -76268,   -73919,
      -71573,   -69231,   -66897,   -64571,
      -62258,   -59959,   -57676,   -55411,
      -53167,   -50945,   -48747,   -46575,
      -44431,   -42315,   -40231,   -38177,
      -36158,   -34172,   -32222,   -30309,
      -28433,   -26596,   -24798,   -23039,
      -21322,   -19645,   -18011,   -16418,
      -14868,   -13361,   -11897,   -10476,
       -9098,    -7764,    -6474,    -5226,
       -4022,    -2861,    -1743,     -668,
         366,     1357,     2306,     3215,
        4083,     4910,     5698,     6447,
        7158,     7830,     8466,     9065,
        9628,    10156,    10650,    11111,
       11538,    11934,    12298,    12632,
       12937,    13213,    13461,    13682,
       13877,    14046,    14191,    14313,
       14412,    14488,    14544,    14580,
       14596,    14594,    14573,    14536,
       14483,    14415,    14331,    14235,
       14125,    14002,    13869,    13724,
       13569,    13405,    13231,    13050,
       12861,    12665,    12463,    12255,
       12042,    11825,    11603,    11377,
       11149,    10918,    10685,    10450,
       10214,     9977,     9739,     9501,
        9264,     9027,     8791,     8556,
        8322,     8090,     7861,     7633,
        7408,     7186,     6966,     6749,
       -6536,    -6326,    -6119,    -5916,
       -5717,    -5522,    -5330,    -5143,
       -4959,    -4780,    -4604,    -4433,
       -4267,    -4104,    -3946,    -3791,
       -3642,    -3496,    -3354,    -3217,
       -3084,    -2955,    -2830,    -2709,
       -2592,    -2479,    -2370,    -2265,
       -2163,    -2065,    -1971,    -1880,
       -1793,    -1709,    -1628,    -1550,
       -1476,    -1405,    -1336,    -1271,
       -1208,    -1148,    -1090,    -1035,
        -983,     -932,     -884,     -839,
        -795,     -753,     -714,     -676,
        -640,     -606,     -574,     -543,
        -513,     -485,     -459,     -433,
        -410,     -387,     -365,     -345,
        -325,     -307,     -289,     -273,
        -257,     -242,     -228,     -215,
        -202,     -190,     -178,     -167,
        -157,     -147,     -138,     -129,
        -121,     -113,     -105,      -98,
         -91,      -85,      -79,      -73,
         -67,      -62,      -57,      -52,
         -48,      -43,      -38,      -38
};

/*
 * D.11 Look-up Table for Downmix Scale Factors
 *
 * Note that the range of the entries in DmixTable[] is between -60 dB and 0 dB
 * with addition of -inf (|DMixCoeff| = 0), which is coded with a DmixCode = 0.
 * Furthermore, the range [-60 to 0] is subdivided into 3 regions, each with a
 * different grid resolution:
 *
 * 1) [-60.000 to -30] with resolution of 0.500 dB
 * 2) [-29.750 to -15] with resolution of 0.250 dB
 * 3) [-14.875 to   0] with resolution of 0.125 dB
 */
const uint16_t ff_dca_dmixtable[FF_DCA_DMIXTABLE_SIZE] = {
        0,    33,    35,    37,    39,    41,    44,    46,
       49,    52,    55,    58,    62,    65,    69,    73,
       78,    82,    87,    92,    98,   104,   110,   116,
      123,   130,   138,   146,   155,   164,   174,   184,
      195,   207,   219,   232,   246,   260,   276,   292,
      309,   328,   347,   368,   389,   413,   437,   463,
      490,   519,   550,   583,   617,   654,   693,   734,
      777,   823,   872,   924,   978,  1036,  1066,  1098,
     1130,  1163,  1197,  1232,  1268,  1305,  1343,  1382,
     1422,  1464,  1506,  1550,  1596,  1642,  1690,  1740,
     1790,  1843,  1896,  1952,  2009,  2068,  2128,  2190,
     2254,  2320,  2388,  2457,  2529,  2603,  2679,  2757,
     2838,  2920,  3006,  3093,  3184,  3277,  3372,  3471,
     3572,  3677,  3784,  3894,  4008,  4125,  4246,  4370,
     4497,  4629,  4764,  4903,  5046,  5193,  5345,  5501,
     5662,  5827,  5912,  5997,  6084,  6172,  6262,  6353,
     6445,  6538,  6633,  6729,  6827,  6925,  7026,  7128,
     7231,  7336,  7442,  7550,  7659,  7771,  7883,  7997,
     8113,  8231,  8350,  8471,  8594,  8719,  8845,  8973,
     9103,  9235,  9369,  9505,  9643,  9783,  9924, 10068,
    10214, 10362, 10512, 10665, 10819, 10976, 11135, 11297,
    11460, 11627, 11795, 11966, 12139, 12315, 12494, 12675,
    12859, 13045, 13234, 13426, 13621, 13818, 14018, 14222,
    14428, 14637, 14849, 15064, 15283, 15504, 15729, 15957,
    16188, 16423, 16661, 16902, 17147, 17396, 17648, 17904,
    18164, 18427, 18694, 18965, 19240, 19519, 19802, 20089,
    20380, 20675, 20975, 21279, 21587, 21900, 22218, 22540,
    22867, 23170, 23534, 23875, 24221, 24573, 24929, 25290,
    25657, 26029, 26406, 26789, 27177, 27571, 27970, 28376,
    28787, 29205, 29628, 30057, 30493, 30935, 31383, 31838,
    32300, 32768,
};

const uint32_t ff_dca_inv_dmixtable[FF_DCA_INV_DMIXTABLE_SIZE] = {
    6553600, 6186997, 5840902, 5514167, 5205710, 4914507, 4639593, 4380059,
    4135042, 3903731, 3685360, 3479204, 3284581, 3100844, 2927386, 2763630,
    2609035, 2463088, 2325305, 2195230, 2072430, 2013631, 1956500, 1900990,
    1847055, 1794651, 1743733, 1694260, 1646190, 1599484, 1554103, 1510010,
    1467168, 1425542, 1385096, 1345798, 1307615, 1270515, 1234468, 1199444,
    1165413, 1132348, 1100221, 1069005, 1038676, 1009206,  980573,  952752,
     925721,  899456,  873937,  849141,  825049,  801641,  778897,  756798,
     735326,  714463,  694193,  674497,  655360,  636766,  618700,  601146,
     584090,  567518,  551417,  535772,  520571,  505801,  491451,  477507,
     463959,  450796,  438006,  425579,  413504,  401772,  390373,  379297,
     368536,  363270,  358080,  352964,  347920,  342949,  338049,  333219,
     328458,  323765,  319139,  314579,  310084,  305654,  301287,  296982,
     292739,  288556,  284433,  280369,  276363,  272414,  268522,  264685,
     260904,  257176,  253501,  249879,  246309,  242790,  239321,  235901,
     232531,  229208,  225933,  222705,  219523,  216386,  213295,  210247,
     207243,  204282,  201363,  198486,  195650,  192855,  190099,  187383,
     184706,  182066,  179465,  176901,  174373,  171882,  169426,  167005,
     164619,  162267,  159948,  157663,  155410,  153190,  151001,  148844,
     146717,  144621,  142554,  140517,  138510,  136531,  134580,  132657,
     130762,  128893,  127052,  125236,  123447,  121683,  119944,  118231,
     116541,  114876,  113235,  111617,  110022,  108450,  106901,  105373,
     103868,  102383,  100921,   99479,   98057,   96656,   95275,   93914,
      92682,   91249,   89946,   88660,   87394,   86145,   84914,   83701,
      82505,   81326,   80164,   79019,   77890,   76777,   75680,   74598,
      73533,   72482,   71446,   70425,   69419,   68427,   67450,   66486,
      65536,
};

const uint16_t ff_dca_xll_refl_coeff[128] = {
        0,  3070,  5110,  7140,  9156, 11154, 13132, 15085,
    17010, 18904, 20764, 22588, 24373, 26117, 27818, 29474,
    31085, 32648, 34164, 35631, 37049, 38418, 39738, 41008,
    42230, 43404, 44530, 45609, 46642, 47630, 48575, 49477,
    50337, 51157, 51937, 52681, 53387, 54059, 54697, 55302,
    55876, 56421, 56937, 57426, 57888, 58326, 58741, 59132,
    59502, 59852, 60182, 60494, 60789, 61066, 61328, 61576,
    61809, 62029, 62236, 62431, 62615, 62788, 62951, 63105,
    63250, 63386, 63514, 63635, 63749, 63855, 63956, 64051,
    64140, 64224, 64302, 64376, 64446, 64512, 64573, 64631,
    64686, 64737, 64785, 64830, 64873, 64913, 64950, 64986,
    65019, 65050, 65079, 65107, 65133, 65157, 65180, 65202,
    65222, 65241, 65259, 65275, 65291, 65306, 65320, 65333,
    65345, 65357, 65368, 65378, 65387, 65396, 65405, 65413,
    65420, 65427, 65434, 65440, 65446, 65451, 65456, 65461,
    65466, 65470, 65474, 65478, 65481, 65485, 65488, 65491
};

const int32_t ff_dca_xll_band_coeff[20] = {
      868669, -5931642, -1228483,  4194304,
      -20577,   122631,  -393647,   904476,
    -1696305,  2825313, -4430736,  6791313,
       41153,  -245210,   785564, -1788164,
     3259333, -5074941,  6928550, -8204883
};

const uint16_t ff_dca_avg_g3_freqs[3] = { 16000, 18000, 24000 };

const uint16_t ff_dca_fst_amp[44] = {
       0,    1,    2,    3,
       4,    6,    8,   10,
      12,   16,   20,   24,
      28,   36,   44,   52,
      60,   76,   92,  108,
     124,  156,  188,  220,
     252,  316,  380,  444,
     508,  636,  764,  892,
    1020, 1276, 1532, 1788,
    2044, 2556, 3068, 3580,
    4092, 5116, 6140, 7164
};

const uint8_t ff_dca_freq_to_sb[32] = {
    0, 1, 2, 2, 3, 3, 3, 3, 4, 4, 4, 4, 4, 4, 4, 4,
    5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5
};

const int8_t ff_dca_ph0_shift[8] = {
    -32, +96, -96, +32, +96, -32, +32, -96
};

const uint8_t ff_dca_grid_1_to_scf[11] = {
    0, 1, 2, 3, 4, 6, 7, 10, 14, 19, 26
};

const uint8_t ff_dca_grid_2_to_scf[3] = {
    4, 10, 18
};

const uint8_t ff_dca_scf_to_grid_1[32] = {
    0, 1, 2, 3, 4, 4, 5, 5, 5, 6, 6, 6, 6, 7, 7, 7,
    7, 7, 8, 8, 8, 8, 8, 8, 9, 9, 9, 9, 9, 9, 9, 10
};

const uint8_t ff_dca_scf_to_grid_2[32] = {
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 1, 1,
    1, 1, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2
};

const uint8_t ff_dca_grid_1_weights[12][32] = {
    {
        128,   0,   0,   0,   0,   0,   0,   0,
          0,   0,   0,   0,   0,   0,   0,   0,
          0,   0,   0,   0,   0,   0,   0,   0,
          0,   0,   0,   0,   0,   0,   0,   0,
    }, {
          0, 128,   0,   0,   0,   0,   0,   0,
          0,   0,   0,   0,   0,   0,   0,   0,
          0,   0,   0,   0,   0,   0,   0,   0,
          0,   0,   0,   0,   0,   0,   0,   0,
    }, {
          0,   0, 128,   0,   0,   0,   0,   0,
          0,   0,   0,   0,   0,   0,   0,   0,
          0,   0,   0,   0,   0,   0,   0,   0,
          0,   0,   0,   0,   0,   0,   0,   0,
    }, {
          0,   0,   0, 128,   0,   0,   0,   0,
          0,   0,   0,   0,   0,   0,   0,   0,
          0,   0,   0,   0,   0,   0,   0,   0,
          0,   0,   0,   0,   0,   0,   0,   0,
    }, {
          0,   0,   0,   0, 128, 128,   0,   0,
          0,   0,   0,   0,   0,   0,   0,   0,
          0,   0,   0,   0,   0,   0,   0,   0,
          0,   0,   0,   0,   0,   0,   0,   0,
    }, {
          0,   0,   0,   0,   0,   0, 128,  85,
         43,   0,   0,   0,   0,   0,   0,   0,
          0,   0,   0,   0,   0,   0,   0,   0,
          0,   0,   0,   0,   0,   0,   0,   0,
    }, {
          0,   0,   0,   0,   0,   0,   0,  43,
         85, 128,  96,  64,  32,   0,   0,   0,
          0,   0,   0,   0,   0,   0,   0,   0,
          0,   0,   0,   0,   0,   0,   0,   0,
    }, {
          0,   0,   0,   0,   0,   0,   0,   0,
          0,   0,  32,  64,  96, 128, 102,  77,
         51,  26,   0,   0,   0,   0,   0,   0,
          0,   0,   0,   0,   0,   0,   0,   0,
    }, {
          0,   0,   0,   0,   0,   0,   0,   0,
          0,   0,   0,   0,   0,   0,  26,  51,
         77, 102, 128, 107,  85,  64,  43,  21,
          0,   0,   0,   0,   0,   0,   0,   0,
    }, {
          0,   0,   0,   0,   0,   0,   0,   0,
          0,   0,   0,   0,   0,   0,   0,   0,
          0,   0,   0,  21,  43,  64,  85, 107,
        128, 110,  91,  73,  55,  37,  18,   0,
    }, {
          0,   0,   0,   0,   0,   0,   0,   0,
          0,   0,   0,   0,   0,   0,   0,   0,
          0,   0,   0,   0,   0,   0,   0,   0,
          0,  18,  37,  55,  73,  91, 110, 128,
    }, {
        0 /* empty */
    }
};

const uint8_t ff_dca_sb_reorder[8][8] = {
    { 0, 1, 2, 3, 4, 5, 6, 7 },
    { 1, 0, 2, 3, 4, 5, 6, 7 },
    { 3, 1, 0, 2, 4, 5, 6, 7 },
    { 1, 2, 3, 0, 4, 5, 6, 7 },
    { 1, 2, 5, 3, 0, 4, 6, 7 },
    { 1, 2, 2, 5, 3, 0, 4, 6 },
    { 1, 2, 2, 6, 5, 3, 0, 4 },
    { 1, 2, 2, 6, 5, 4, 0, 3 }
};

const int8_t ff_dca_lfe_delta_index_16[8] = {
    -4, -3, -2, -1, 2, 4, 6, 8
};

const int8_t ff_dca_lfe_delta_index_24[32] = {
    -8, -8, -7, -7, -6, -6, -5, -5, -4, -4, -3, -3, -2, -2, -1, -1,
     1,  1,  2,  2,  3,  3,  4,  4,  5,  5,  6,  6,  7,  7,  8,  8
};

const uint16_t ff_dca_rsd_pack_5_in_8[256] = {
    0x0000, 0x0100, 0x0200, 0x0040, 0x0140, 0x0240, 0x0080, 0x0180,
    0x0280, 0x0010, 0x0110, 0x0210, 0x0050, 0x0150, 0x0250, 0x0090,
    0x0190, 0x0290, 0x0020, 0x0120, 0x0220, 0x0060, 0x0160, 0x0260,
    0x00a0, 0x01a0, 0x02a0, 0x0004, 0x0104, 0x0204, 0x0044, 0x0144,
    0x0244, 0x0084, 0x0184, 0x0284, 0x0014, 0x0114, 0x0214, 0x0054,
    0x0154, 0x0254, 0x0094, 0x0194, 0x0294, 0x0024, 0x0124, 0x0224,
    0x0064, 0x0164, 0x0264, 0x00a4, 0x01a4, 0x02a4, 0x0008, 0x0108,
    0x0208, 0x0048, 0x0148, 0x0248, 0x0088, 0x0188, 0x0288, 0x0018,
    0x0118, 0x0218, 0x0058, 0x0158, 0x0258, 0x0098, 0x0198, 0x0298,
    0x0028, 0x0128, 0x0228, 0x0068, 0x0168, 0x0268, 0x00a8, 0x01a8,
    0x02a8, 0x0001, 0x0101, 0x0201, 0x0041, 0x0141, 0x0241, 0x0081,
    0x0181, 0x0281, 0x0011, 0x0111, 0x0211, 0x0051, 0x0151, 0x0251,
    0x0091, 0x0191, 0x0291, 0x0021, 0x0121, 0x0221, 0x0061, 0x0161,
    0x0261, 0x00a1, 0x01a1, 0x02a1, 0x0005, 0x0105, 0x0205, 0x0045,
    0x0145, 0x0245, 0x0085, 0x0185, 0x0285, 0x0015, 0x0115, 0x0215,
    0x0055, 0x0155, 0x0255, 0x0095, 0x0195, 0x0295, 0x0025, 0x0125,
    0x0225, 0x0065, 0x0165, 0x0265, 0x00a5, 0x01a5, 0x02a5, 0x0009,
    0x0109, 0x0209, 0x0049, 0x0149, 0x0249, 0x0089, 0x0189, 0x0289,
    0x0019, 0x0119, 0x0219, 0x0059, 0x0159, 0x0259, 0x0099, 0x0199,
    0x0299, 0x0029, 0x0129, 0x0229, 0x0069, 0x0169, 0x0269, 0x00a9,
    0x01a9, 0x02a9, 0x0002, 0x0102, 0x0202, 0x0042, 0x0142, 0x0242,
    0x0082, 0x0182, 0x0282, 0x0012, 0x0112, 0x0212, 0x0052, 0x0152,
    0x0252, 0x0092, 0x0192, 0x0292, 0x0022, 0x0122, 0x0222, 0x0062,
    0x0162, 0x0262, 0x00a2, 0x01a2, 0x02a2, 0x0006, 0x0106, 0x0206,
    0x0046, 0x0146, 0x0246, 0x0086, 0x0186, 0x0286, 0x0016, 0x0116,
    0x0216, 0x0056, 0x0156, 0x0256, 0x0096, 0x0196, 0x0296, 0x0026,
    0x0126, 0x0226, 0x0066, 0x0166, 0x0266, 0x00a6, 0x01a6, 0x02a6,
    0x000a, 0x010a, 0x020a, 0x004a, 0x014a, 0x024a, 0x008a, 0x018a,
    0x028a, 0x001a, 0x011a, 0x021a, 0x005a, 0x015a, 0x025a, 0x009a,
    0x019a, 0x029a, 0x002a, 0x012a, 0x022a, 0x006a, 0x016a, 0x026a,
    0x00aa, 0x01aa, 0x02aa, 0x0155, 0x0155, 0x0155, 0x0155, 0x0155,
    0x0155, 0x0155, 0x0155, 0x0155, 0x0155, 0x0155, 0x0155, 0x0155
};

const uint8_t ff_dca_rsd_pack_3_in_7[128][3] = {
    { 0, 0, 0 }, { 0, 0, 1 }, { 0, 0, 2 }, { 0, 0, 3 },
    { 0, 0, 4 }, { 0, 1, 0 }, { 0, 1, 1 }, { 0, 1, 2 },
    { 0, 1, 3 }, { 0, 1, 4 }, { 0, 2, 0 }, { 0, 2, 1 },
    { 0, 2, 2 }, { 0, 2, 3 }, { 0, 2, 4 }, { 0, 3, 0 },
    { 0, 3, 1 }, { 0, 3, 2 }, { 0, 3, 3 }, { 0, 3, 4 },
    { 0, 4, 0 }, { 0, 4, 1 }, { 0, 4, 2 }, { 0, 4, 3 },
    { 0, 4, 4 }, { 1, 0, 0 }, { 1, 0, 1 }, { 1, 0, 2 },
    { 1, 0, 3 }, { 1, 0, 4 }, { 1, 1, 0 }, { 1, 1, 1 },
    { 1, 1, 2 }, { 1, 1, 3 }, { 1, 1, 4 }, { 1, 2, 0 },
    { 1, 2, 1 }, { 1, 2, 2 }, { 1, 2, 3 }, { 1, 2, 4 },
    { 1, 3, 0 }, { 1, 3, 1 }, { 1, 3, 2 }, { 1, 3, 3 },
    { 1, 3, 4 }, { 1, 4, 0 }, { 1, 4, 1 }, { 1, 4, 2 },
    { 1, 4, 3 }, { 1, 4, 4 }, { 2, 0, 0 }, { 2, 0, 1 },
    { 2, 0, 2 }, { 2, 0, 3 }, { 2, 0, 4 }, { 2, 1, 0 },
    { 2, 1, 1 }, { 2, 1, 2 }, { 2, 1, 3 }, { 2, 1, 4 },
    { 2, 2, 0 }, { 2, 2, 1 }, { 2, 2, 2 }, { 2, 2, 3 },
    { 2, 2, 4 }, { 2, 3, 0 }, { 2, 3, 1 }, { 2, 3, 2 },
    { 2, 3, 3 }, { 2, 3, 4 }, { 2, 4, 0 }, { 2, 4, 1 },
    { 2, 4, 2 }, { 2, 4, 3 }, { 2, 4, 4 }, { 3, 0, 0 },
    { 3, 0, 1 }, { 3, 0, 2 }, { 3, 0, 3 }, { 3, 0, 4 },
    { 3, 1, 0 }, { 3, 1, 1 }, { 3, 1, 2 }, { 3, 1, 3 },
    { 3, 1, 4 }, { 3, 2, 0 }, { 3, 2, 1 }, { 3, 2, 2 },
    { 3, 2, 3 }, { 3, 2, 4 }, { 3, 3, 0 }, { 3, 3, 1 },
    { 3, 3, 2 }, { 3, 3, 3 }, { 3, 3, 4 }, { 3, 4, 0 },
    { 3, 4, 1 }, { 3, 4, 2 }, { 3, 4, 3 }, { 3, 4, 4 },
    { 4, 0, 0 }, { 4, 0, 1 }, { 4, 0, 2 }, { 4, 0, 3 },
    { 4, 0, 4 }, { 4, 1, 0 }, { 4, 1, 1 }, { 4, 1, 2 },
    { 4, 1, 3 }, { 4, 1, 4 }, { 4, 2, 0 }, { 4, 2, 1 },
    { 4, 2, 2 }, { 4, 2, 3 }, { 4, 2, 4 }, { 4, 3, 0 },
    { 4, 3, 1 }, { 4, 3, 2 }, { 4, 3, 3 }, { 4, 3, 4 },
    { 4, 4, 0 }, { 4, 4, 1 }, { 4, 4, 2 }, { 4, 4, 3 },
    { 4, 4, 4 }, { 2, 2, 2 }, { 2, 2, 2 }, { 2, 2, 2 }
};

const float ff_dca_rsd_level_2a[2] = {
    -0.47, 0.47
};

const float ff_dca_rsd_level_2b[2] = {
    -0.645, 0.645
};

const float ff_dca_rsd_level_3[3] = {
    -0.645, 0.0, 0.645
};

const float ff_dca_rsd_level_5[5] = {
    -0.875, -0.375, 0.0, 0.375, 0.875
};

const float ff_dca_rsd_level_8[8] = {
    -1.0, -0.625, -0.291666667, 0.0, 0.25, 0.5, 0.75, 1.0
};

const float ff_dca_rsd_level_16[16] = {
    -1.3125, -1.1375, -0.9625, -0.7875,
    -0.6125, -0.4375, -0.2625, -0.0875,
     0.0875,  0.2625,  0.4375,  0.6125,
     0.7875,  0.9625,  1.1375,  1.3125
};

const float ff_dca_synth_env[32] = {
    0.00240763666390, 0.00960735979838, 0.02152983213390, 0.03806023374436,
    0.05903936782582, 0.08426519384873, 0.11349477331863, 0.14644660940673,
    0.18280335791818, 0.22221488349020, 0.26430163158700, 0.30865828381746,
    0.35485766137277, 0.40245483899194, 0.45099142983522, 0.5,
    0.54900857016478, 0.59754516100806, 0.64514233862723, 0.69134171618254,
    0.73569836841300, 0.77778511650980, 0.81719664208182, 0.85355339059327,
    0.88650522668137, 0.91573480615127, 0.94096063217418, 0.96193976625564,
    0.97847016786610, 0.99039264020162, 0.99759236333610, 1.0
};

const float ff_dca_corr_cf[32][11] = {
    {-0.01179, 0.04281, 0.46712, 0.46345,-3.94525, 3.94525,
     -0.46345,-0.46712,-0.04281, 0.01179,-0.00299 },
    {-0.00929, 0.04882, 0.45252, 0.37972,-3.85446, 4.03189,
     -0.55069,-0.48040,-0.03599, 0.01445,-0.00229 },
    {-0.00696, 0.05403, 0.43674, 0.29961,-3.75975, 4.11413,
     -0.64135,-0.49221,-0.02834, 0.01726,-0.00156 },
    {-0.00481, 0.05847, 0.41993, 0.22319,-3.66138, 4.19175,
     -0.73529,-0.50241,-0.01983, 0.02021,-0.00080 },
    {-0.00284, 0.06216, 0.40224, 0.15053,-3.55963, 4.26452,
     -0.83239,-0.51085,-0.01047, 0.02328,-0.00003 },
    {-0.00105, 0.06515, 0.38378, 0.08168,-3.45475, 4.33225,
     -0.93249,-0.51738,-0.00024, 0.02646, 0.00074 },
    { 0.00054, 0.06745, 0.36471, 0.01668,-3.34703, 4.39475,
     -1.03543,-0.52184, 0.01085, 0.02973, 0.00152 },
    { 0.00195, 0.06912, 0.34515,-0.04445,-3.23676, 4.45185,
     -1.14105,-0.52410, 0.02280, 0.03306, 0.00228 },
    { 0.00318, 0.07017, 0.32521,-0.10168,-3.12422, 4.50339,
     -1.24914,-0.52400, 0.03561, 0.03643, 0.00302 },
    { 0.00422, 0.07065, 0.30503,-0.15503,-3.00969, 4.54921,
     -1.35952,-0.52141, 0.04925, 0.03981, 0.00373 },
    { 0.00508, 0.07061, 0.28471,-0.20450,-2.89348, 4.58919,
     -1.47197,-0.51618, 0.06370, 0.04319, 0.00440 },
    { 0.00577, 0.07007, 0.26436,-0.25013,-2.77587, 4.62321,
     -1.58627,-0.50818, 0.07895, 0.04652, 0.00501 },
    { 0.00629, 0.06909, 0.24410,-0.29194,-2.65716, 4.65118,
     -1.70219,-0.49727, 0.09494, 0.04979, 0.00556 },
    { 0.00666, 0.06769, 0.22400,-0.33000,-2.53764, 4.67302,
     -1.81949,-0.48335, 0.11166, 0.05295, 0.00604 },
    { 0.00687, 0.06592, 0.20416,-0.36435,-2.41760, 4.68866,
     -1.93791,-0.46627, 0.12904, 0.05597, 0.00642 },
    { 0.00694, 0.06383, 0.18468,-0.39506,-2.29732, 4.69806,
     -2.05720,-0.44593, 0.14705, 0.05881, 0.00671 },
    { 0.00689, 0.06144, 0.16561,-0.42223,-2.17710, 4.70120,
     -2.17710,-0.42223, 0.16561, 0.06144, 0.00689 },
    { 0.00671, 0.05881, 0.14705,-0.44593,-2.05720, 4.69806,
     -2.29732,-0.39506, 0.18468, 0.06383, 0.00694 },
    { 0.00642, 0.05597, 0.12904,-0.46627,-1.93791, 4.68865,
     -2.41759,-0.36435, 0.20416, 0.06592, 0.00687 },
    { 0.00604, 0.05295, 0.11166,-0.48334,-1.81949, 4.67301,
     -2.53763,-0.33000, 0.22400, 0.06769, 0.00666 },
    { 0.00556, 0.04979, 0.09494,-0.49727,-1.70219, 4.65117,
     -2.65715,-0.29194, 0.24409, 0.06909, 0.00629 },
    { 0.00501, 0.04652, 0.07894,-0.50818,-1.58627, 4.62321,
     -2.77587,-0.25013, 0.26436, 0.07007, 0.00577 },
    { 0.00440, 0.04319, 0.06370,-0.51618,-1.47197, 4.58919,
     -2.89348,-0.20450, 0.28471, 0.07061, 0.00508 },
    { 0.00373, 0.03981, 0.04925,-0.52141,-1.35952, 4.54921,
     -3.00970,-0.15503, 0.30503, 0.07065, 0.00422 },
    { 0.00302, 0.03643, 0.03561,-0.52400,-1.24915, 4.50339,
     -3.12422,-0.10168, 0.32521, 0.07017, 0.00318 },
    { 0.00228, 0.03306, 0.02280,-0.52410,-1.14105, 4.45186,
     -3.23677,-0.04445, 0.34515, 0.06912, 0.00195 },
    { 0.00152, 0.02973, 0.01085,-0.52184,-1.03544, 4.39477,
     -3.34704, 0.01668, 0.36471, 0.06745, 0.00054 },
    { 0.00074, 0.02646,-0.00024,-0.51738,-0.93249, 4.33226,
     -3.45476, 0.08168, 0.38378, 0.06515,-0.00105 },
    {-0.00003, 0.02328,-0.01047,-0.51085,-0.83239, 4.26452,
     -3.55963, 0.15053, 0.40224, 0.06216,-0.00284 },
    {-0.00080, 0.02021,-0.01983,-0.50241,-0.73529, 4.19174,
     -3.66138, 0.22319, 0.41993, 0.05847,-0.00481 },
    {-0.00156, 0.01726,-0.02834,-0.49221,-0.64135, 4.11413,
     -3.75974, 0.29961, 0.43674, 0.05403,-0.00696 },
    {-0.00229, 0.01445,-0.03599,-0.48040,-0.55069, 4.03188,
     -3.85445, 0.37972, 0.45251, 0.04882,-0.00929 },
};

const float ff_dca_quant_amp[57] = {
    4.88281250E-04, 1.46484375E-03, 2.32267031E-03, 3.28475167E-03,
    4.64534014E-03, 6.56950334E-03, 9.29068029E-03, 1.31390067E-02,
    1.85813606E-02, 2.62780134E-02, 3.71627212E-02, 5.25560267E-02,
    7.43254423E-02, 1.05112053E-01, 1.48650885E-01, 2.10224107E-01,
    2.97301769E-01, 4.20448214E-01, 5.94603539E-01, 8.40896428E-01,
    1.18920708E+00, 1.68179286E+00, 2.37841415E+00, 3.36358571E+00,
    4.75682831E+00, 6.72717142E+00, 9.51365662E+00, 1.34543428E+01,
    1.90273132E+01, 2.69086857E+01, 3.80546265E+01, 5.38173714E+01,
    7.61092529E+01, 1.07634743E+02, 1.52218506E+02, 2.15269485E+02,
    3.04437012E+02, 4.30538971E+02, 6.08874023E+02, 8.61077942E+02,
    1.21774805E+03, 1.72215588E+03, 2.43549609E+03, 3.44431177E+03,
    4.87099219E+03, 6.88862354E+03, 9.74198438E+03, 1.37772471E+04,
    1.94839688E+04, 2.75544941E+04, 3.89679375E+04, 5.51089883E+04,
    7.79358750E+04, 1.10217977E+05, 1.55871750E+05, 2.20435953E+05,
    0.00000000E+00,
};

const float ff_dca_st_coeff[34] = {
    2.69086857E+01, 2.69086857E+01, 1.34543419E+01, 6.72717142E+00,
    3.36358571E+00, 1.68179286E+00, 8.40896428E-01, 5.94603479E-01,
    4.20448214E-01, 2.97301799E-01, 2.10224107E-01, 1.48650900E-01,
    1.05112098E-01, 7.43253976E-02, 5.25560006E-02, 3.71626988E-02,
    3.12500000E-02, 2.62780003E-02, 1.85813997E-02, 1.31390002E-02,
    9.29069985E-03, 6.56950008E-03, 4.64530010E-03, 3.28480010E-03,
    2.32270011E-03, 1.64240005E-03, 1.16130000E-03, 5.80699998E-04,
    2.90299999E-04, 1.45200000E-04, 7.25999998E-05, 3.62999999E-05,
    1.82000003E-05, 0.00000000E+00,
};

const float ff_dca_long_window[128] = {
    0.00000000E+00, 7.42882412E-06, 5.28020973E-05, 1.71007006E-04,
    3.96653224E-04, 7.63946096E-04, 1.30655791E-03, 2.05750111E-03,
    3.04900459E-03, 4.31239139E-03, 5.87796280E-03, 7.77488295E-03,
    1.00310687E-02, 1.26730874E-02, 1.57260559E-02, 1.92135461E-02,
    2.31574941E-02, 2.75781266E-02, 3.24938744E-02, 3.79213169E-02,
    4.38751020E-02, 5.03679104E-02, 5.74104004E-02, 6.50111660E-02,
    7.31767192E-02, 8.19114447E-02, 9.12176073E-02, 1.01095326E-01,
    1.11542597E-01, 1.22555278E-01, 1.34127125E-01, 1.46249816E-01,
    1.58912972E-01, 1.72104210E-01, 1.85809180E-01, 2.00011641E-01,
    2.14693516E-01, 2.29834959E-01, 2.45414421E-01, 2.61408776E-01,
    2.77793378E-01, 2.94542134E-01, 3.11627686E-01, 3.29021394E-01,
    3.46693635E-01, 3.64613682E-01, 3.82750064E-01, 4.01070446E-01,
    4.19541985E-01, 4.38131332E-01, 4.56804723E-01, 4.75528270E-01,
    4.94267941E-01, 5.12989700E-01, 5.31659782E-01, 5.50244689E-01,
    5.68711281E-01, 5.87027133E-01, 6.05160415E-01, 6.23080134E-01,
    6.40756190E-01, 6.58159554E-01, 6.75262392E-01, 6.92038059E-01,
    7.08461344E-01, 7.24508464E-01, 7.40157187E-01, 7.55386829E-01,
    7.70178556E-01, 7.84515142E-01, 7.98381269E-01, 8.11763465E-01,
    8.24650168E-01, 8.37031603E-01, 8.48900259E-01, 8.60250235E-01,
    8.71077836E-01, 8.81381273E-01, 8.91160548E-01, 9.00417745E-01,
    9.09156621E-01, 9.17382956E-01, 9.25104082E-01, 9.32328999E-01,
    9.39068437E-01, 9.45334494E-01, 9.51140642E-01, 9.56501782E-01,
    9.61433768E-01, 9.65953648E-01, 9.70079303E-01, 9.73829389E-01,
    9.77223217E-01, 9.80280578E-01, 9.83021557E-01, 9.85466540E-01,
    9.87635851E-01, 9.89549816E-01, 9.91228402E-01, 9.92691338E-01,
    9.93957877E-01, 9.95046616E-01, 9.95975435E-01, 9.96761382E-01,
    9.97420728E-01, 9.97968733E-01, 9.98419642E-01, 9.98786569E-01,
    9.99081731E-01, 9.99315977E-01, 9.99499321E-01, 9.99640644E-01,
    9.99747574E-01, 9.99826968E-01, 9.99884665E-01, 9.99925494E-01,
    9.99953628E-01, 9.99972343E-01, 9.99984324E-01, 9.99991655E-01,
    9.99995887E-01, 9.99998152E-01, 9.99999285E-01, 9.99999762E-01,
    9.99999940E-01, 1.00000000E+00, 1.00000000E+00, 1.00000000E+00,
};

const float ff_dca_lfe_step_size_16[101] = {
    2.1362956633198035E-004, 2.4414807580797754E-004, 2.7466658528397473E-004,
    2.7466658528397473E-004, 3.0518509475997192E-004, 3.3570360423596911E-004,
    3.9674062318796350E-004, 4.2725913266396069E-004, 4.5777764213995788E-004,
    5.1881466109195227E-004, 5.7985168004394665E-004, 6.1037018951994385E-004,
    6.7140720847193823E-004, 7.6296273689992981E-004, 8.2399975585192419E-004,
    9.1555528427991577E-004, 1.0071108127079073E-003, 1.0986663411358989E-003,
    1.2207403790398877E-003, 1.3428144169438765E-003, 1.4648884548478652E-003,
    1.6174810022278512E-003, 1.7700735496078372E-003, 1.9531846064638203E-003,
    2.1362956633198035E-003, 2.3499252296517838E-003, 2.5940733054597613E-003,
    2.8687398907437361E-003, 3.1434064760277108E-003, 3.4485915707876827E-003,
    3.7842951750236518E-003, 4.1810357982116153E-003, 4.6082949308755760E-003,
    5.0660725730155339E-003, 5.5543687246314890E-003, 6.1037018951994385E-003,
    6.7445905941953795E-003, 7.4159978026673177E-003, 8.1484420300912512E-003,
    8.9419232764671782E-003, 9.8574785607470940E-003, 1.0834070863979004E-002,
    1.1932737205114903E-002, 1.3122959074678793E-002, 1.4435254982146673E-002,
    1.5869624927518540E-002, 1.7456587420270394E-002, 1.9196142460402233E-002,
    2.1118808557390057E-002, 2.3224585711233862E-002, 2.5543992431409649E-002,
    2.8107547227393413E-002, 3.0915250099185155E-002, 3.4028138065736867E-002,
    3.7415692617572556E-002, 4.1169469283120215E-002, 4.5258949552903834E-002,
    4.9806207464827418E-002, 5.4780724509414958E-002, 6.0274056215094456E-002,
    6.6286202581865905E-002, 7.2908719138157288E-002, 8.0202642902920618E-002,
    8.8229010895107887E-002, 9.7048860133671075E-002, 1.0675374614703818E-001,
    1.1743522446363720E-001, 1.2918485061189611E-001, 1.4209418012024294E-001,
    1.5628528702658162E-001, 1.7191076387829218E-001, 1.8912320322275461E-001,
    2.0804467909787286E-001, 2.2882778405102694E-001, 2.5171666615802485E-001,
    2.7689443647572254E-001, 3.0457472457045198E-001, 3.3503219702749720E-001,
    3.6854152043214211E-001, 4.0537736136967073E-001, 4.4593646046327096E-001,
    4.9052400280770286E-001, 5.3956724753563035E-001, 5.9352397228919340E-001,
    6.5288247322000792E-001, 7.1816156498916595E-001, 7.9000213629566329E-001,
    8.6898403881954400E-001, 9.5590075380718409E-001, 1.0514847254860074E+000,
    1.1566209906308176E+000, 1.2722861415448470E+000, 1.3995178075502792E+000,
    1.5394756920072024E+000, 1.6934110538041323E+000, 1.8627582628864405E+000,
    2.0490432447279274E+000, 2.2539445173497725E+000, 2.4793237098300120E+000,
    2.7272865993224893E+000, 3.0000000000000000E+000
};

const float ff_dca_lfe_step_size_24[144] = {
    3.5762791128491298E-006, 3.9339070241340428E-006, 4.4107442391805934E-006,
    4.7683721504655064E-006, 5.2452093655120570E-006, 5.8412558843202453E-006,
    6.4373024031284336E-006, 7.0333489219366219E-006, 7.7486047445064479E-006,
    8.4638605670762738E-006, 9.4175349971693751E-006, 1.0252000123500839E-005,
    1.1324883857355578E-005, 1.2516976894971954E-005, 1.3709069932588331E-005,
    1.5139581577727983E-005, 1.6570093222867636E-005, 1.8239023475530564E-005,
    2.0146372335716766E-005, 2.2053721195902969E-005, 2.4318697967374082E-005,
    2.6702884042606836E-005, 2.9444698029124504E-005, 3.2305721319403807E-005,
    3.5643581824729662E-005, 3.9100651633817152E-005, 4.3034558657951193E-005,
    4.7326093593370149E-005, 5.2094465743835655E-005, 5.7339675109347712E-005,
    6.3061721689906320E-005, 6.9379814789273121E-005, 7.6293954407448102E-005,
    8.3923349848192912E-005, 9.2268001111507552E-005, 1.0156632680491529E-004,
    1.1169911762465449E-004, 1.2290479217824841E-004, 1.3518335046569711E-004,
    1.4865400179076216E-004, 1.6355516476096688E-004, 1.7988683937631122E-004,
    1.9788744424431852E-004, 2.1767618866875036E-004, 2.3949149125713007E-004,
    2.6345256131321922E-004, 2.8979781744454115E-004, 3.1876567825861912E-004,
    3.5059456236297636E-004, 3.8564209766889782E-004, 4.2426591208766842E-004,
    4.6670442422681142E-004, 5.1331526199761173E-004, 5.6469447191887759E-004,
    6.2108047259813216E-004, 6.8318851985794547E-004, 7.5149545091336386E-004,
    8.2671652158695713E-004, 9.0932856909377204E-004, 1.0002852678639017E-003,
    1.1003018737199156E-003, 1.2103320610919071E-003, 1.3314487137137310E-003,
    1.4646055060154803E-003, 1.6109945310347714E-003, 1.7721655097205054E-003,
    1.9493105351102991E-003, 2.1442177467605765E-003, 2.3586752842277626E-003,
    2.5945904963720436E-003, 2.8539899413573674E-003, 3.1393770145627278E-003,
    3.4533743206708813E-003, 3.7987236736683454E-003, 4.1785245154529228E-003,
    4.5963531251374630E-003, 5.0560242004423382E-003, 5.5617100669992049E-003,
    6.1178214690472445E-003, 6.7296036159519689E-003, 7.4025401356864135E-003,
    8.1428299120461841E-003, 8.9571486660419298E-003, 9.8527681652031147E-003,
    1.0838033060793050E-002, 1.1921884050593860E-002, 1.3114096297513997E-002,
    1.4425517848195773E-002, 1.5868069633015350E-002, 1.7454864675386508E-002,
    1.9200327301064409E-002, 2.1120431556753107E-002, 2.3232462791498040E-002,
    2.5555613703204836E-002, 2.8111222757246822E-002, 3.0922297349250002E-002,
    3.4014586688826884E-002, 3.7415985753057691E-002, 4.1157608170224208E-002,
    4.5273428591898514E-002, 4.9800759530157987E-002, 5.4780847404104160E-002,
    6.0258872539862694E-002, 6.6284783635709721E-002, 7.2913297762071824E-002,
    8.0204615617348624E-002, 8.8225017574431602E-002, 9.7047578936526643E-002,
    1.0675228914645780E-001, 1.1742748229831246E-001, 1.2917031397465634E-001,
    1.4208735729305236E-001, 1.5629603341770570E-001, 1.7192568444319778E-001,
    1.8911816944100493E-001, 2.0803001022696618E-001, 2.2883310661710579E-001,
    2.5171640535788598E-001, 2.7688804589367461E-001, 3.0457679087839018E-001,
    3.3503452957088109E-001, 3.6853794676517804E-001, 4.0539174144169587E-001,
    4.4593089174400469E-001, 4.9052399283933557E-001, 5.3957635636047796E-001,
    5.9353406352210802E-001, 6.5288742219059737E-001, 7.1817609288407480E-001,
    7.8999373793527339E-001, 8.6899314749159184E-001, 9.5589243839889027E-001,
    1.0514817299225008E+000, 1.1566298194682383E+000, 1.2722928848615747E+000,
    1.3995221137430804E+000, 1.5394743131964581E+000, 1.6934218041207556E+000,
    1.8627639845328312E+000, 2.0490403233814627E+000, 2.2539444272451910E+000,
    2.4793389414952922E+000, 2.7272728356448215E+000, 2.9999998807906962E+000
};

const float ff_dca_bank_coeff[10] = {
    0.022810893, 0.41799772, 0.90844810, 0.99973983,
    0.068974845, 0.34675997, 0.29396889, 0.19642374,
    0.308658270, 0.038060233
};

const float ff_dca_lfe_iir[5][4] = {
    { -0.98618466, 1.9861259, 1.0, -1.9840510 },
    { -0.98883152, 1.9887193, 1.0, -1.9979848 },
    { -0.99252087, 1.9923381, 1.0, -1.9990897 },
    { -0.99591690, 1.9956781, 1.0, -1.9993745 },
    { -0.99872285, 1.9984550, 1.0, -1.9994639 }
};
